// Emacs -*- C++ -*-

//
//	Copyright 1993, Center for Computer Vision and Visualization,
//	University of Florida.  All rights reserved.
//

/*  $RCSfile: ValueSet.c,v $
    $Log: ValueSet.c,v $
 * Revision 1.12  1993/11/17  17:57:17  thoth
 * use value_type_compare any place you need a comparison
 * of values.
 *
 * Revision 1.11  1993/06/10  17:08:50  thoth
 * Fix array constructor.
 *
 * Revision 1.10  93/05/26  16:32:04  thoth
 * Copyright Notices
 * 
 * Revision 1.9  93/05/26  13:49:20  thoth
 * move the special functions into another module
 * 
 * Revision 1.8  93/02/20  15:25:02  thoth
 * The LatticeRelation that was causing a compiler bug is now
 * used from IA instead of IA_ValueSet<T>.
 * 
 * Revision 1.7  93/02/08  19:56:57  thoth
 * added output method for testing.
 * Also added ifdefs since template implementation file are really
 * "headers".  We were having problems with multiple inclusion.
 * 
 * Revision 1.6  92/12/14  16:18:50  jm
 *  Renamed class ValueSet to IA_ValueSet
 * 
 * Revision 1.5  92/10/22  14:04:47  jm
 * fixed operator "<=" on two value sets
 * 
 * Revision 1.4  92/09/29  15:27:24  jm
 * *** empty log message ***
 * 
 * Revision 1.3  92/09/16  16:41:30  jm
 * added comments to remove_dup and count_dup functions.
 * changed NULL to 0.
 * 
 * Revision 1.2  92/09/16  14:47:23  jm
 * Included value_range function as an independent function
 * function contains() now returns 1 or 0
 * added keyword 'const' to function count_dup
 * added keyword 'const' to all member operators
 * added definition for operators '>' and '<'
 * added definition for function compare
 * 
 * Revision 1.1  92/09/16  14:46:28  jm
 * Initial revision
 * 
Revision 1.6  92/09/09  14:36:35  jm
cleaned up use of dynamic memory in operator.

Revision 1.5  92/09/08  14:37:44  jm
 the operators have been redefined to be class members in the case of
 valueset op element, and valueset op valueset. The combination
 element op valueset remains as a friend.
 the function make_set does not take parameters anymore. It operates on
 the internal value_array and size.
 function value_range has been changed from class member to independent. It
 will only work with integer arguments.
 predicate operators now return an int instead of a char, in order to be
 able to print the result.

Revision 1.4  92/09/01  13:59:36  jm
added friend functions for operators between elements and ValueSets.
Added a guard for the "bsearch" function
templatized ValueSet
changed the "-" operator to the "/" operator

Revision 1.3  92/08/28  10:01:36  jm
added RCS keywords "RCSfile" and "Log"

*/

#ifndef ValueSet_c_
#define ValueSet_c_

#include <iostream.h>
#include <stdlib.h>
#include "ValueSet.h"

template <class ValueType>
int IA_ValueSet<ValueType>::contains(ValueType e) const {
    
    if (this->size == 0) return 0;
	
    return ((int) bsearch((ValueType *)(&e), 
			 (ValueType *)this->value_array,
			 this->size, sizeof(ValueType),
			 this->value_type_compare) == 0 ? 0 : 1);
}    
    
template <class ValueType>
int IA_ValueSetComparator<ValueType>::value_type_compare(
    const void *t1, const void *t2) {
    if (*(ValueType *)t1 > *(ValueType *)t2)
	return 1;
    
    if (*(ValueType *)t1 < *(ValueType *)t2)
	return -1;
    return 0;
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(const IA_ValueSet<ValueType>& vs) {
    this->size = vs.size;
    if (this->size == 0)
	this->value_array = 0;
    else {
	this->value_array = new ValueType[this->size];
	for (unsigned int i = 0; i < this->size; i++)
	    this->value_array[i] = vs.value_array[i];
    }
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(ValueType v0) {
    this->value_array = new ValueType[this->size = 1];
    
    this->value_array[0] = v0;
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(ValueType v0, ValueType v1) {
    
    this->value_array = new ValueType[this->size = 2];

    this->value_array[0] = v0;
    this->value_array[1] = v1;
    make_set();
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(ValueType v0, ValueType v1, ValueType v2) {

    this->value_array = new ValueType[this->size = 3];
    
    this->value_array[0] = v0;
    this->value_array[1] = v1;
    this->value_array[2] = v2;
    make_set();
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(ValueType v0, ValueType v1,ValueType v2, 
			      ValueType v3) {
    this->value_array = new ValueType[this->size = 4];
    
    this->value_array[0] = v0;
    this->value_array[1] = v1;
    this->value_array[2] = v2;
    this->value_array[3] = v3;
    make_set();

}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(ValueType v0, ValueType v1, ValueType v2, 
			      ValueType v3, ValueType v4) {
    this->value_array = new ValueType[this->size = 5];
    
    this->value_array[0] = v0;
    this->value_array[1] = v1;
    this->value_array[2] = v2;
    this->value_array[3] = v3;
    this->value_array[4] = v4;
    make_set();
}

template <class ValueType>
IA_ValueSet<ValueType>::IA_ValueSet(const ValueType* ar, unsigned int sz) {
    if (sz == 0) {
	this->size = 0;
	this->value_array = 0;
    }
    else {
	this->value_array = new ValueType[this->size = sz];
	for (unsigned int i=0; i < sz; i++)
	    this->value_array[i] = ar[i];
	make_set();
    }
}

// make_set: sorts the value_array and removes duplicates.

template <class ValueType>
void IA_ValueSet<ValueType>::make_set() {
    if (this->size == 0)
	this->value_array = 0;
    else {
	qsort(this->value_array, this->size, sizeof(ValueType),
	      this->value_type_compare);
	remove_dup();
    }
}

// remove_dup is for the private use of make_set
// this->size must not be 0.

template <class ValueType>
void IA_ValueSet<ValueType>::remove_dup() {
    
    ValueType *clean_array = new ValueType[this->size - count_dup()];
    
    unsigned int di = 0;
    
    for (unsigned int i=0; i < this->size - 1; i++)
	if (this->value_array[i] != this->value_array[i+1])
	    clean_array[di++] = this->value_array[i];
    clean_array[di++] = this->value_array[this->size - 1];
    delete [] this->value_array;
    this->value_array = clean_array;
    this->size = di;
}

// count_dup is for the private use of make_set

template <class ValueType>
int IA_ValueSet<ValueType>::count_dup() const {
    int count = 0;
    for (unsigned int i=0; i < this->size - 1; i++)
	if (this->value_array[i] == this->value_array[i+1])
	    count++;
    return count;
}

template <class ValueType>
IA_ValueSet<ValueType>& 
IA_ValueSet<ValueType>::operator=  (const IA_ValueSet<ValueType>& vs) {
    if (this == &vs) 
	return *this;

    // handle empty set possibility
    
    if (vs.size == 0) {
	delete[] this->value_array;  // OK even if value_array already 0
	this->value_array = 0; 
	this->size = 0;
	return *this;
    }
    
    // handle the typical case
    
    if (vs.size != this->size) {
	delete[] this->value_array;
	this->value_array = new ValueType[vs.size];
	this->size = vs.size;
    }
    
    for (unsigned int i = 0; i < this->size; i++)
	this->value_array[i] = vs.value_array[i];
    
    return *this;
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator| (const IA_ValueSet<ValueType> &right) const {
    unsigned int li = 0, ri = 0, ci = 0;

    if (this == &right) 
	return IA_ValueSet<ValueType>(*this);

    ValueType* temp = new ValueType[this->size + right.size];
    while ((li < this->size) && (ri < right.size))
	if (value_type_compare(&this->value_array[li],&right.value_array[ri])<0)
	    temp[ci++] = this->value_array[li++];
	else {
	    temp[ci++] = right.value_array[ri];
	    if (this->value_array[li] == right.value_array[ri++])
		li++;
	}
    if (li == this->size)
	for (;ri < right.size;ri++)
	    temp[ci++] = right.value_array[ri];
    else
	for (;li < this->size;li++)
	    temp[ci++] = this->value_array[li];
    return IA_ValueSet<ValueType>(temp, ci);
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator| (const ValueType &right) const {
    return *this | IA_ValueSet<ValueType>(right);
}

template <class ValueType>
IA_ValueSet<ValueType> 
operator| (const ValueType &left, const IA_ValueSet<ValueType> &right) {
    return IA_ValueSet<ValueType>(left) | right;
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator& (const IA_ValueSet<ValueType> &right)  const {
    unsigned int li = 0, ri = 0, ci = 0;

    if ((this->empty()) || (right.empty()))
	return IA_ValueSet<ValueType>();
    {
	ValueType	lmax = this->max();
	ValueType	lmin = this->min();
	ValueType	rmax = right.max();
	ValueType	rmin = right.min();
	if (value_type_compare(&lmax, &rmin)<0 ||
	    value_type_compare(&rmax, &lmin)<0)
	    return IA_ValueSet<ValueType>();
    }
    if (this == &right) 
	return IA_ValueSet<ValueType>(*this);

    ValueType* temp = new ValueType[this->card() < right.card() 
				    ? this->size : right.size];

    while ((li < this->size) && (ri < right.size))
	if (value_type_compare(&this->value_array[li],&right.value_array[ri])<0)
	    li++;
	else {
	    if (this->value_array[li] == right.value_array[ri]) {
		temp[ci++] = right.value_array[ri];
		li++;
	    }
	    ri++;
	}
    return IA_ValueSet<ValueType>(temp, ci);
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator& (const ValueType &right) const {
    if (this->contains(right))
	return IA_ValueSet<ValueType>(right);
    else
	return IA_ValueSet<ValueType>();
}

template <class ValueType>
IA_ValueSet<ValueType> 
operator& (const ValueType &left, const IA_ValueSet<ValueType> &right) {
    if (right.contains(left))
	return IA_ValueSet<ValueType>(left);
    else
	return IA_ValueSet<ValueType>();
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator/ (const IA_ValueSet<ValueType> &right) const {
    unsigned int li = 0, ri = 0, ci = 0;

    if (this->empty())
	return IA_ValueSet<ValueType>();
    if (right.empty())
	return IA_ValueSet<ValueType>(*this);
    if (this == &right) 
	return IA_ValueSet<ValueType>();

    ValueType* temp = new ValueType[this->size];
    while ((li < this->size) && (ri < right.size)) {
	int cmp = value_type_compare(&this->value_array[li],
				     &right.value_array[ri]);
	if (cmp<0)
	    temp[ci++] = this->value_array[li++];
	else {
	    if (cmp==0)
		li++;
	    ri++;
	}
    }
    if (ri == right.size)
	for (;li<this->size;li++)
	    temp[ci++] = this->value_array[li];
    return IA_ValueSet<ValueType>(temp, ci);
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator/ (const ValueType &right) const {
    return *this / IA_ValueSet<ValueType>(right);
}

template <class ValueType>
IA_ValueSet<ValueType> 
operator/ (const ValueType &left, const IA_ValueSet<ValueType> &right) {
    if (right.contains(left))
	return IA_ValueSet<ValueType>();
    else
	return IA_ValueSet<ValueType>(left);
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator^ (const IA_ValueSet<ValueType> &right) const {
    unsigned int li = 0, ri = 0, ci = 0;

    if (this->empty())
	return IA_ValueSet<ValueType>(right);
    if (right.empty())
	return IA_ValueSet<ValueType>(*this);
    if (this == &right) 
	return IA_ValueSet<ValueType>();

    ValueType* temp = new ValueType[this->size + right.size];
    while ((li < this->size) && (ri < right.size)) {
	int cmp = value_type_compare(&this->value_array[li],
				     &right.value_array[ri]);
	if (cmp<0)
	    temp[ci++] = this->value_array[li++];
	else if (cmp>0)
	    temp[ci++] = right.value_array[ri++];
	else { 
	    li++;
	    ri++;
	}
    }
    if (ri == right.size)
	for (;li<this->size;li++)
	    temp[ci++] = this->value_array[li];
    if (li == this->size)
	for (;ri<right.size;ri++)
	    temp[ci++] = right.value_array[ri];
    return IA_ValueSet<ValueType>(temp, ci);
}

template <class ValueType>
IA_ValueSet<ValueType> 
IA_ValueSet<ValueType>::operator^ (const ValueType &right) const {

    if (this->contains(right))
	return IA_ValueSet<ValueType>(*this / right);
    else
	return IA_ValueSet<ValueType>(*this | right);
}

template <class ValueType>
IA_ValueSet<ValueType> 
operator^ (const ValueType &left, const IA_ValueSet<ValueType> &right) {

    if (right.contains(left))
	return IA_ValueSet<ValueType>(right / left);
    else
	return IA_ValueSet<ValueType>(right | left);
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator<= (const IA_ValueSet<ValueType> &right) const {

    unsigned int li = 0, ri = 0;

    if (this->empty())
	return 1;
    if (right.empty())
	return 0;
    if (this == &right) return 1;
    while ((li < this->size) && (ri < right.size)) {
	int cmp = value_type_compare(&this->value_array[li],
				     &right.value_array[ri]);
	if (cmp<0)
	    return 0;
	else {
	    if (cmp==0)
		li++;
	    ri++;
	}
    }
    if (ri == right.size)
	if (li == this->size)
	    return 1;
	else
	    return 0;
    return 1;
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator<= (const ValueType &right) const {

    if (this->size == 0) 
	return 1;
    if (this->size == 1)
	return this->max() == right;
    return 0;
}

template <class ValueType> 
int 
operator<= (const ValueType &left, const IA_ValueSet<ValueType> &right){

    if (right.contains(left)) 
	return 1;
    return 0;
}

template <class ValueType>
int 
IA_ValueSet<ValueType>::operator>= (const IA_ValueSet<ValueType> &right) const {
    return right <= *this;
}

template <class ValueType>
int 
IA_ValueSet<ValueType>::operator>= (const ValueType &right) const {
    return right <= *this;
}

template <class ValueType>
int 
operator>= (const ValueType &left, const IA_ValueSet<ValueType> &right){
    return right <= left;
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator< (const IA_ValueSet<ValueType> &right) const {

    unsigned int li = 0, ri = 0;

    if (this->empty())
	return 1;
    if (right.empty())
	return 0;
    if (this == &right) return 0;
    while ((li < this->size) && (ri < right.size)) {
	int cmp = value_type_compare(&this->value_array[li],
				     &right.value_array[ri]);
	if (cmp<0)
	    return 0;
	else {
	    if (cmp==0)
		li++;
	    ri++;
	}
    }

    if (li == this->size)
	if (ri == right.size)
	    return 0;
	else
	    return 1;
    return 0;
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator< (const ValueType &) const {

    return (this->size == 0);
}

template <class ValueType> 
int 
operator< (const ValueType &left, const IA_ValueSet<ValueType> &right){

    return (right.contains(left) && (right.size > 1));

}

template <class ValueType>
int 
IA_ValueSet<ValueType>::operator> (const IA_ValueSet<ValueType> &right) const {

    return (right < *this);

}

template <class ValueType>
int 
IA_ValueSet<ValueType>::operator> (const ValueType &right) const {
    return (right < *this);
}

template <class ValueType>
int 
operator> (const ValueType &left, const IA_ValueSet<ValueType> &right){
    return (right < left);
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator== (const IA_ValueSet<ValueType> &right) const {

    if (this->size != right.size)
	return 0;
    for (unsigned int i = 0; i < this->size; i++)
       if (this->value_array[i] != right.value_array[i])
	   return 0;
    return 1;
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator== (const ValueType &right) const {

    if (this->size != 1)
	return 0;
    return this->max() == right;
}

template <class ValueType> 
int 
operator== (const ValueType &left, const IA_ValueSet<ValueType> &right){

    if (right.size != 1)
	return 0;
    return right.max() == left;
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator!= (const IA_ValueSet<ValueType> &right) const {
    return !(*this == right);
}

template <class ValueType> 
int 
IA_ValueSet<ValueType>::operator!= (const ValueType &right) const {
    return !(*this == right);
}

template <class ValueType> 
int 
operator!= (const ValueType &left, const IA_ValueSet<ValueType> &right){
    return !(left == right);
}

template <class ValueType>
IA::LatticeRelation
compare(const IA_ValueSet<ValueType> &lhs, const IA_ValueSet<ValueType> &rhs) {

   if (lhs == rhs)
       return IA::EQUAL;
   if (lhs < rhs)
       return IA::P_SUBSET;
   if (lhs > rhs)
       return IA::P_SUPERSET;
   return IA::NO_REL;

}

template <class ValueType>
ostream& operator<< (ostream& st, const IA_ValueSet<ValueType>& vs) {
    int last_index = vs.size-1;
    st << "<"; 
    if (vs.size > 0) {
	for (int i=0; i < last_index; i++)
	    st << vs.value_array[i] << " ";
	if (vs.size > 0)
	    st << vs.value_array[last_index];
    }
    st << ">" << '\n';
    return st;
}

#endif // ValueSet_c_
