// Emacs: -*- C++ -*-
// fft-nd.c   Image Algebra C++ implementation of the N-D fast
//            Fourier transform using spatial transforms
//
// Cross-references: IA with Apps. pp. x-X; IA Handbook pp. y-Y

static char rcsid[] = "$Id: fft-nd.c,v 1.1 1994/02/20 04:05:09 rjj Exp $";

#include "imagealgebra.h"


//==============================================================
// Utility routines here
//==============================================================


////////////////////////////////////////////////////////////////////
// idiv2 - integer divide x by 2^n

inline int  idiv2( int x, int n ) { return x >> n; }


////////////////////////////////////////////////////////////////////
// ipow2 - integer calculate  2^n

inline int  ipow2( int n ) { return 1 << n; }


////////////////////////////////////////////////////////////////////
// ilog2 - calculate the log base 2 of an integer power of 2

int  ilog2(int i)
{
    int j=-1;
    while( i ) {
	j++;
	i = i >> 1;
    }
    return j;
}


////////////////////////////////////////////////////////////////////
// pointsetDimenWidth -
//    Calculate the  WIDTH  for a specific dimension, of the
//    bounding box for a given pointset.
//    NOTE that there is no guarantee that the  inf()  or  sup()
//    of a pointset are actually members of the pointset.

inline int
pointsetDimenWidth(const IA_IntPointSet &ips, int dimen) {
    return (ips.sup()-ips.inf()+1)[dimen];
}


////////////////////////////////////////////////////////////////////
// coordProduct - Calculate the product of a point\'s coordinates.
double
coordProduct( const IA_IntPoint &ip ) {
    double ipprod = 1;
    for ( int d=0; d < ip.dim(); d++ ) {
	ipprod *= ip[d];
    }
    return ipprod;
}



//==============================================================
//
// FFT specific utility routines follow
//
//==============================================================


////////////////////////////////////////////////////////////////////
// conjugate - Calculate the conjuate of a complex-valued image.

IA_ComplexDiscreteImage
conjugate(const IA_ComplexDiscreteImage &img)
{
#if IA_MAJOR_VERSION_NUMBER<1 && IA_MINOR_VERSION_NUMBER<10
    return compose( conj, img );
#else
    return conj(img);
#endif
}


////////////////////////////////////////////////////////////////////
// rho  - permutation function used in  reordering of input that the
//      Fast Fourier Transform requires  

int
rho(int i, int n)
{
    // Reverse the bit representation of i in field log2(n) bits
    int j=0, m = i;
    for( int k=ilog2(n); k!=0; k-- ) {
	j = (j<<1) + (m&1);
	m = m >> 1;
    }
    return j;

//==============================================================
// Sample code as suggested in the IA Handbook
//    int h, j=0, m=i;
//    for( int l=0; l<=ilog2(n)-1 ; l++ ) {
//	h = int( floor((double(m))/2.0) );
//	j = 2*j + m - 2*h;
//	m = h;
//    }
//    return j;
//==============================================================
}


////////////////////////////////////////////////////////////////////
// class R_m  - row permutation function, implemented as an
//     iac++ closure.  Instances provide a point to point mapping
//     object that looks like a function due to the definition of
//     operator().
//     m  should be a power of 2.
//     d  is the dimension utilized for mapping purposes.

class R_m: public IA_Closure<IA_IntPoint,IA_IntPoint> {
private:
    int  m;   // dimension width
    int  d;   // Which dimension are we dealing with ?
public:
    R_m( int dimenWidth, int whichDimen )
    : m(dimenWidth), d(whichDimen) {}

    IA_IntPoint  operator() (const IA_IntPoint &ip) const {
	IA_IntPoint rmip = ip;
	rmip[d] = rho(rmip[d], m);
	return  rmip;
    }

    IA_Closure<IA_IntPoint,IA_IntPoint> *clone_self() const {
	return new R_m(*this);
    }
};


////////////////////////////////////////////////////////////////////
// class F_i -
//     An iac++ closure used to implement the f() function as
//     defined for the FFT.
//     Instances of this class can be  COMPOSED  with an image,
//     in a similar fashion as a function.
//     Note that the quantity 2^(i-1) is computed once, when
//     an instance of this class is created.

class F_i: public IA_Closure<IA_IntPoint,IA_IntPoint> {
private:
    int  i_1;               // i-1
    int  ipow2_i_1;         // pre-compute 2^(i-1)
    int  d;                 // What dimension are we dealing with ?
public:
    F_i( int i, int whichDimen )
    : i_1(i-1), ipow2_i_1(ipow2(i-1)), d(whichDimen) {}

    IA_IntPoint  operator() (const IA_IntPoint &ip) const {
	IA_IntPoint fiip = ip;
	int val = idiv2(fiip[d], i_1) %2 ? 0 : ipow2_i_1;
	fiip[d] += val;
	return fiip;
    }

    IA_Closure<IA_IntPoint,IA_IntPoint> *clone_self() const {
        return new F_i(*this);
    }
};


////////////////////////////////////////////////////////////////////
// class G_i -
//     Instances represent the g() function as defined for the FFT.
//     See f() above for other notes.

class G_i: public IA_Closure<IA_IntPoint,IA_IntPoint> {
private:
    int  i_1;               // i-1
    int  ipow2_i_1;         // pre-compute 2^(i-1)
    int  d;                 // What dimension are we dealing with ?
public:
    G_i( int i, int whichDimen )
    : i_1(i-1), ipow2_i_1(ipow2(i-1)), d(whichDimen) {}

    IA_IntPoint  operator() (const IA_IntPoint &ip) const {
	IA_IntPoint giip = ip;
	int val = idiv2(giip[d], i_1) %2 ? ipow2_i_1 : 0;
	giip[d] -= val;
	return giip;
    }

    IA_Closure<IA_IntPoint,IA_IntPoint> *clone_self() const {
        return new G_i(*this);
    }
};


////////////////////////////////////////////////////////////////////
// class w_image -
//     An image closure implementation of the omega function
//     associated with the FFT.

class  w_image: public IA_ClosureI<IA_IntPoint,complex> {
private:
    int     w_i;    // The index this instance is concerned with.
    int     u;      // Pre-compute for this instance.
    double  piU;    // Pre-compute PI/u.
    int     d;      // Which dimension are we working on ?
public:
    w_image( IA_IntPointSet ps, int i, int whichDimen )
    :IA_ClosureI<IA_IntPoint,complex>(ps),
     w_i(i), u(ipow2(i-1)), d(whichDimen) {
    piU = PI/double(u); }

    complex operator() (const IA_IntPoint& xy) const {
	int evenodd = (idiv2(xy[d],w_i-1) % 2); // 0 if even
	double v    = piU * double(xy[d]%u);

	if (evenodd) {
	    return  complex( -cos(v), -sin(v) );
	} else {
	    return  complex(  cos(v),  sin(v) );
	}
    }
    
    IA_ClosureI<IA_IntPoint,complex> *clone_self() const {
        return new w_image(*this);
    }
};


////////////////////////////////////////////////////////////////////
// fft - Calcuate the Fast Fourier Transform.
//   This version is very similar to the one presented in the
//   IA Handbook, except that has been genealized to N dimensions.
//   Because of this the transpose() is no longer needed.

IA_ComplexDiscreteImage
fft(const IA_ComplexDiscreteImage & image) {

    IA_ComplexDiscreteImage  a(image);     // Make a local copy

    // Calculate the width of each dimension.
    // It should be a power of 2.
    IA_IntPoint a_dim_width( a.domain().sup() - a.domain().inf() + 1 );


    // We will do the forward transform for each dimension of the image.
    // iac++ library first dimension index is zero (0), not one (1).

    IA_IntPointSet  aps = a.domain();

    for ( int d=0; d< a_dim_width.dim(); d++) {
	R_m  rm(a_dim_width[d], d);
	a = compose(a,rm).restrict(aps);  // permutate rows of the image

	for ( int i=1;
	      i<=ilog2( a_dim_width[d] );
	      i++ ) {

	    w_image w(aps,i,d);  // Create an omega image
	    F_i fi(i,d);         // Create a  f()  `function`
	    G_i gi(i,d);         // Create a  g()  `function`

	    // We want to say: a = compose(a,gi) + w*compose(a,fi);

	    // However, compose() creates a comprehensive image,
	    // it must be restricted to make it an extensive image.

	    a = compose(a,gi).restrict(aps)
		+ w*compose(a,fi).restrict(aps);
	}
    }
    return a;
}


////////////////////////////////////////////////////////////////////
// fft_inv - Calculate the inverse FFT image using the forward FFT

IA_ComplexDiscreteImage
fft_inv(const IA_ComplexDiscreteImage & a) {
    IA_IntPoint a_dim_width( a.domain().sup() - a.domain().inf() + 1 );

    double nm = coordProduct( a_dim_width );

    return conjugate( fft( conjugate(a) )/nm );
}
