// Emacs: -*- C++ -*-
// fft-nd-iter.c   Image Algebra C++ implementation of the ND fast
//         Fourier transform using spatial transforms.
//         This version is implemented with an iterative algorithm.
//
// Cross-references: IA with Apps. pp. x-X; IA Handbook pp. y-Y

static char rcsid[] = "$Id: fft-nd-iter.c,v 1.1 1994/02/20 04:05:09 rjj Exp $";

#include "imagealgebra.h"

typedef IA_PSIter<IA_IntPoint> IA_IPSIter;


//==============================================================
// Utility routines here
//==============================================================


////////////////////////////////////////////////////////////////////
// idiv2 - integer divide x by 2^n

inline int  idiv2( int x, int n ) { return x >> n; }


////////////////////////////////////////////////////////////////////
// ipow2 - integer calculate  2^n

inline int  ipow2( int n ) { return 1 << n; }


////////////////////////////////////////////////////////////////////
//ilog2 - calculate the log base 2 of an integer power of 2

int ilog2(int i) {

    int j=-1;
    while( i ) {
	j++;
	i = i >> 1;
    }
    return j;
}


////////////////////////////////////////////////////////////////////
// pointsetDimenWidth -
//    Calculate the  WIDTH  for a specific dimension, of the
//    bounding box for a given pointset.
//    NOTE that there is no guarantee that the  inf()  or  sup()
//    of a pointset are actually members of the pointset.

inline int
pointsetDimenWidth(const IA_IntPointSet &ips, int dimen) {
    return (ips.sup()-ips.inf()+1)[dimen];
}


////////////////////////////////////////////////////////////////////
// coordSum -  Calculate the sum of a point\'s coordinates.

int
coordSum(const IA_IntPoint &ip) {
    int ipsum = 0;
    for ( int d=0; d < ip.dim(); d++ ) {
	ipsum += ip[d];
    }
    return ipsum;
}


////////////////////////////////////////////////////////////////////
// coordProduct - Calculate the product of a point\'s coordinates.

double
coordProduct(const IA_IntPoint &ip) {
    double ipprod = 1;
    for ( int d=0; d < ip.dim(); d++ ) {
	ipprod *= ip[d];
    }
    return ipprod;
}


//==============================================================
//
// FFT specific utility routines follow
//
//==============================================================


////////////////////////////////////////////////////////////////////
// conjugate - Calculate the conjuate of a complex-valued image.

IA_ComplexDiscreteImage
conjugate(const IA_ComplexDiscreteImage &img) {

#if IA_MAJOR_VERSION_NUMBER<1 && IA_MINOR_VERSION_NUMBER<10
    IA_ComplexDiscreteImage tmp(img.domain(),complex(0.0,0.0));
    IA_IPSIter psi(img.domain());

    IA_IntPoint ip;
    while(psi(ip)) {
	tmp[ip] = conj(img(ip));
    }
    return tmp;
#else
    // iac++ library now supports this directly.
    return conj(img);
#endif
}


////////////////////////////////////////////////////////////////////
// rho  - permutation function used in  reordering of input that the
//      Fast Fourier Transform requires  

int
rho(int i, int n) {

    // Reverse the bit representation of i in field log2(n) bits
    int j=0, m = i;
    for( int k=ilog2(n); k!=0; k-- ) {
	j = (j<<1) + (m&1);
	m = m >> 1;
    }
    return j;
} //end p


////////////////////////////////////////////////////////////////////
// permute_rows - uses p function above to permute the
//  dimension  whichDimen  rows.

IA_ComplexDiscreteImage
permute_rows(IA_ComplexDiscreteImage img, int whichDimen) {

    int dWidth = pointsetDimenWidth(img.domain(), whichDimen);
    IA_ComplexDiscreteImage tmp(img.domain(),0.0);
    IA_IPSIter psi(img.domain());

    IA_IntPoint ip, pip;
    while(psi(ip)) {
	pip = ip;
	pip[whichDimen] = rho( pip[whichDimen], dWidth );
	tmp[ip] = img( pip );
    }
    return tmp;
}


////////////////////////////////////////////////////////////////////
// f -  An iterative version of the f() function.
//     It handles the odd indices.

IA_ComplexDiscreteImage
f (int i, IA_ComplexDiscreteImage img, int d) {

    IA_ComplexDiscreteImage tmp( img );
    IA_IPSIter psi(img.domain());

    IA_IntPoint ip, fip;
    int evenodd, ipow2i_1(ipow2(i-1));

    while(psi(ip)) {
	fip = ip;
	evenodd = idiv2( ip[d], i-1 ) % 2;
	if(evenodd == 0) {
	    fip[d] += ipow2i_1;
	    tmp[ip] = img( fip );
	}
    }
    return tmp;
}


////////////////////////////////////////////////////////////////////
// g -  An iterative version of the g() function.
//     It handles the odd indices cases.

IA_ComplexDiscreteImage
g(int i,IA_ComplexDiscreteImage img, int d) {

    IA_ComplexDiscreteImage tmp(img);
    IA_IPSIter psi(img.domain());

    IA_IntPoint ip, gip;
    int evenodd, ipow2i_1(ipow2(i-1));

    while(psi(ip)) {
	gip = ip;
	evenodd = idiv2(ip[d], i-1) % 2;
	if(evenodd) {
	    gip[d] -= ipow2i_1;
	    tmp[ip] = img( gip );
	}
    }
    return tmp;
}


////////////////////////////////////////////////////////////////////
// class w_image -
//    An image closure implementation of the omega function
//    associated with the FFT.

class w_image: public IA_ClosureI<IA_IntPoint,complex> {
private:
    int         w_i;
    int         u;
    double      piU;
    int         d;
public:
    w_image( IA_IntPointSet ps, int i, int whichDimen )
    :IA_ClosureI<IA_IntPoint,complex>(ps), 
     w_i(i), u(ipow2(i-1)), d(whichDimen) {
	 piU = PI/double(u);
     }

    complex operator() (const IA_IntPoint& xy) const {
        int evenodd( idiv2(xy[d],w_i-1) % 2 );
	double v = piU * double(xy[d] % u);

	if(evenodd) {
	    return  complex( -cos(v), -sin(v) );
	} else {
	    return  complex(  cos(v),  sin(v) );
	}
    }

    IA_ClosureI<IA_IntPoint,complex> *clone_self() const {
        return new w_image(*this);
    }
};


////////////////////////////////////////////////////////////////////
// fft -  Calcualte the Fast Fourier Transform.
//   This version utilizes iterative versions of the f() and g()
//   functions, it does not utilize composition.

IA_ComplexDiscreteImage
fft(const IA_ComplexDiscreteImage & image) {

    IA_ComplexDiscreteImage a(image);  // Make a local copy

    // We will do the forward transform for each dimension of the image.
    // iac++ library first dimension index is zero (0), not one (1).

    IA_IntPointSet  aps = a.domain();

    for ( int d=0; d < a.domain().dim(); d++ ) {
	a = permute_rows(a, d);

	// The following is the acutal FFT code for single dimension.
	for( int i = 1;
	     i <= ilog2( pointsetDimenWidth(aps,d) );
	     i++ ) {

	    w_image w(a.domain(), i, d);

	    a = g(i, a, d) + w * f(i, a, d);
	}
    }
    return a;
}


////////////////////////////////////////////////////////////////////
// fft_inv - Calculate the inverse FFT image using the forward FFT

IA_ComplexDiscreteImage
fft_inv(const IA_ComplexDiscreteImage & a) {
    IA_IntPoint a_dim_width( a.domain().sup() - a.domain().inf() + 1 );

    double nm = coordProduct( a_dim_width );  // area of point set

    return conjugate( fft( conjugate(a) )/nm );
}
