package JIMSCore;
/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    State.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

  This class represents the internal state of the simulator.
  It is also used as a layer between the simulator control and
  state variables (such as main memory and register values).
  The simulator manipulates its state by using methods in
  this class (instead of modifying main memory and register
  modules directly).

*/
import java.util.Vector;

public class State {

  private MainMemory mainMemory;
  // Reference to the MainMemory system

  private RegisterFile registerFile;
  // Reference to the CPU RegisterFile (GPR and FPR)

  private Coprocessor0 coprocessor0;
  // Reference to Coprocessor0, CPU control.

  private Coprocessor1 coprocessor1;
  // Reference to Coprocessor1, FP

  private StateHistoryBuffer currentStateHistoryBuffer;
  // Reference to a state history buffer.  If not null,
  //   then we need to record state changes as they
  //   happen.  Otherwise, if this reference is null,
  //   state history changes are not being recorded.
  // NOTE: This is set by the setStateHistoryBuffer() method.

  private StringBuffer sbInputBuffer;  // From client into the I/O console
  private StringBuffer sbOutputBuffer;  // From simulator to I/O console

  private boolean baFlag[];
  private Vector vModifiedMemoryAddresses;
  // A vector of Integer objects, representing a list of
  //   memory address that have been modified.  Entries to this
  //   list are added only when storeByte() is called.  To
  //   obtain the list of addresses, use vGetModifiedAddresses()

  public static final int NUM_FLAGS = 10;
  public static final int FLAG_SC_PRINT_INT    = 0;
  public static final int FLAG_SC_PRINT_FLOAT  = 1;
  public static final int FLAG_SC_PRINT_DOUBLE = 2;
  public static final int FLAG_SC_PRINT_STRING = 3;
  public static final int FLAG_SC_READ_INT     = 4;
  public static final int FLAG_SC_READ_FLOAT   = 5;
  public static final int FLAG_SC_READ_DOUBLE  = 6;
  public static final int FLAG_SC_READ_STRING  = 7;
  public static final int FLAG_SC_SBRK         = 8;
  public static final int FLAG_SC_EXIT         = 9;

  public static final String STATE_CHANGE_FLAG = "F";  // FLAG
  public static final String STATE_CHANGE_MEM = "M";  // MEM
  public static final String STATE_CHANGE_GPR = "R";  // GPR
  public static final String STATE_CHANGE_CP0 = "0";  // CP0
  public static final String STATE_CHANGE_CP1 = "1";  // CP1

  public State() {
	reset();
  }    

  public void reset() {
	// This method (re)initializes the entire state to the
	//   initial condition.

	mainMemory = new MainMemory();
	registerFile = new RegisterFile();

	baFlag = new boolean[NUM_FLAGS];

	sbInputBuffer = new StringBuffer();
	sbOutputBuffer = new StringBuffer();

	coprocessor0 = new Coprocessor0();
	coprocessor1 = new Coprocessor1();

	vModifiedMemoryAddresses = new Vector();

	iModifiedMemoryCount = 0;

  }  // end method reset()      

  public Coprocessor0 getCoprocessor0() {
	return coprocessor0;
  }    

  public void setStateHistoryBuffer(StateHistoryBuffer stateHistoryBuffer) {
	// Set stateHistoryBuffer == null to disable state change recording.
	currentStateHistoryBuffer = stateHistoryBuffer;
  }    

  public StateHistoryBuffer getStateHistoryBuffer() {
	// Return the buffer currently used to hold state change information.
	//   This may return null if state change recording has been disabled.
	return currentStateHistoryBuffer;
  }    

  private void addStateValue(String s) {
	// This method should be called BEFORE a state change is
	//   recorded.  The state change buffer saves values
	//   before the state is actually changed.  If the
	//   currentStateHistoryBuffer == null, then state change
	//   recording has been disabled.

	if (currentStateHistoryBuffer != null)
	  currentStateHistoryBuffer.addStateValue(s);
	// else
	//   State change recording has been disabled.

  }  // end method addStateValue(String)    

  public void setFlag(int x, boolean value) {

	String s = STATE_CHANGE_FLAG + " " + x + " ";
	if (baFlag[x])
	  s += "T";
	else
	  s += "F";
	addStateValue(s);

	baFlag[x] = value;
  }    

  public boolean getFlag(int iIndex) {
	return baFlag[iIndex];
  }    

  public void storeByte(int iAddress, byte newValue) {
	// Store the newValue byte at iAddress.

	byte currValue = loadByte(iAddress);
	if (currValue != newValue) {
	  addStateValue( STATE_CHANGE_MEM + " " +
		Utility.sAsHexPadded(iAddress,8) + " " +
		Utility.sAsHexPadded(currValue,2) );
	}
	// Even if the value is the same, we still should apply
	//   the memory store so that the "modified" count
	//   is incremented.
	mainMemory.storeByte(iAddress, newValue);

	addToModifiedMemoryList(iAddress);

  }  // end method storeByte(int, byte)    

  public void storeHalfWord(int iAddress, int newValue) {
	// Store the newValue halfword at iAddress

	int currValue = loadHalfWord(iAddress);
	if (currValue != newValue) {

	  byte value1 = (byte)( (newValue & 0x0000FF00L) >>> 8  );
	  byte value2 = (byte)( (newValue & 0x000000FFL)        );

	  storeByte(iAddress, value1);
	  storeByte(iAddress+1, value2);

	}
  }  // end method storeHalfWord(int, int)    

  public void storeWord(int iAddress, int newValue) {
	// Store the newValue word at iAddress

	int currValue = loadWord(iAddress);
	if (currValue != newValue) {

	  byte value1 = (byte)( (newValue & 0xFF000000L) >>> 24 );
	  byte value2 = (byte)( (newValue & 0x00FF0000L) >>> 16 );
	  byte value3 = (byte)( (newValue & 0x0000FF00L) >>> 8  );
	  byte value4 = (byte)( (newValue & 0x000000FFL)        );

	  storeByte(iAddress, value1);
	  storeByte(iAddress+1, value2);
	  storeByte(iAddress+2, value3);
	  storeByte(iAddress+3, value4);

	}

  }  // end method storeWord(int, int)    

  public byte loadByte(int iAddress) {
	return mainMemory.loadByte(iAddress);
  }    

  public int loadHalfWord(int iAddress) {
	return mainMemory.loadHalfWord(iAddress);
  }    

  public int loadWord(int iAddress) {
	return mainMemory.loadWord(iAddress);
  }    

  public void setRegister(int iReg, int iNewValue) {

	if (iReg == 0)
	  return;

	// Get the current value of REG[r]
	int iCurrValue = iGetRegister(iReg);

	// If the value has changed, update with the new value...
	if (iCurrValue != iNewValue) {
	  addStateValue(
		STATE_CHANGE_GPR + " " +
		iReg + " " +
		Utility.sAsHexPadded(iCurrValue,8)
	  );
	  //registerFile.setReg(iReg, iNewValue);
	  registerFile.iaReg[iReg] = iNewValue;
	}

  }    

  public void setCP0Register(int iReg, int iNewValue) {

	// Get the current value of REG[r]
	int iCurrValue = iGetCP0Register(iReg);

	// If the value has changed, update with the new value...
	if (iCurrValue != iNewValue) {
	  addStateValue( STATE_CHANGE_CP0 + " " +
		iReg + " " +
		Utility.sAsHexPadded(iCurrValue,8)
	  );
	  //coprocessor0.setReg(iReg, iNewValue);
	  coprocessor0.iaReg[iReg] = iNewValue;
	}
   
  }  // end method setCP0Register(int, int);    

  public void setCP1Register(int iReg, int iNewValue) {

	// Get the current value of REG[r]
	int iCurrValue = iGetCP1Register(iReg);

	// If the value has changed, update with the new value...
	if (iCurrValue != iNewValue) {
	  addStateValue( STATE_CHANGE_CP1 + " " +
		iReg + " " +
		Utility.sAsHexPadded(iCurrValue,8)
	  );
	  //coprocessor1.setReg(iReg, iNewValue);
	  coprocessor1.iaReg[iReg] = iNewValue;
	}
   
  }  // end method setCP0Register(int, int);    

  public int iGetRegister(int iReg) {
	//return registerFile.iGetReg(iReg);
	return registerFile.iaReg[iReg];
  }    

  public int iGetCP0Register(int iReg) {
	//return coprocessor0.iGetReg(iReg);
	return coprocessor0.iaReg[iReg];
  }    

  public int iGetCP1Register(int iReg) {
	//return coprocessor1.iGetReg(iReg);
	return coprocessor1.iaReg[iReg];
  }    

  public void appendInputBuffer(String s) {
	sbInputBuffer.append(s);
  }    

  public void appendOutputBuffer(String s) {
	sbOutputBuffer.append(s);
  }    

  public StringBuffer sbGetOutputBuffer() {
	StringBuffer sb = sbOutputBuffer;
	sbOutputBuffer = null;
	return sbOutputBuffer;
  }  // end mehod sbGetOutputBuffer()    

  public boolean isOutputBufferEmpty() {
	return (sbOutputBuffer == null);
  }    

  public Vector vGetModifiedAddresses() {
	// This method returns a vector of Integer objects.  Each integer
	//   represents a memory address that has been modified in
	//   some way.  This method can be implemented in two ways:

	// [1] For a slower, but less memory intensive implementation, you
	//   can use the following:

//	return mainMemory.vGetModifiedAddresses();

	// NOTE: If using the above implementation, then comment
	//   out the body of the addToModifiedMemoryList()
	//   method below (but keep the method declaration).

	// [2] The following is faster, but uses more memory:
	return vModifiedMemoryAddresses;
  }  // end method vGetModifiedAddresses()      

private void addToModifiedMemoryList(int iAddress) {
  // Insert the value iAddress as an Integer object into
  //   vModifiedMemoryAddresses.  The elements of this
  //   vector are to be sorted in increasing order,
  //   and contain no duplicate entries.

  int iModifiedCount = vModifiedMemoryAddresses.size();

  int i = 0;
  for (i = 0; i < iModifiedCount; i++) {
	int iCompareAddress =
	  ((Integer) vModifiedMemoryAddresses.elementAt(i)).intValue();
	if (iAddress == iCompareAddress)
	  return; // The address is already in the modified addresses list.
	if (iAddress < iCompareAddress)
	  break;
  }

  iModifiedMemoryCount++;

  vModifiedMemoryAddresses.insertElementAt(new Integer(iAddress), i);

} // end method addToModifiedMemoryList(int)          

  public String toString() {
	String result = "";
	result += registerFile;
	result += "\n" + mainMemory;
	return result;
  }    

  // An array containing various state flags.

  public int iModifiedMemoryCount;
}