package JIMSCore;
/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    Simulator.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

  Simulator is the primary simulator class.  It models the
  (MIPS) CPU execution path: IF, ID, and EX.  At this time,
  this is not a pipelined model.

  The Simulator class automatically handles the state change
  buffer and checkpoints.  This allows the user of this class
  to easily step forward or step backwards.

*/
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.io.PrintWriter;
import java.io.BufferedWriter;
import java.io.FileWriter;

public class Simulator {

  // ** DEFAULT SIMULATOR SETTINGS ***************************
  private static final int CHECKPOINT_CREATION_FACTOR = 5;
  // See iCheckpointCreationFactor

  private static final boolean ENABLE_HISTORY_RECORDING = true;
  // See bHistoryRecordingEnabled
  // *********************************************************

  // ** EXCEPTION CODES **************************************
  public static final int EXCEPTION_NONE = -1;
  // No exception has occurred.

  // The following are exceptions that can not be recovered
  //   from.  They represent an internal error in the simulator.
  public static final int EXCEPTION_FETCH_ERROR = 0;
  public static final int EXCEPTION_DECODE_ERROR = 1;
  public static final int EXCEPTION_EXECUTE_ERROR = 2;

  public static final int EXCEPTION_FP_REQUIRED = 3;
  // Attempted to execute a FP instruction when no FP
  //   hardware is available.  Begin FP emulation.

  public static final int EXCEPTION_OUTPUT_WAITING = 4;
  // System is waiting for output buffer to be read.

  public static final int EXCEPTION_INPUT_WAITING = 5;
  // System is waiting for input buffer to be satisfied.

  public static final int EXCEPTION_EXIT = 6;
  // Indicates that an EXIT system call has been made,
  //   which should mean to reset the simulator.

  public static final int EXCEPTION_INVALID_OPCODE = 7;
  // The EX() stage has attempted to execute an invalid
  //   opcode.  This can be caused by instruction memory
  //   being overwritten by the user program, the simulator
  //   not supporting an instruction used by the assembler,
  //   the assembler generating incorrect code, etc.

  public static final int EXCEPTION_SYSCALL = 8;
  // SYSCALL instruction was issued.  Look at value of
  //   reg[$v0] to determine what action to take.

  public static final int EXCEPTION_BREAK = 9;
  // BREAK instruction was issued.  Look at value of
  //   low 24-bits to get "code" parameter.

  public static final int EXCEPTION_OVERFLOW = 10;
  // OVERFLOW caused by an ADDI, ADD, or SUB instruction.

  public static final int EXCEPTION_FP_ERROR = 11;
  // Any general FP error causes this exception.  In reality,
  //   it is caused only by a divide by zero.

  // **INCOMPLETE
  public static final int EXCEPTION_UNALIGNED_MEMORY_LOAD = 12;
  public static final int EXCEPTION_UNALIGNED_MEMORY_STORE = 13;

  // *********************************************************

  // ** MISC CONSTANTS ***************************************
  public static final int NO_GUARD = -1;
  public static final int NO_BREAKPOINT = -1;
  // These two constants are the default values for
  //   iGuardIndex and iBreakpointIndex respectively.

  private static final long MAX_CYCLE_INDEX = Long.MAX_VALUE;
  // The cycle index should be an unsigned infinite time index.
  //   Java does not support unsigned numbers, nor can integers
  //   be infinite.  So we must define a maximum value for
  //   the cycle index.  If this maximum index is reached,
  //   the cycle index is reset back to 0, and the history buffer
  //   information is cleared (because it can not be undone).
  // NOTE: A "long" is used to reduce the probability that
  //   the size of the cycle index ever becomes an issue.
  //   To reach this maximum cycle index, the simulated program
  //   would need to execute an extremely large number of
  //   instructions (more than 9.2 * 10^18).

  private static final String CHECKPOINT_GPR = "R";  // GPR
  private static final String CHECKPOINT_CP0 = "0";  // CP0
  private static final String CHECKPOINT_CP1 = "1";  // CP1
  private static final String CHECKPOINT_MEM = "M";  // MEM
  // *********************************************************

  // ** INSTANCE VARIABLES ***********************************
  private State state;
  // The state holds the state of memory and register values.

  private Vector vBreakpoint;
  // Vector of Integer types, stores location of user
  //   defined breakpoints.

  private Vector vStateHistory;
  // Vector of StateHistoryBuffer, store state changes
  //   per each cycle.

  private Vector vCheckpoint;
  // Vector of CheckpointBuffer, store entire system state
  //   periodically (so that vStateHistory can be cleared).

  private BreakpointList breakpointList;
  // Maintains a list of user-defined breakpoints.
  private int iBreakpointIndex;
  // If a breakpoint has been encounted, this integer represents
  //   which breakpointi t was (it is an index into breakpointList).

  private GuardList guardList;
  // Maintains a list of user-defined guards.
  private int iGuardIndex;
  // If a guard has been encounted, this integer represents
  //   which guard it was (it is an index into guardList).

  private int iIR;
  // This is the Instruction Register, which is an internal
  //   CPU register and not part of the Register File.

  private long lLastCheckPointCycleIndex;
  // This integer variable keeps track of what cycle the last
  //   checkpoint was created.  A value of -1 means that no
  //   checkpoint has ever been created.

  private long lCycleIndex;
  // What cycle the simulator is on.  Incremented in the
  //   performCycle() method.  

  private StringBuffer sbOutputFromUndo;
  // When performing an undo, sometimes it is necessary to
  //   reexecute code from a previous checkpoint.  If an exception
  //   occures, usually the re-execution process must terminate.
  //   But if the exception is due to an output request, there is no
  //   need to terminate the re-execution process (since the output
  //   should already be available from the state history).  The
  //   output during re-execution is stored in this StringBuffer object.
  //   After performing an undo, the client should use the method
  //   sbGetOutputFromUndo() to determine what (if any) the output is.

  private CheckpointBuffer pendingCheckpoint;
  // In order to determine if a checkpoint should be created,
  //   we need to know the size of the checkpoint if it were
  //   created.  This means creating a temporary checkpoint.
  //   If we decide to keep the checkpoint, we store its object
  //   reference in pendingCheckpoint, which will automatically
  //   be used when makeCheckpoint() is called.

  private int iStateHistoryByteSize;
  // For better performance, the size of the state history buffer
  //   is incremented each cycle (by the amount of whatever information
  //   was added to the history buffer).  This means that the
  //   history buffer size does not need to be re-calculated whenever
  //   we check for if a checkpoint should be created or not.

  private int iExceptionCode;
  // Used to keep track of which exception has triggered.
  //   Only one exception may trigger at a time.  The exception
  //   code is one of the EXCEPTION_XXX constants.

  private InstructionDecodeBuffer decodeBuffer;
  // This holds the results of the Instruction Decode phase.
  //   It is initialized in the ID() method and used in EX().

  private boolean bHistoryRecordingEnabled;
  // Use this to toggle whether or not history recording
  //   is enabled.  Disabling this feature improves
  //   runtime performance, decrease memory consumption,
  //   but disables support for undo operations.

  private int iCheckpointCreationFactor;
  // Larger values conserve more memory, but increase
  //   the distance between checkpoint creation.
  // *********************************************************

  public Simulator(boolean bHistoryRecordingEnabled, int iCheckpointCreationFactor) {

	this.bHistoryRecordingEnabled = bHistoryRecordingEnabled;
	this.iCheckpointCreationFactor = iCheckpointCreationFactor;

	reset();

  }  // end constructor Simulator(boolean, int)    

  public Simulator() {
	this(ENABLE_HISTORY_RECORDING, CHECKPOINT_CREATION_FACTOR);
  }    

  public void reset() {

	state = new State();

	vBreakpoint = new Vector();

	if (bHistoryRecordingEnabled) {
	  vStateHistory = new Vector();
	  vCheckpoint = new Vector();
	} else {
	  vStateHistory = null;
	  vCheckpoint = null;
	}

	breakpointList = new BreakpointList();
	iBreakpointIndex = NO_BREAKPOINT;

	guardList = new GuardList();
	iGuardIndex = NO_GUARD;

	iIR = 0;

	lLastCheckPointCycleIndex = -1;

	lCycleIndex = 0;

	sbOutputFromUndo = null;

	pendingCheckpoint = null;

	iStateHistoryByteSize = 0;

	iExceptionCode = EXCEPTION_NONE;

  }  // end method reset()    

private void performCycle() throws SimulatorException {
  // This the primary method responsible for simulating
  //   the CPU cycles.  This method should only be called
  //   from the iPerformStep(int) method.

  StateHistoryBuffer stateHistoryBuffer = null;
  // Keep stateHistoryBuffer == null to disable state change recording.
  //   This history buffer is passed to the "state" instance
  //   by the setStateHistoryBuffer() method, and then later
  //   retrived by using the getStateHistoryBuffer() method.

  if (bHistoryRecordingEnabled) {

	if (bTimeToMakeCheckpoint()) {

	  makeCheckpoint();
	  // Since we just performed a checkpoint, we do not need to
	  //   record state changes for this cycle.  This means
	  //   that stateHistoryBuffer should remain null.

	  performCheckpointCulling();

	}
	else {

	  if ((vCheckpoint != null)
		&& (vCheckpoint.size() > 0)
		&& (lLastCheckPointCycleIndex != -1)) {

                // lLastCheckPointCycleIndex should always be the cycle
                //   index of the most recently created checkpoint.

		lLastCheckPointCycleIndex =
		  ((CheckpointBuffer) vCheckpoint.lastElement()).lGetCycleIndex();
	  }

	  if (lCycleIndex > lLastCheckPointCycleIndex) {

		// We only want to record state changes if the current
		//   cycle is past (greater than) that of the last checkpoint.
		// NOTE:
		//   If (lCycleIndex == lLastCheckpointCycleIndex then
		//     a checkpoint was created on the last cycle.
		//   If (lCycleIndex < lLastCheckpointCycleIndex then
		//     there is a bug in the system.

		stateHistoryBuffer = new StateHistoryBuffer(lCycleIndex);

	  }

	}

  }

  state.setStateHistoryBuffer(stateHistoryBuffer);

  // -- BEGIN CYCLE -----------------------------------------

  // INSTRUCTION FETCH
  IF();

  // INSTRUCTION DECODE
  ID();

  // EXECUTE
  EX();

  // EXCEPTION-HANDLER
  if (iExceptionCode != EXCEPTION_NONE) {

	int ExcCode = -1;

	switch (iExceptionCode) {

	  case EXCEPTION_FP_REQUIRED :
		ExcCode = Coprocessor0.EXCCODE_CpU;
		break;

	  case EXCEPTION_OUTPUT_WAITING :
		// would not be set yet
		break;

	  case EXCEPTION_INPUT_WAITING :
		// would not be set yet
		break;

	  case EXCEPTION_EXIT :
		// would not be set yet
		break;

	  case EXCEPTION_INVALID_OPCODE :
		ExcCode = Coprocessor0.EXCCODE_RI;
		break;

	  case EXCEPTION_SYSCALL :
		// Technically, the exception handler should process
		//   by the SYSCALL.  The Cause register would indicate
		//   that the exception is a SYSCALL, then it can
		//   look at $v0 to determine what action to take.
		// Crurently, for this simulator, user I/O is very
		//   simplified and is processed internally. We can
		//   examine the FLAG_SC_XXX flags to determine what
		//   action to take. 

		iExceptionCode = EXCEPTION_OUTPUT_WAITING;
		if (state.getFlag(State.FLAG_SC_PRINT_INT)
		  | state.getFlag(State.FLAG_SC_PRINT_FLOAT)
		  | state.getFlag(State.FLAG_SC_PRINT_DOUBLE)
		  | state.getFlag(State.FLAG_SC_PRINT_STRING))
		  break;

		iExceptionCode = EXCEPTION_INPUT_WAITING;
		if (state.getFlag(State.FLAG_SC_READ_INT)
		  | state.getFlag(State.FLAG_SC_READ_FLOAT)
		  | state.getFlag(State.FLAG_SC_READ_DOUBLE)
		  | state.getFlag(State.FLAG_SC_READ_STRING))
		  break;

		iExceptionCode = EXCEPTION_EXIT;
		if (state.getFlag(State.FLAG_SC_EXIT))
		  break;

		// If no relavant flag was set, we proceed
		//   with going to the exception handler
		//   as expected.
		ExcCode = Coprocessor0.EXCCODE_Syscall;
		break;

	  case EXCEPTION_BREAK :
		ExcCode = Coprocessor0.EXCCODE_Bp;
		break;

	  case EXCEPTION_OVERFLOW :
		ExcCode = Coprocessor0.EXCCODE_Ov;
		break;

	  case EXCEPTION_FP_ERROR :
		ExcCode = Coprocessor0.EXCCODE_FPE;
		break;

	  default :
		// This cases should never happen -- if so, it indicates
		//   an internal bug in the simulator.
		System.err.println(
		  "Internal Simulator Error (Exception Code == " + iExceptionCode + ")");
		break;
	}

	if (ExcCode != -1) {

	  // Set the EPC register in Coprocessor0.
	  int iCurrentPC = state.iGetRegister(RegisterFile.R_PC);
	  state.setCP0Register(Coprocessor0.R_EPC, iCurrentPC);
	  // NOTE: In a pipelined environment, we would need to examine
	  //   if Cause(BD) should be set.  Cause(BD) would be set
	  //   if the exception occurred in the branch delay slot.
	  //   Currently, this simulator does not emulate branch
	  //   delayed slots (since it is not a pipelined model).

	  // CPU changes into kernel mode and disables interrupts.
	  // **INCOMPLETE

	  // Set the ExcCode bits in the CP0(Cause) register.
	  ExcCode = ExcCode << 2;
	  int iCause = state.iGetCP0Register(Coprocessor0.R_CAUSE);
	  iCause = iCause | ExcCode;
	  state.setCP0Register(Coprocessor0.R_CAUSE, iCause);

	  // On address exception, BadVaddr register is also set in CP0.
	  // **INCOMPLETE

	  // Set the PC to the exception entry point.
	  state.setRegister(RegisterFile.R_PC, MainMemory.DEFAULT_EXCEPTION_ADDRESS);

	}

  }

  // Increment the cycle counter.
  incrementCycleIndex();

  // -- END CYCLE -------------------------------------------

  if (bHistoryRecordingEnabled && (stateHistoryBuffer != null)) {
	// Push the state history buffer into the state history vector,
	//   then disable the recording of state changes.
	addToStateHistoryBuffer(stateHistoryBuffer);
	state.setStateHistoryBuffer(null);
  }

} // end method performCycle()              

// ** CYCLE METHODS ******************************

  private void IF() {

	try {

	  // Fetch the instruction at the Program Counter ($PC).
	  int iCurrentPC = state.iGetRegister(RegisterFile.R_PC);

	  // Put the fetched instruction into the Instruction Register ($IR).
	  iIR = state.loadWord(iCurrentPC);

	  // Increment the Program Counter.
	  state.setRegister(RegisterFile.R_PC, iCurrentPC + 4);

	} catch (Exception e) {

	  iExceptionCode = EXCEPTION_FETCH_ERROR;
	  System.err.println("CRITICAL ERROR IN FETCH: " + e);

	}

  }  // end method IF()    

  private void ID() {

	// Decode the instruction currently in the Instruction
	//   Register ($IR).  The results are returned in an
	//   InstructionDecodeBuffer.

	InstructionDecoder id = new InstructionDecoder();
	decodeBuffer = id.decode(iIR);

	if (decodeBuffer == null) {
	  // Some critical error occured
	  iExceptionCode = EXCEPTION_DECODE_ERROR;
	  System.err.println("CRITICAL ERROR IN DECODE");
	}

  }  // end method ID()    

  private void EX() {

	InstructionExecuter ex = new InstructionExecuter(state.getStateHistoryBuffer(), this);
	int i = ex.iExecute(decodeBuffer);
	switch (i) {

	  case 0:  // SUCCESS
		// No error, execute was successful
		break;

	  case -2:  // INVALID_OPCODE
		iExceptionCode = EXCEPTION_INVALID_OPCODE;
		break;

	  case -4:  // FP_HARDWARE_REQUIRED
		iExceptionCode = EXCEPTION_FP_REQUIRED;
		break;

	  case -5:  // SYSCALL
		iExceptionCode = EXCEPTION_SYSCALL;
		break;

	  case -6:  // BREAK
		iExceptionCode = EXCEPTION_BREAK;
		break;

	  case -7:  // ARITHMETIC OVERFLOW
		iExceptionCode = EXCEPTION_OVERFLOW;
		break;

	  case -8:  // FP EXCEPTION (divide by zero)
		iExceptionCode = EXCEPTION_FP_ERROR;
		break;

	  default:  // Unexpected return value from iExecute()
		iExceptionCode = EXCEPTION_EXECUTE_ERROR;
		System.err.println("CRITICAL ERROR IN EXECUTE (UNEXPECTED VALUE " + i + ")");
	}

  }  // end method EX()    

// ***********************************************

  private boolean bTimeToMakeCheckpoint() {
	// This method implements the heuristic that determines
	//   when a checkpoint is to be created.
	// If it is determined that a checkpoint should be created,
	//   then this method returns TRUE.  Otherwise, it returns FALSE.

	boolean bResult = false;

	int iCheckpointByteSize = iGetNextCheckpointByteSize();

	if ( (iCheckpointCreationFactor * iCheckpointByteSize) <= iStateHistoryByteSize ) {
	  bResult = true;
	}

	return bResult;

  }  // end method bTimeToMakeCheckpoint()    

  public void makeCheckpoint() {
	// Perform the actual process of creating a checkpoint.

	// Create a checkpoint that records the current state 
	//   of the simulator.
	CheckpointBuffer cpb = pendingCheckpoint;
	pendingCheckpoint = null;
	if (cpb == null) {
	  cpb = doCreateCheckpoint();
	}
	vCheckpoint.add(cpb);

	// Clear the current state history vector.
	clearStateHistoryBuffer();

	// Mark which cycle the checkpoint was created.  This is
	//   used to determine when state history recording
	//   should start again.
	lLastCheckPointCycleIndex = lCycleIndex;

  }  // end method makeCheckpoint()    

private void performCheckpointCulling() {

  // t == lCycleIndex (current cycle index)
  // C == vCheckpoint (current set of checkpoints)

  if (lCycleIndex <= 0)
	return;

  // For all k from 0 to floor( log base 2 of t ) inclusive,
  //  examine all checkpoints within the region:
  //
  //     t - 2^(k+1)  and  t - 2^k
  //
  // Remove all checkpoints within this region, except the
  // one that occurs earliests.

  int iChkIndex = 0; // Index into vCheckpoint vector.

  // Some constant values to experiment with:
  //   LN_1.5 = 0.17609
  //   LN_2 = 0.6931471805599453
  //   LN_3 = 1.0986122886681096
  //   LN_4 = 1.3862943611198906
  // 
  // NOTE:
  //   logn(n,x) = log(x) / log(n), where log(k) == log base e of k

  // Let k == floor( log base 2 of t )
  int K = (int) (Math.log((double) lCycleIndex) / 0.6931471805599453);

  for (int k = K; k > 0; k--) {

	int z = (int) Math.pow(2, k);
	int iUpper = (int) lCycleIndex - z; // Upper = t - 2^k
	int iLower = (int) lCycleIndex - (z << 2) + 1; // Lower = t - 2^(k+1) + 1

	if (iLower < 0)
	  iLower = 0;

	// Since z was already declared, but is no longer used,
	// we re-use it also as a flag to indicate when we first
	// encounter a checkpoint in this region (rather than using
	// one more variable, like a boolean).
	z = -1;

	do {

	  // Get the time index of the checkpoint specified
	  //   by the current checkpoint index.
	  CheckpointBuffer cp = (CheckpointBuffer) vCheckpoint.elementAt(iChkIndex);
	  int iTimeIndex = (int) cp.lGetCycleIndex();

	  if ((iTimeIndex >= iLower) && (iTimeIndex <= iUpper)) {

		if (z == -1) {

		  // This is the first checkpoint encountered in this
		  //   region.  Keep it and increment the checkpoint
		  //   index to the next checkpoint.
		  iChkIndex++;
		  z = 0; // Clear the "first encountered" flag

		}
		else {

		  // We already encountered the first checkpoint in ths
		  //   region, therefore this checkpoint is marked
		  //   to be deleted.  Go ahead and do so.
		  vCheckpoint.removeElementAt(iChkIndex);

		}

	  }
	  else {

		// We have encountered a checkpoint whose time index
		//   is outside this region.  This means there are no
		//   more checkpoints that can be in this region, so
		//   move to the next region.
		break;

	  }

	}
	while (true);

  } // end for k == K down to 1

} // end method performCheckpointCulling()      

  public int iGetNextCheckpointByteSize() {
	// This method returns an approximation to the size of the
	//   next checkpoint (in number of bytes).

	int iResult = 0;

	// 12 * (32+32+32) = 1152
	iResult += 1152;

	/*

	// Add all GPR registers that have a non-zero value.
	iResult += 2;  // == length("R ")
	for (int i = 0; i < RegisterFile.NUM_REGS; i++) {
	  int iRegValue = state.iGetRegister(i);
	  if (iRegValue != 0) {
		iResult += 12;  // == length(i+"="+HexPadded(iRegValue,8)+" ")
	  }
	}

	// Add all CP0 registers that have a non-zero value.
	iResult += 2;  // == length("0 ")
	for (int i = 0; i < Coprocessor0.NUM_REGS; i++) {
	  int iRegValue = state.iGetCP0Register(i);
	  if (iRegValue != 0) {
		iResult += 12;  // == length(i+"="+HexPadded(iRegValue, 8)+" ")
	  }
	}

	// Add all CP1 registers that have a non-zero value.
	iResult += 2;  // == length("1 ");
	for (int i = 0; i < Coprocessor1.NUM_REGS; i++) {
	  int iRegValue = state.iGetCP1Register(i);
	  if (iRegValue != 0) {
		iResult += 12;  // == length(i+"="+HexPadded(iRegValue,8)+" ")
	  }
	}
	*/

	// Add modified memory values.
	//Vector v = state.vGetModifiedAddresses();
	//iResult += 11;  // == length("M AABBCCDD "), address
	//iResult += v.size() * 3;  // 3 == length("XX "), hex values
	iResult += state.iModifiedMemoryCount * 3;

	return iResult;

  }  // end method iGetNextCheckpointSize()        

  private CheckpointBuffer doCreateCheckpoint() {

	CheckpointBuffer cpb = new CheckpointBuffer(lCycleIndex);

	StringBuffer sb = null;

	// Add all GPR registers that have a non-zero value.
	sb = new StringBuffer(CHECKPOINT_GPR + " ");
	for (int i = 0; i < RegisterFile.NUM_REGS; i++) {
	  int iRegValue = state.iGetRegister(i);
	  if (iRegValue != 0) {
		sb.append(i + "=" + Utility.sAsHexPadded(iRegValue, 8) + " ");
	  }
	}
	cpb.addStateValue(sb.toString());

	// Add all CP0 registers that have a non-zero value.
	sb = new StringBuffer(CHECKPOINT_CP0 + " ");
	for (int i = 0; i < Coprocessor0.NUM_REGS; i++) {
	  int iRegValue = state.iGetCP0Register(i);
	  if (iRegValue != 0) {
		sb.append(i + "=" + Utility.sAsHexPadded(iRegValue, 8) + " ");
	  }
	}
	cpb.addStateValue(sb.toString());

	// Add all CP1 registers that have a non-zero value.
	sb = new StringBuffer(CHECKPOINT_CP1 + " ");
	for (int i = 0; i < Coprocessor1.NUM_REGS; i++) {
	  int iRegValue = state.iGetCP1Register(i);
	  if (iRegValue != 0) {
		sb.append(i + "=" + Utility.sAsHexPadded(iRegValue, 8) + " ");
	  }
	}
	cpb.addStateValue(sb.toString());

	// Add modified memory values -- this same technique
	//   is used in CommandGet.java
	Vector v = state.vGetModifiedAddresses();
	if ((v != null) && (v.size() > 0)) {
	  // There is at least one modified address.

	  Enumeration e = v.elements();

	  // Get the first element, so we know which address we
	  //   are starting at.
	  int i = ((Integer)e.nextElement()).intValue();
	  int iAddress = i;

	  sb = new StringBuffer(CHECKPOINT_MEM + " " + Utility.sAsHexPadded(iAddress, 8) + " ");

	  do {

		// Get the memory value at the current address, and add
		//   it to the string buffer.
		byte value = state.loadByte(iAddress);
		sb.append(Utility.sAsHexPadded(value, 2) + " ");

		if (!e.hasMoreElements()) {
		  // No more addresses to process.
		  break;
		}

		// Increment the address counter.  We expect the
		//   next address to match this incremented value.
		iAddress++;

		// Get the next address.
		i = ((Integer)e.nextElement()).intValue();

		if (i != iAddress) {
		  // The new address does not match the address that
		  //   we expected (i.e. we have a disjoint set).
		  //   Add the current string buffer value to the
		  //   command response, and prepare a new string buffer.
		  cpb.addStateValue(sb.toString());

		  iAddress = i;
		  sb = new StringBuffer(CHECKPOINT_MEM + " " + Utility.sAsHexPadded(iAddress, 8) + " ");
		}

	  } while (true);

	  cpb.addStateValue(sb.toString());

	}  // end if-modified memory values

	return cpb;

  }  // end method doCreateCheckpoint()    

  public int iLoadCodeString(String sInput) {
  /*

  RETURNS
	0 = SUCCESS
   -1 = INVALID_LINE_IDENTIFIER_TOKEN
   -2 = CRITICAL_ERROR  (some exception, such as out of memory)


  EXPECTED FORMAT OF sInput STRING
  ================================

	  S filename

	or

	  T addr x1 x2 x3 x4 Lxx

	or

	  D addr x1 x2 ... xN

	If the string starts with T,
	  it represents a text entry. Text entries have the address
	  (in 8-digit hex), followed by four bytes (in 2-digit hex)
	  representing the instruction at that address. The Lxx portion
	  is optional.  If it exists, it is suppose to indicate the
	  line number of where this instruction comes from in
	  the original source file.

	If the string starts with D,
	  it represents a data entry.  Data entries have the address
	  (in 8-digit hex), followed by any number of byte entries
	  (specified in 2-digit hex).  The first byte number starts
	  at the specified address, the second byte number starts at
	  the next address (+1), the third byte at +2, etc.

	If the string starts with S,
	  it can be ignored.  It can be used by a client for debugging
	  purposes.  The filename designates the name of the original
	  source file from where the code came from.  The Lxx token
	  in text entries specify the line number into the file specified
	  by this S entry.

  */ 

	int iResult = 0;

	try {

	  switch (sInput.charAt(0)) {

		case 'S':  // Source File
		  // This is currently ignored.  The client should detect
		  //   and use this information if desired.
		  break;

		case 'T':  // Text Entry
		case 'D':  // Data Entry

		  // Construct a string tokenizer, using everything past the
		  //   line identifier token.
		  StringTokenizer st = new StringTokenizer(sInput.substring(2));

		  String token = st.nextToken(); // token == the 8-digit hex address

		  int iAddress = Utility.iHexStrToInt(token);

		  while (st.hasMoreTokens()) {

			token = st.nextToken();

			if (token.charAt(0) == 'L')
			  break;  // Assume we can finish, since Lxx can be ignored

			// Otherwise, assume the token is a 2-digit hex byte.  We need
			//   to store this byte at the current iAddress;
			byte value = (byte)(Utility.iHexStrToInt(token) & 0x000000FF);
			state.storeByte(iAddress, value);

			// Increment to the next memory address.
			iAddress++;

		  }
		  break;

		default:
		  iResult = -1;

	  }  // end switch

	} catch (Exception err) {
	   
	  iResult = -2;

	}

	return iResult;

  }  // end method iLoadCodeString(String)    

  public int iPerformStep(long lStepDistance) throws SimulatorException {
  /*

  RETURNS
	0 = SUCCESS
   -1 = BREAKPOINT ENCOUNTERED (which one is set in iBreakpointIndex)
   -2 = GUARD ENCOUNTED (which one is set in iGuardIndex)
   -3 = CRITICAL ERROR (out of memory)

  */

	int iResult = 0;

	if ( bHistoryRecordingEnabled && (lCycleIndex == 0) && (vCheckpoint.size() == 0) ) {
	  // This is the first cycle and no checkpoint has been created.
	  //   We assume that a program was loaded already.  So we need
	  //   to mark the current state of the system as being the
	  //   "initial state".
	  makeCheckpoint();
	}

	// Check system flags that were set in the last
	//   step, and clear them if appropriate.
	if (state.getFlag(State.FLAG_SC_EXIT)) {
	  // The EXIT system call happened during the last step.
	  //   We clear the flag now, since the user has decided
	  //   to continue the simulator.
	  state.setFlag(State.FLAG_SC_EXIT, false);
	}

	// Check for any exceptions that happened during
	//   the last step.  Some exceptions can be cleared,
	//   others can not be cleared, or something else has
	//   to happen in order to clear the exception.
	switch (iExceptionCode) {

	  case EXCEPTION_NONE:
		// No exception has occurred.

	  case EXCEPTION_INVALID_OPCODE:
		// Ignore the opcode and just continue with the next one.

	  case EXCEPTION_EXIT:
		// User has decided to continue the program.

		break;

	  default:
		// Some other exception has triggered, and has not been
		//   cleared yet.  So re-throw the SimulatorException.
		throw new SimulatorException("Forced exception handling.");
	}

	// No critical exception is pending.  Reset the exception code.
	iExceptionCode = EXCEPTION_NONE;

	if (iBreakpointIndex != NO_BREAKPOINT) {

	  // In the previous step, a breakpoint was reached.
	  //   The command was not executed, so that the user
	  //   had a chance to respond. We now clear the
	  //   breakpoint index and proceed with executing
	  //   the instruction.
	  iBreakpointIndex = NO_BREAKPOINT;

	} else {

	  // Check if any breapoints have been encounted.

	  int[] iaBreakpoint = breakpointList.iaGetBreakpoints();

	  int iCurrentPC = state.iGetRegister(RegisterFile.R_PC);

	  if (iaBreakpoint != null) {
		// There is at least one breakpoint set.  If the current PC
		//   matches the address of any breakpoint, set
		//   the breakpoint index.
		for (int x = 0; x < iaBreakpoint.length; x++) {
		  if (iaBreakpoint[x] == iCurrentPC) {
			iBreakpointIndex = x;
			break;
		  }
		}
	  }
	}

	if (iGuardIndex != NO_GUARD) {

	  // In the previous step, a guard condition was satisfied.
	  //   The command was not executed, so that the user had
	  //   a chance to respond.  We now clear the guard index
	  //   and proceed with executing the instruction.
	  iGuardIndex = NO_GUARD;

	} else {

	  // Check if any of the guard conditions have been met.
	  Vector vGuardList = guardList.vGetGuards();

	  if (vGuardList != null) {
		// There is at least one guard.  Check each guard
		//   condition, and set the guard index to the first
		//   guard that is satisfied.
		Enumeration e = vGuardList.elements();
		int x = -1;
		while (e.hasMoreElements()) {
		  GuardExpression g = (GuardExpression)e.nextElement();
		  x++;
		  if (GuardExpression.isSatisfied(g)) {
			iGuardIndex = x;
			break;
		  }
		}
	  }
	}

	// If a breakpoint was encounted, let the user respond first
	//   before proceeding with this step.
	if (iBreakpointIndex != NO_BREAKPOINT) {
	  return -1;
	}

	// If a guard was encounted, let the user respond first
	//   before proceeding with this step.
	if (iGuardIndex != NO_GUARD) {
	  return -2;
	}

	// Ready to perform lStepDistance cycles...
	do {

	  try {

		performCycle();

		if (iExceptionCode != EXCEPTION_NONE)
		  throw new SimulatorException("Forced exception handling.");

	  } catch (SimulatorException e) {

		int iOldExceptionCode = iExceptionCode;

		// ****************************************************
		// Check for SYSCALL related exceptions that are to
		//   be processed internally.
		iExceptionCode = EXCEPTION_OUTPUT_WAITING;
		if (state.getFlag(State.FLAG_SC_PRINT_INT))
		  throw new SimulatorException("Waiting to output integer.");
		if (state.getFlag(State.FLAG_SC_PRINT_FLOAT))
		  throw new SimulatorException("Waiting to output float.");
		if (state.getFlag(State.FLAG_SC_PRINT_DOUBLE))
		  throw new SimulatorException("Waiting to output double.");
		if (state.getFlag(State.FLAG_SC_PRINT_STRING))
		  throw new SimulatorException("Waiting to output string.");
		iExceptionCode = EXCEPTION_INPUT_WAITING;
		if (state.getFlag(State.FLAG_SC_READ_INT))
		  throw new SimulatorException("Waiting to read integer.");
		if (state.getFlag(State.FLAG_SC_READ_FLOAT))
		  throw new SimulatorException("Waiting to read float.");
		if (state.getFlag(State.FLAG_SC_READ_DOUBLE))
		  throw new SimulatorException("Waiting to read double.");
		if (state.getFlag(State.FLAG_SC_READ_STRING))
		  throw new SimulatorException("Waiting to read string.");
		iExceptionCode = EXCEPTION_EXIT;
		if (state.getFlag(State.FLAG_SC_EXIT))
		  throw new SimulatorException("Exit system call.");
		// ****************************************************

		iExceptionCode = iOldExceptionCode;

		// The user should use iGetExceptionCode() to examine
		//   what exception happened, and respond appropiately.
		//   Some exceptions can not be cleared, such as
		//   FETCH_ERROR.

		throw new SimulatorException(e.toString());

	  } catch (Exception e) {

		iResult = -3;  // CRITICAL_ERROR

	  }

	  lStepDistance--;

	} while ( (iResult == 0) && (lStepDistance > 0) );

	return iResult;

  }  // end method iPerformStep(long)    

  public int iPerformUndo(long lUndoDistance) {
  /*

	Perform an undo by getting the last state change
	  history buffer from the state history vector.

	For reference, see HISTORY.TXT or refer to the
	  addStateValue(String) method in State.java

	RETURNS
	  0 = SUCCESS
	 -1 = NO MORE STATE HISTORY CHANGE
	 -2 = ERROR APPLYING STATE HISTORY CHANGE
	 -3 = ERROR APPLYING CHECKPOINT STATE
	 -4 = ERROR RE-EXECUTING FROM LAST CHECKPOINT
	 -5 = EXCEPTION ENCOUNTED WHILE UNDO FROM CHECKPOINT
	 -6 = OUTPUT ERROR WHILE RE-EXECUTING INSTRUCTIONS

  */

	int iResult = 0;

	if (!bHistoryRecordingEnabled) {
	  return -1;
	}

	int iStateHistorySize = vStateHistory.size();
	int iCheckpointSize = vCheckpoint.size();

	sbOutputFromUndo = null;  // Clear any undo output from any previous undo.

	if ( (iStateHistorySize == 0) && (iCheckpointSize == 0) ) {
	  // There is no history information.  Either we are in
	  //   the initial state, or all the history information
	  //   was cleared.  Eitherway, we have no information
	  //   to perform the undo with.
	  return -1;
	}

	long lTargetCycle = lCycleIndex - lUndoDistance;
	// The target cycle is what execution cycle we are trying
	//   to get to, which is simply the current cycle
	//   minus the number of cycles we want to undo.

	if (lTargetCycle < 0) {
	  // We can not go to a state that is earlier than
	  //   the initial state.
	  lTargetCycle = 0;
	}

	if ( (iStateHistorySize == 0) || (lUndoDistance > iStateHistorySize) ) {
	  // There is no state history information.  Or the distance
	  //   of the undo is larger than the size of the state history
	  //   buffer.  Regardless, perform the undo using the last checkpoint.

	  if (lUndoDistance > iStateHistorySize) {
		// Clear the state history, since all of the state changes
		//   recorded in it will not be used.  The last
		//   checkpoint will be used instead. 
		clearStateHistoryBuffer();
	  }

	  if (iCheckpointSize > 0) {
		// Set the state to that of the last checkpoint
		//   that was created.
	  
		CheckpointBuffer cpb = null;
		long lCheckpointCycleIndex = -1;
		// Find a checkpoint buffer to start from.
		try {
		  do {
			cpb = (CheckpointBuffer)vCheckpoint.elementAt(iCheckpointSize-1);
			lCheckpointCycleIndex = cpb.lGetCycleIndex();
			if (lCheckpointCycleIndex > lTargetCycle) {
			  // The checkpoint represents a state at a point in time
			  //   AFTER the desired target cycle.  It would be better
			  //   to examine the next checkpoint.

			  vCheckpoint.removeElementAt(iCheckpointSize-1);
			  iCheckpointSize--;
			  if (iCheckpointSize <= 0) {
				// This is a safety check.  If no more checkpoints
				//   exists, then go back to the initial state.
				state.reset();
				cpb = null;
				break;
			  }
			} else {
			  break;
			}
		  } while (true);

		  if (lCheckpointCycleIndex == lTargetCycle) {
			// The previous cycle (iTargetCycle) was when the
			//   checkpoint was created.  We can remove the checkpoint,
			//   since if the user goes back further, the top checkpoint
			//   is no longer useful.  If the user steps forward,
			//   the checkpoint will be recreated.
			vCheckpoint.removeElementAt(iCheckpointSize-1);
			iCheckpointSize--;
			// ** THIS IF-CHECK IS OPTIONAL:  This checkpoint
			//   would be removed anyway during the next call
			//   to this method.  So it's either do it now or
			//   have it be done later.  By doing this now,
			//   we can conserve some memory.
		  }

		} catch (Exception e) {
		  cpb = null;
		}

		// We found no suitable checkpoint buffer.  Either an error occurred,
		//   or there is no more suitable undo information available.
		if (cpb == null) {
		  return -1;  // NO MORE STATE HISTORY CHANGE
		}

		// Set the state back to the initial state.  All memory and
		//   register values should be defaulted to have the value 0.
		state.reset();

		lCycleIndex = lCheckpointCycleIndex;
		// Set the current cycle count to the time index stored
		//   in the checkpoint buffer.

		// Apply all of the state settings stored in the checkpoint.
		Vector v = cpb.vGetStateRecord();
		Enumeration e = v.elements();
		while (e.hasMoreElements()) {
		  String s = (String)e.nextElement();
		  int x = iApplyCheckpointSetting(s);
		  if (x != 0) {
			// This is most likely the result of an 
			//   internal bug in the simulator.
			System.err.println("APPLY CHECKPOINT ERROR: " + x);
			return -3;
		  }
		}

	  } else {
		// Set the state to the reset state.
		state.reset();
	  }

	  // Disable any exceptions that triggered from the previously
	  //   executed instruction.
	  iExceptionCode = EXCEPTION_NONE;

	  // *** RE-EXECUTION PHASE **************************************
	  // Perform cycles until the cycle count reaches lTargetCycle,
	  //   or an exception triggers.
	  while (lCycleIndex < lTargetCycle) {

		try {
		  int i = iPerformStep(lTargetCycle - lCycleIndex);
		  switch (i) {
			case 0:
			  // No Error
			  break;

			case -1:  // = BREAKPOINT ENCOUNTERED (which one is set in iBreakpointIndex)
			  // Ignore breakpoint?
			  break;

			case -2:  // = GUARD ENCOUNTED (which one is set in iGuardIndex)
			  // Ignore guard?
			  break;

			default:  // = CRITICAL ERROR (out of memory?)
			  throw new Exception();
		  }
		} catch (SimulatorException e) {

		  // SimulatorException triggered (invalid opcode, etc).
		  //   Refer to EXCEPTION_CODE value.

		  if (iExceptionCode == EXCEPTION_OUTPUT_WAITING) {
			StringBuffer sbOutput = new StringBuffer();
			if (iReadOutput(sbOutput) == 0) {
			  if (sbOutputFromUndo == null) {
				sbOutputFromUndo = new StringBuffer();
			  }
			  sbOutputFromUndo.append(sbOutput);
			} else {
			  return -6;
			}
		  } else {
			return -5;
		  }
  
		} catch (Exception e) {

		  // CRITICAL_ERROR (index out of range, null pointer --
		  //   indicates either a bug in the simulator, or
		  //   perhaps an out of memory condition).
		  return -4;

		}
	  }  // end while (performing cycles until reach iTargetCycle)
	  // *************************************************************

	} else {
	  // There is undo information in the state history buffer.
	  //   These are performed first, before checkpoints.

	  do {

		StateHistoryBuffer shb = null;
		// Get the last state history buffer instance.
		try {
		  shb = (StateHistoryBuffer)vStateHistory.elementAt(iStateHistorySize-1);
		  vStateHistory.removeElementAt(iStateHistorySize-1);
		  iStateHistorySize--;
		} catch (Exception e) {
		  shb = null;
		}
		if (shb == null) {
		  // Either an error occurred, or there was no state history
		  //   buffer information (meaning there is no undo information).
		  return -1;  // NO MORE STATE HISTORY CHANGE
		}

		// Apply the state values stored in the change buffer.
		//   These must be applied in reverse, so that the
		//   state is returned to its earliest state per this
		//   change buffer.
		Vector v = shb.vGetStateHistory();
		for (int i = v.size()-1; i >= 0; i--) {
		  String s = (String)v.elementAt(i);
		  int x = iApplyStateSetting(s);
		  if (x != 0) {
			// This is most likely the result of an 
			//   internal bug in the simulator.
			System.err.println("APPLY STATE ERROR: " + x);
			return -2;
		  }
		}
		// It is assumed that each state history element represents
		//   one cycle of execution (because at least $PC will change
		//   on every cycle).  Therefore, undoing this state history
		//   change decreases the cycle count by one.  But to be certain,
		//   use the cycle index stored in the SHB instead.
		lCycleIndex = shb.lGetCycleIndex();

		// Decrease the size of the state history by the size of the
		//   state history buffer that was just applied.  This is for
		//   performance reasons, so that the size of the history buffer
		//   does not need to be directly re-calculated each cycle.
		iStateHistoryByteSize -= shb.iGetSize();

	  } while (lCycleIndex > lTargetCycle);

	  // Disable any exception that was caused by the previous command.
	  //   If we don't, then the instruction will be undone, but a pending
	  //   exception will be waiting (e.g. for input) when it shouldn't be.
	  iExceptionCode = EXCEPTION_NONE;

	}  // end state-history undo

	return iResult;

  }  // end method iPerformUndo()      

  private int iApplyCheckpointSetting(String sInput) {
  /*

	GPR x1=value1 ... xN=valueN
	CP0 x1=value1 ... xN=valueN
	CP1 x1=value1 ... xN=valueN
	MEM addr value1 ... valueN

  RETURNS:
	0 = SUCCESS
   -1 = ERROR BAD_TOKEN
   -2 = CRITICAL ERROR

  */
	int iResult = 0;

	try {

	  StringTokenizer st = new StringTokenizer(sInput);

	  String s = st.nextToken();

	  if (s.startsWith(CHECKPOINT_GPR)) {
		while (st.hasMoreTokens()) {
		  s = st.nextToken();
		  int i = s.indexOf('=');
		  int iReg = Integer.parseInt(s.substring(0,i));
		  int iValue = (int)Utility.IntegerLiteralToLong("0x"+s.substring(i+1));
		  state.setRegister(iReg, iValue);
		}
	  }
	  else if (s.startsWith(CHECKPOINT_CP0)) {
		while (st.hasMoreTokens()) {
		  s = st.nextToken();
		  int i = s.indexOf('=');
		  int iReg = Integer.parseInt(s.substring(0,i));
		  int iValue = (int)Utility.IntegerLiteralToLong("0x"+s.substring(i+1));
		  state.setCP0Register(iReg, iValue);
		}
	  }
	  else if (s.startsWith(CHECKPOINT_CP1)) {
		while (st.hasMoreTokens()) {
		  s = st.nextToken();
		  int i = s.indexOf('=');
		  int iReg = Integer.parseInt(s.substring(0,i));
		  int iValue = (int)Utility.IntegerLiteralToLong("0x"+s.substring(i+1));
		  state.setCP1Register(iReg, iValue);
		}
	  }
	  else if (s.startsWith(CHECKPOINT_MEM)) {
		s = st.nextToken();  // s == should be starting address
		int iAddress = (int)Utility.IntegerLiteralToLong("0x"+s);
		while (st.hasMoreTokens()) {
		  s = st.nextToken();  // s == should be 2-digit byte value
		  int iValue = Integer.parseInt(s, 16);
		  state.storeByte(iAddress, (byte)iValue);
		  iAddress++;  // Increment address by SIZE_BYTE
		}

	  } else {
		iResult = -1;
	  }

	} catch (Exception e) {
	  iResult = -2;
	}

	return iResult;
  }  // end method iApplyCheckpointSetting(String)    

  private int iApplyStateSetting(String s) {
  /*

	See HISTORY.TXT for details on interpreting the
	  input string s.

	MEM address value
	GPR r value
	CP0 r value
	CP1 r value
	CYCLE x
	FLAG x value

  RETURNS:
	0 = SUCCESS
   -1 = BAD_TOKEN
   -2 = CRITICAL_ERROR

  */

	int iResult = 0;

	try {

	  StringTokenizer st = new StringTokenizer(s);
	  String t = st.nextToken();

	  if (t.startsWith(State.STATE_CHANGE_MEM)) {

		// MEM addr value

		t = st.nextToken();  // address (8-digit hex)
		int iAddress = Utility.iHexStrToInt(t);

		t = st.nextToken();  // s == should be 2-digit byte value
		int iValue = Integer.parseInt(t, 16);
		state.storeByte(iAddress, (byte)iValue);

	  } else
	  if (t.startsWith(State.STATE_CHANGE_GPR)) {

		// GPR r value

		t = st.nextToken();  // register integer index
		byte n = Byte.parseByte(t);

		t = st.nextToken();  // register value (8-digit hex)
		int value = Utility.iHexStrToInt(t);
		state.setRegister(n, value);

	  } else
	  if (t.startsWith(State.STATE_CHANGE_CP0)) {

		// CP0 r value

		t = st.nextToken();  // register integer index
		byte n = Byte.parseByte(t);

		t = st.nextToken();  // register value (8-digit hex)
		int value = Utility.iHexStrToInt(t);
		state.setCP0Register(n, value);

	  } else
	  if (t.startsWith(State.STATE_CHANGE_CP1)) {

		// CP1 r value

		t = st.nextToken();  // register integer index
		byte n = Byte.parseByte(t);

		t = st.nextToken();  // register value (8-digit hex)
		int value = Utility.iHexStrToInt(t);
		state.setCP1Register(n, value);

	  } else
	  if (t.startsWith(State.STATE_CHANGE_FLAG)) {

		// FLAG x value

		t = st.nextToken();  // flag index number
		int flagID = Integer.parseInt(t);

		t = st.nextToken();  // flag value ("T" or "F")
		boolean b = (t.equals("T"));
		state.setFlag(flagID, b);

	  } else {

		iResult = -1;  // BAD_TOKEN

	  }

	} catch (Exception e) {

	  System.err.println("APPLY STATE ERROR: " + e);
	  iResult = -2;  // CRITICAL_ERROR (out of memory, index out of bounds?)

	}

	return iResult;
	
  }  // end method iApplyStateSetting(String)    

  // ** GET METHODS ******************************************

  public int iGetNumCheckpoints() {
	if (vCheckpoint == null)
	  return -1;
	return vCheckpoint.size();
  }    

  public int iGetNumStateChanges() {
	if (vStateHistory == null)
	  return -1;
	return vStateHistory.size();
  }    

  public int iGetStateHistoryByteSize() {
	return iStateHistoryByteSize;
  }    

  public BreakpointList getBreakpointList() {
	if (breakpointList == null)
	  breakpointList = new BreakpointList();
	return breakpointList;
  }    

  public GuardList getGuardList() {
	if (guardList == null)
	  guardList = new GuardList();
	return guardList;
  }    

  public Vector vGetStateHistory() {
	return vStateHistory;
  }    

  public Vector vGetCheckpoint() {
	return vCheckpoint;
  }    

  public State getState() {
	return state;
  }    

  public int iGetBreakpointIndex() {
	return iBreakpointIndex;
  }    

  public int iGetGuardIndex() {
	return iGuardIndex;
  }    

  public long lGetCycleIndex() {
	return lCycleIndex;
  }    

  public int iGetExceptionCode() {
	return iExceptionCode;
  }    

  public void writeHistoryLog(String sFilename) {
  // This method outputs the contents of the history buffer and
  //   checkpoints to the specified filename.  This allows a way
  //   to write the history buffer without any memory overhead
  //   (such as when the sbGetHistoryLog() method is used)

	PrintWriter out = null;
	try {
	  out = new PrintWriter( new BufferedWriter(new FileWriter(sFilename)) );
	} catch (Exception e) {
	  System.err.println("Could not create history log: " + e);
	  return;
	}

	try {

	  for (int i = 0; i < vCheckpoint.size(); i++) {
		CheckpointBuffer cpb = (CheckpointBuffer)vCheckpoint.elementAt(i);
		out.write("CP@" + cpb.lGetCycleIndex() + "\n");
		Enumeration e = cpb.vGetStateRecord().elements();
		while (e.hasMoreElements()) {
		  out.write((String)e.nextElement() + "\n");
		}
	  }

	  for (int i = 0; i < vStateHistory.size(); i++) {
		StateHistoryBuffer shb = (StateHistoryBuffer)vStateHistory.elementAt(i);
		out.write("C " + shb.lGetCycleIndex() + "\n");
		Enumeration e = shb.vGetStateHistory().elements();
		while (e.hasMoreElements()) {
		  out.write((String)e.nextElement() + "\n");
		}
	  }

	} catch (Exception e) {
	  System.err.println("Error writing history log: " + e);
	}

	try {
	  out.close();
	} catch (Exception e) {
	  System.err.println("Error closing history log: " + e);
	}

	// ************************************************************
	// This following can be used to stream the checkpoint and
	//   history buffer objects to a file.
	/*
	NOTE: CheckpointBuffer and StateHistoryBuffer need to implement
	  class Serializable in order for this to work.
	try {
	  FileOutputStream fos = new FileOutputStream(sFilename);
	  ObjectOutput out = new ObjectOutputStream(fos);
	  out.writeObject(vCheckpoint);
	  out.writeObject(vStateHistory);
	  out.flush();
	  out.close();
	} catch (Exception e) {
	  System.err.println("Error writing history log: " + e);
	}
	*/
  }    

  public StringBuffer sbGetHistoryLog() {
  // This methods returns the current checkpoint and
  //   history buffer information in a StringBuffer.
  //   Keep in mind that this can easily cause an out of
  //   memory exception if the history information is
  //   very large.

	StringBuffer sbResult = new StringBuffer();

	for (int i = 0; i < vCheckpoint.size(); i++) {
	  CheckpointBuffer cpb = (CheckpointBuffer)vCheckpoint.elementAt(i);
	  sbResult.append("CP@" + cpb.lGetCycleIndex() + "\n");
	  Enumeration e = cpb.vGetStateRecord().elements();
	  while (e.hasMoreElements()) {
		sbResult.append((String)e.nextElement() + "\n");
	  }
	}

	for (int i = 0; i < vStateHistory.size(); i++) {
	  StateHistoryBuffer shb = (StateHistoryBuffer)vStateHistory.elementAt(i);
	  sbResult.append("C " + shb.lGetCycleIndex() + "\n");
	  Enumeration e = shb.vGetStateHistory().elements();
	  while (e.hasMoreElements()) {
		sbResult.append((String)e.nextElement() + "\n");
	  }
	}

	return sbResult;

  }  // end method sbGetHistoryLog()    

  public StringBuffer sbGetOutputFromUndo() {
  // Returns the output from the previous undo (which is caused
  //   in the case of when the undo had to re-execute cycles).
  //   This method also clears the output buffer.  A value of
  //   null is returned if there is no output information from
  //   the previous undo operation.
	StringBuffer sbResult = sbOutputFromUndo;
	sbOutputFromUndo = null;
	return sbResult;
  }  // end method sbGetOutputFromUndo()    

  // ** END OF GET METHODS ***********************************

  private void addToStateHistoryBuffer(StateHistoryBuffer shb) {
	vStateHistory.add(shb);
	iStateHistoryByteSize += shb.iGetSize();
  }    

  private void clearStateHistoryBuffer() {
	vStateHistory.clear();
	iStateHistoryByteSize = 0;
  }    

  public boolean bHasFPA() {
	int i = state.getCoprocessor0().iGetReg(Coprocessor0.R_SR);
	return Utility.bBitSet(i, 29, true);
  }  // end method boolean bHasFPA()    

  private void incrementCycleIndex() {

	long lNewCycleIndex = lCycleIndex;
	lNewCycleIndex++;
	if (lNewCycleIndex == MAX_CYCLE_INDEX) {
	  // We have reached the maximum cycle index.  This means
	  //   the cycle index counter must be reset, which in turn
	  //   means all history information must be erased.
	  lNewCycleIndex = 0;
	  vCheckpoint.clear();
	  clearStateHistoryBuffer();
	  lLastCheckPointCycleIndex = -1;
	}
	lCycleIndex = lNewCycleIndex;

  }  // end method incrementCycleIndex()    

  private void decrementCycleIndex() {

	long lNewCycleIndex = lCycleIndex;
	lNewCycleIndex--;
	if (lNewCycleIndex < 0) {
	  // We can decrement to a cycle count that is less than zero.
	  lNewCycleIndex = 0;
	}
	lCycleIndex = lNewCycleIndex;

  }  // end method decrementCycleIndex()    

  public int iWriteInput(String sInput) {
  /*

  This method stores the given input string (sInput) into the
	appropriate memory or register location.  The action taken
	depends on which input flag is set.  For instance, if an
	integer input is detected, then the sInput string is
	first convereted into an integer (then stored in a register).

  If the string input buffer is filled, the remaining input is
	discarded.  The input string may contain C-style \-codes if
	the expected input is a string.

  RETURNS
	0 = SUCCESS
   -1 = ERR_NO_INPUT_EXPECTED (input discarded)
   -2 = ERR_INPUT_FORMAT (e.g. specifying a string when integer is expected)

  */

	int iResult = 0;

	try {

	  if (state.getFlag(State.FLAG_SC_READ_INT)) {

		// The input is expected to be an integer value.
		int x = (int)Utility.IntegerLiteralToLong(sInput);

		// Store the integer value into REG[$v0]
		state.setRegister(RegisterFile.R_V0, x);

		// Turn off the READ_INT flag.
		state.setFlag(State.FLAG_SC_READ_INT, false);
 
	  } else if (state.getFlag(State.FLAG_SC_READ_FLOAT)) {

		// The input is expected to be a float value.
		float f = Float.parseFloat(sInput);

		// Store the float value into CP1REG[$f0].
		state.setCP1Register(Coprocessor1.R_F0, Float.floatToIntBits(f));

		// Turn off the READ_FLOAT flag.
		state.setFlag(State.FLAG_SC_READ_FLOAT, false);
 
	  } else if (state.getFlag(State.FLAG_SC_READ_DOUBLE)) {

		// The input is expected to be a double value.
		double d = Double.parseDouble(sInput);

		// Store the double value into CP1REG[$f0]
		// ($f0 gets HI bits, $f1 gets LO bits)
		long l = Double.doubleToLongBits(d);
		int x1 = (int)( (l & 0xFFFFFFFF00000000L) >>> 32 );
		int x2 = (int)( (l & 0x00000000FFFFFFFFL)        );
		state.setCP1Register(Coprocessor1.R_F0, x2);
		state.setCP1Register(Coprocessor1.R_F1, x1);

		// Turn off the READ_DOUBLE flag.
		state.setFlag(State.FLAG_SC_READ_DOUBLE, false);

	  } else if (state.getFlag(State.FLAG_SC_READ_STRING)) {

		// The input is expected to be a C-style string.
		/*
		  \n == new line
		  \t == tab
		  \r == carrier return
		  \\ == backslash
		  \" == double quote
		  \0 == null terminator

		  The input string is to be stored in a buffer,
			reserved starting at address REG[$a0] of
			length REG[$a1].
		*/

		if (sInput.equals("")) {

		  // No input was actually specified.  Maybe this was
		  //   intended, so it is not an error.

		} else {

		  // Get the parameters of the input buffer (size and location).
		  //   $a1 = buffer length, $a0 = buffer starting address
		  int iBufferLength = state.iGetRegister(RegisterFile.R_A1);
		  int iBufferStartingAddress = state.iGetRegister(RegisterFile.R_A0);

		  int iCurrentAddress = iBufferStartingAddress;

		  int iCharIndex = 0;
		  int iLastSlash = -1;  // Index position of the last \-token occurrence.
		  do {

			// Get the character at the current character index.
			char c = sInput.charAt(iCharIndex);

			if (iLastSlash != -1) {
			  // In the last interation of this loop, we encountered
			  //   a backslash token.  This means now the current token
			  //   is used to indicate some special character.
			  switch (c) {

				case '\\':
				  state.storeByte(iCurrentAddress, (byte)'\\');
				  break;

				case 'n':
				  state.storeByte(iCurrentAddress, (byte)'\n');
				  break;

				case 't':
				  state.storeByte(iCurrentAddress, (byte)'\t');
				  break;

				case 'r':
				  state.storeByte(iCurrentAddress, (byte)'\r');
				  break;

				case '0':
				  state.storeByte(iCurrentAddress, (byte)'\0');
				  break;

				case '"':
				  state.storeByte(iCurrentAddress, (byte)'\"');
				  break;
  
				default:
				  // An invalid \x character was specified.  We'll
				  //   just ignore this.

			  }

			  // We are done processing the current \-token, so
			  //   reset iLastSlash index.
			  iLastSlash = -1;

			} else 
			if (c == '\\') {
			  // Mark the index of where we encounted the
			  //   \ special-character identifier. We'll
			  //   process this character on the next iteration.
			  iLastSlash = iCharIndex;
			}
			else {
			  // Store the input character at the current memory address.
			  state.storeByte(iCurrentAddress, (byte)c);
			}

			if (iCharIndex >= sInput.length()) {
			  // We have reached the end of the input string.
			  break;
			}
			if (iCharIndex >= iBufferLength) {
			  // The input buffer has become full.  The rest of the
			  //   input is ignored.
			  break;
			}
			iCharIndex++;

			// Increment the address counter, so that the next input
			//   is stored at the next address.
			iCurrentAddress++;

		  } while (true);

		  // Turn off the READ_STRING flag.
		  state.setFlag(State.FLAG_SC_READ_STRING, false);

		}

	  } else {
		iResult = -1; // ERR_NO_INPUT_EXPECTED
	  }

	} catch (Exception e) {
	  System.err.println("INPUT ERROR: " + e);
	  iResult = -2;  // ERR_INPUT_FORMAT
	}

	// Clear the INPUT_WAITING exception, since input has been
	//   sent into the system.
	if ( (iResult == 0) && (iExceptionCode == EXCEPTION_INPUT_WAITING) )
	  iExceptionCode = EXCEPTION_NONE;

	return iResult;

  }  // end method iWriteInput(String)    

  public int iReadOutput(StringBuffer sBuffer) {
  /*

  If an output flag has been set, this method returns the
	output in a StringBuffer.  The parameter sBuffer must
	already have been initialized by the calling method.

  RETURNS
	0 = SUCESS
   -1 = ERR_NO_OUTPUT_EXPECTED
   -2 = ERR_OUTPUT_FORMAT (out of memory?)
   -3 = ERR_NO_STRING_BUFFER: String buffer parameter not initialized.
  */

	int iResult = 0;

	if (sBuffer == null)
	  return -3;
				
	try {

	  if (state.getFlag(State.FLAG_SC_PRINT_INT)) {

		// We are ready to output an integer.  The
		//   integer value to be printed is stored in REG[$a0].
		int x = state.iGetRegister(RegisterFile.R_A0);

		// Prepare to return the integer as a string.
		sBuffer.append("" + x);

		// Turn off the PRINT_INT flag.
		state.setFlag(State.FLAG_SC_PRINT_INT, false);

	  } else if (state.getFlag(State.FLAG_SC_PRINT_FLOAT)) {

		// We are ready to output a float.  The
		//   float value to be printed is stored in REG[$f12].
		int x = state.iGetCP1Register(Coprocessor1.R_F12);
		float f = Float.intBitsToFloat(x);

		// Prepare to return the float as a string.
		sBuffer.append("" + f);

		// Turn off the PRINT_FLOAT flag.
		state.setFlag(State.FLAG_SC_PRINT_FLOAT, false);

	  } else if (state.getFlag(State.FLAG_SC_PRINT_DOUBLE)) {

		// We are ready to output a double.  The
		//   double value to be printed is stored in REG[$f12].
		int i1 = state.iGetCP1Register(Coprocessor1.R_F12);
		int i2 = state.iGetCP1Register(Coprocessor1.R_F13);
		long l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		double d = Double.longBitsToDouble(l1);

		// Prepare to return the double as a string.
		sBuffer.append("" + d);

		// Turn off the PRINT_DOUBLE flag.
		state.setFlag(State.FLAG_SC_PRINT_DOUBLE, false);

	  } else if (state.getFlag(State.FLAG_SC_PRINT_STRING)) {

		// We are ready to output a string. The address of the
		//   string to be printed starts at REG[$a0].  The
		//   end of the string is indicated by a null-terminater.
		int iStartAddress = state.iGetRegister(RegisterFile.R_A0);
		int iCurrentAddress = iStartAddress;
		do {

		  char c = (char)state.loadByte(iCurrentAddress);

		  if (c == '\0') {
			// We have reached the null terminater, so we are done
			//   reading the string.
			break;
		  }

		  // Append the character to the string result buffer.
		  sBuffer.append(c);

		  // Increment to the next address, to get the next character
		  //   of the string.
		  iCurrentAddress++;

		} while (true);

		// Turn off the PRINT_STRING flag.
		state.setFlag(State.FLAG_SC_PRINT_STRING, false);

	  } else {
		iResult = -1; // ERR_NO_OUTPUT_EXPECTED
	  }

	} catch (Exception e) {
	  System.err.println("OUTPUT ERROR: " + e);
	  iResult = -2;  // ERR_OUTPUT_FORMAT
	}

	// Clear the OUTPUT_WAITING exception, since output has been
	//   sent to the output buffer.
	if ( (iResult == 0) && (iExceptionCode == EXCEPTION_OUTPUT_WAITING) )
	  iExceptionCode = EXCEPTION_NONE;

	return iResult;

  }  // end method sReadOutput()    

}
