package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    InstructionExecuter.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/

public class InstructionExecuter {

  // *** SEARCH FOR "**INCOMPLETE" TO FIND INCOMPLETED INSTRUCTIONS

  // system call codes
  public static final int SC_PRINT_INT    = 1;
  public static final int SC_PRINT_FLOAT  = 2;
  public static final int SC_PRINT_DOUBLE = 3;
  public static final int SC_PRINT_STRING = 4;
  public static final int SC_READ_INT     = 5;
  public static final int SC_READ_FLOAT   = 6;
  public static final int SC_READ_DOUBLE  = 7;
  public static final int SC_READ_STRING  = 8;
  public static final int SC_SBRK         = 9;
  public static final int SC_EXIT         = 10;

  // Primary 6-bit opcode constants
  public static final int OP_MULTI_1   = 0;
  public static final int OP_MULTI_2   = 1;
  public static final int OP_J         = 2;
  public static final int OP_JAL       = 3;
  public static final int OP_BEQ       = 4;
  public static final int OP_BNE       = 5;
  public static final int OP_BLEZ      = 6;
  public static final int OP_BGTZ      = 7;
  public static final int OP_ADDI      = 8;
  public static final int OP_ADDIU     = 9;
  public static final int OP_SLTI      = 10;
  public static final int OP_SLTIU     = 11;
  public static final int OP_ANDI      = 12;
  public static final int OP_ORI       = 13;
  public static final int OP_XORI      = 14;
  public static final int OP_LUI       = 15;
  public static final int OP_MULTI_3   = 16;  // z == 0
  public static final int OP_MULTI_4   = 17;  // z == 1
  public static final int OP_MULTI_5   = 18;  // z == 2
  public static final int OP_MULTI_6   = 19;  // z == 3
  public static final int OP_LB        = 32;
  public static final int OP_LH        = 33;
  public static final int OP_LWL       = 34;
  public static final int OP_LW        = 35;
  public static final int OP_LBU       = 36;
  public static final int OP_LHU       = 37;
  public static final int OP_LWR       = 38;
  public static final int OP_SB        = 40;
  public static final int OP_SH        = 41;
  public static final int OP_SWL       = 42;
  public static final int OP_SW        = 43;
  public static final int OP_SWR       = 46;
  public static final int OP_LWC0      = 48;
  public static final int OP_LWC1      = 49;
  public static final int OP_LWC2      = 50;
  public static final int OP_LWC3      = 51;
  public static final int OP_SWC0      = 56;  // 38
  public static final int OP_SWC1      = 57;  // 39
  public static final int OP_SWC2      = 58;  // 3A
  public static final int OP_SWC3      = 59;  // 3B

  // 6-bit Function constants (used when opcode == OP_MULTI_1)
  public static final int OP_SLL       = 0;
  public static final int OP_SRL       = 2;
  public static final int OP_SRA       = 3;
  public static final int OP_SLLV      = 4;
  public static final int OP_SRLV      = 6;
  public static final int OP_SRAV      = 7;
  public static final int OP_JR        = 8;
  public static final int OP_JALR      = 9;
  public static final int OP_SYSCALL   = 12;
  public static final int OP_BREAK     = 13;
  public static final int OP_MFHI      = 16;
  public static final int OP_MTHI      = 17;
  public static final int OP_MFLO      = 18;
  public static final int OP_MTLO      = 19;
  public static final int OP_MULT      = 24;
  public static final int OP_MULTU     = 25;
  public static final int OP_DIV       = 26;
  public static final int OP_DIVU      = 27;
  public static final int OP_ADD       = 32;
  public static final int OP_ADDU      = 33;
  public static final int OP_SUB       = 34;
  public static final int OP_SUBU      = 35;
  public static final int OP_AND       = 36;
  public static final int OP_OR        = 37;
  public static final int OP_XOR       = 38;
  public static final int OP_NOR       = 39;
  public static final int OP_SLT       = 42;
  public static final int OP_SLTU      = 43;

  // 5-bit Code constants (used when opcode == OP_MULTI_2)
  public static final int OP_BLTZ      = 0;
  public static final int OP_BGEZ      = 1;
  public static final int OP_BLTZAL    = 16;
  public static final int OP_BGEZAL    = 17;

  // These match db.rs (25:21)
  public static final int OP_MFCZ      = 0;
  public static final int OP_CFCZ      = 2;
  public static final int OP_MTCZ      = 4;
  public static final int OP_CTCZ      = 6;
  public static final int OP_MULTI_7   = 8;
  public static final int OP_COPZ_1    = 16;
  public static final int OP_COPZ_2    = 17;

  // These match db.funct (4:0)
  public static final int OP_TLBR      = 1;
  public static final int OP_TLBWL     = 2;
  public static final int OP_TLBWR     = 6;
  public static final int OP_TLBP      = 8;
  public static final int OP_RFE       = 16;
  
  // These match db.funct (5:0)
  public static final int OP_ADD_F     = 0;
  public static final int OP_SUB_F     = 1;
  public static final int OP_MUL_F     = 2;
  public static final int OP_DIV_F     = 3;
  public static final int OP_ABS_F     = 5;
  public static final int OP_MOV_F     = 6;
  public static final int OP_NEG_F     = 7;
  public static final int OP_CVT_S_F   = 32;
  public static final int OP_CVT_D_F   = 33;
  public static final int OP_CVT_W_F   = 36;
  public static final int OP_C_F_F     = 48;
  public static final int OP_C_UN_F    = 49;
  public static final int OP_C_EQ_F    = 50;
  public static final int OP_C_UEQ_F   = 51;
  public static final int OP_C_OLT_F   = 52;
  public static final int OP_C_ULT_F   = 53;
  public static final int OP_C_OLE_F   = 54;
  public static final int OP_C_ULE_F   = 55;
  public static final int OP_C_ST_F    = 56;
  public static final int OP_C_NGLE_F  = 57;
  public static final int OP_C_SEQ_F   = 58;
  public static final int OP_C_NGL_F   = 59;
  public static final int OP_C_LT_F    = 60;
  public static final int OP_C_NGE_F   = 61;
  public static final int OP_C_LE_F    = 62;
  public static final int OP_C_NGT_F   = 63;

  public static final int FP_S         = 0;
  public static final int FP_D         = 1;

  public static final int UNDEFINED_DIVIDE = 0x00000000;

  private Simulator simulator;
  private State state;
  private StateHistoryBuffer stateHistoryBuffer;

  private boolean bInvalidOpcodeDetected;
  // This flag becomes TRUE if the executed instruction
  //   is not supported or simply invalid.

  public InstructionExecuter(StateHistoryBuffer stateHistoryBuffer, Simulator simulator) {
	this.simulator = simulator;
	this.state = simulator.getState();
	this.stateHistoryBuffer = stateHistoryBuffer;
  }    

  public void status(String sMessage) {
	// When an instruction is executed, this method is called.
	//   The sMessage parameter contains a string that describes
	//   the result of the executed instruction.  It can be
	//   displayed or logged for debugging purposes.
	System.out.println(sMessage);
  }    

  private void unknownOpcode() {
	status("UNKOWN OPCODE");
	iExecuteResult = ERR_INVALID_OPCODE;
  }  // end method unknownOpcode()    

  private int iExecuteResult;
  private static final int ERR_NONE = 0;
  private static final int ERR_INVALID_OPCODE = 1;
  private static final int ERR_FP_HARDWARE_REQUIRED = 2;
  private static final int ERR_SYSCALL = 3;
  private static final int ERR_BREAK = 4;
  private static final int ERR_OVERFLOW = 5;
  private static final int ERR_FP_EXCEPTION = 6;

  public int iExecute(InstructionDecodeBuffer db) {
  /*
  RETURNS
	0 = SUCCESS
   -1 = CRITICAL_ERROR
   -2 = INVALID_OPCODE
   -3 = INTERNAL_ERROR
   -4 = FP_HARDWARE_REQUIRED
   -5 = SYSCALL
   -6 = BREAK
   -7 = ARITHMETIC OVERFLOW
   -8 = FP ERROR (divide by zero)
  */

	int iResult = -3;

	iExecuteResult = ERR_NONE;

	try {
	  execute(db);
	} catch (Exception e) {
	  System.err.println("INTERNAL ERROR: " + e);
	  return -1;  // CRITICAL_ERROR (out of memory, index out of bounds?)
	}

	switch (iExecuteResult) {

	  case ERR_NONE:
		iResult = 0;
		break;

	  case ERR_INVALID_OPCODE:
		iResult = -2;
		break;

	  case ERR_FP_HARDWARE_REQUIRED:
		iResult = -4;
		break;

	  case ERR_SYSCALL:
		iResult = -5;
		break;

	  case ERR_BREAK:
		iResult = -6;
		break;

	  case ERR_OVERFLOW:
		iResult = -7;
		break;

	  case ERR_FP_EXCEPTION:
		iResult = -8;
		break;

	  default:
		System.err.println("EXECUTE ERROR: " + iExecuteResult);
		iResult = -3;
		break;

	}

	return iResult;
  }  // end method iExecute(InstructionDecodeBuffer)    

  private void execute(InstructionDecodeBuffer db) {

	int i1,i2,i3,i4;
	long l1,l2;
	short s1,s2,s3;

	bInvalidOpcodeDetected = false;

	switch (db.opcode) {

	  case OP_MULTI_1:
		executeMulti1Instruction(db);
		break;

	  case OP_MULTI_2:
		executeMulti2Instruction(db);
		break;

	  case OP_J:  // Unconditional Jump
		// A-65: j target
		// Unconditionally jump to the instruction at target

		// target should be an unsigned 26-bit number
		i1 = db.target * 4;
		state.setRegister(RegisterFile.R_PC, i1);

		// status("PC == " + Utility.sAsHexPadded(i1,8));
		break;

	  case OP_JAL:  // Jump and link
		// A-65: jal target
		// Unconditionally jump to the instruction at target.
		//   Save the address of the next instruction in register $ra
		i1 = state.iGetRegister(RegisterFile.R_PC);
		i2 = db.target * 4;
		state.setRegister(RegisterFile.R_RA, i1);
		state.setRegister(RegisterFile.R_PC, i2);

		// status("REG[RA] = " + Utility.sAsHexPadded(i1,8));
		// status("PC = " + Utility.sAsHexPadded(i2,8));
		break;

	  case OP_BEQ:  // Branch on equal
		// A-62: beq rs, rt, label
		// Conditionally branch the number of instructions specified
		//   by the offset if register rs equals rt.
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);

		// status("REG[" + db.rs + "] == REG[" + db.rt + "]?");

		if (i1 == i2) {
		  i3 = state.iGetRegister(RegisterFile.R_PC) - 4;  // get current PC
		  i3 = i3 + (db.offset * 4);

		  // db.offset is a signed 16-bit number, so we can branch
		  //   forward or backwards.

		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i3,8) + "\t[offset = " + db.offset + "]");

		  state.setRegister(RegisterFile.R_PC, i3);
		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_BNE:  // Branch on not equal
		// A-63: bne rs, rt, label
		// Conditionally branch the number of instructions specified
		//   by the offset if register rs is not equal to rt.
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);

		// status("REG[" + db.rs + "] != REG[" + db.rt + "]?");

		if (i1 != i2) {
		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);

		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1,8) );

		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_BLEZ: // Branch on less than equal zero
		// A-63: blez rs, label
		// Conditionally branch the number of instructions specified by
		//   the offset if register rs is less than or equal to 0.
		i1 = state.iGetRegister(db.rs);

		// status("REG["+db.rs+"] <= 0?");

		if (i1 <= 0) {
		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);

		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1,8) );

		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_BGTZ:  // Branch on greater than zero
		// A-62: bgtz rs, label
		// Conditionally branch the number of instructions specified by
		//   the offset if register rs is greater than 0.
		i1 = state.iGetRegister(db.rs);

		// status("REG["+db.rs+"] > 0?");

		if (i1 > 0) {
		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);

		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1,8) );

		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_ADDI:  // Addition immediate (with overflow)
		// A-55: addi rt, rs, imm
		// Put the sum of register rs and the signed-extended
		//   immediate into register rt

		i1 = state.iGetRegister(db.rs);
		i2 = (int)((short)db.imm);  // sign-extend db.imm
		i3 = i1 + i2;  // Java will sign extend s1

		if (  ( (i1 >= 0) & (i2 >= 0) & (i3 < 0) )
			| ( (i1 < 0) & (i2 < 0) & (i3 >= 0) )
		   ) {
		  // report overflow exception
		  // status("OVERFLOW EXCEPTION");
		  iExecuteResult = ERR_OVERFLOW;
		} else {
		  state.setRegister(db.rt, i3);
		}

		// status("REG["+db.rt+"] = " + "REG["+db.rs+"] + " + i2 + " == " + Utility.sAsHexPadded(i3,8));

		break;

	  case OP_ADDIU:  // Addition immediate (without overflow)
		// A-55: addiu rt, rs, imm

		i1 = state.iGetRegister(db.rs);
		i2 = (int)((short)db.imm);  // sign-extend db.imm
		i3 = i1 + i2;  // Java will sign extend s1

		state.setRegister(db.rt, i3);

		// status("REG["+db.rt+"] = " + "REG["+db.rs+"] + " + i2 + " == " + Utility.sAsHexPadded(i3,8));

		break;

	  case OP_SLTI:  // Set less than immediate
		// A-60: slti rt, rs, imm
		// Set register rt to 1 if register rs is less than the
		//   sign-extended immediate, and to 0 otherwise.
		i1 = state.iGetRegister(db.rs);
		i2 = 0;
		if (i1 < (int)((short)db.imm))  // sign-extend db.imm
		  i2 = 1;
		state.setRegister(db.rt, i2);

		// status("REG["+db.rs+"] < " + Utility.sAsHexPadded(db.imm,8) + "?");
		// status("SET REG["+db.rt+"] == " + i2);

		break;

	  case OP_SLTIU:  // Set less than unsigned immediate
		// A-60: sltiu rt, rs, imm
		// Set register rt to 1 if register rs is less than the
		//   unsigned immediate, and to 0 otherwise.

		// sltiu d,s,j
		// d = ((unsigned)s < (unsigned)j) ? 1:0;

		l1 = (long)(state.iGetRegister(db.rs) & 0x00000000FFFFFFFFL);
		l2 = (long)(db.imm & 0x00000000FFFFFFFFL);
		i3 = 0;
		if (l1 < l2)
		  i3 = 1;
		state.setRegister(db.rt, i3);

		// status("REG["+db.rs+"] < " + Utility.sAsHexPadded(l2,8) + "?");
		// status("SET REG["+db.rt+"] == " + i3);

		break;

	  case OP_ANDI:  // AND immediate
		// A-55: and rd, rs, rt
		// Put the logical AND of register rs and the
		//   zero-extended immediate into register rt
		// reg[rt] = reg[rs] & imm
		i1 = state.iGetRegister(db.rs);
		i2 = i1 & db.imm;
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = REG["+db.rs+"] & " + Utility.sAsHexPadded(db.imm,8) + " == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_ORI:  // OR immediate
		// A-57: ori rt, rs, imm
		// Put the logical OR of register rs and the
		//   zero-extended immediate into register rt
		// reg[rt] = reg[rs] | imm
		i1 = state.iGetRegister(db.rs);
		i2 = i1 | db.imm;
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = REG["+db.rs+"] | " + Utility.sAsHexPadded(db.imm,8) + " == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_XORI:  // XOR immediate
		// A-59: xori rt, rs, imm
		// Put the logical XOR of register rs and the
		//   zero-extended immediate into register rt
		// reg[rt] = reg[rs] ^ imm
		i1 = state.iGetRegister(db.rs);
		i2 = i1 ^ db.imm;
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = REG["+db.rs+"] ^ " + Utility.sAsHexPadded(db.imm,8) + " == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LUI:  // Load upper immediate
		// A-59: lui rt, imm
		// Load the lower halfword of the immediate imm into the upper
		//   halfword of register rt.  The lower bits of the register
		//   are set to 0.

		// lui t, u
		//   t = u << 16;

		i1 = db.imm << 16;
		i2 = i1 & 0xFFFF0000;
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_MULTI_3:  // z == 0
		executeMultiZInstruction(db, 0);
		break;

	  case OP_MULTI_4:  // z == 1
		executeMultiZInstruction(db, 1);
		break;

	  case OP_MULTI_5:  // z == 2
		executeMultiZInstruction(db, 2);
		break;

	  case OP_MULTI_6:  // z == 3
		executeMultiZInstruction(db, 3);
		break;

	  case OP_LB:  // Load byte
		// A-65: lb rt, address
		// Load the byte at address into register rt.  The byte is
		//   signed-extended by lb, but not by lbu.

		// lb t,addr
		//   ==>  t = *((signed char *)addr);

		i1 = state.iGetRegister(db.rs);
		i2 = state.loadByte(db.offset + i1);  // Java sign-extends
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = byte MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LH:
		// A-66: lh rt, address
		// Load the 16-bit quantity (halfword) at address into register rt.
		//   The halfword is sign-extended by lh, but not by lhu.
		i1 = state.iGetRegister(db.rs);
		i2 = state.loadHalfWord(db.offset + i1);  // Java sign-extends
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = halfword MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LWL:
		// A-66: lwl rt, address
		// Load the left bytes from the word at the possibly 
		//   unaligned address into register rt.
		i1 = db.offset + state.iGetRegister(db.rs);
		i3 = 24;  // 8, 16, 24
		i4 = 0x00000000;
		do {
		  i2 = (state.loadByte(i1) & 0x000000FF) << i3;

		  i4 = i4 | i2;

		  if ( (i1 & 0x00000003) == 0)
			break;

		  i1 = i1 - 1;
		  i3 = i3 - 8;

		} while (true);

		state.setRegister(db.rt, i4);
		break;

	  case OP_LW:  // Load word
		// A-66: lw rt, address

		// lw t, addr
		// t = *((*int)(addr));

		// load word @ address into register rt
		i1 = state.iGetRegister(db.rs);
		i2 = state.loadWord(db.offset + i1);
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = word MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LBU:  // Load byte unsigned
		// A-66: lbu rt, address
		// Load the byte at address into register rt.  The byte is
		//   signed-extended by lb, but not by lbu.

		i1 = state.iGetRegister(db.rs);
		i2 = state.loadByte(db.offset + i1) & 0x000000FF;  // zero-extend
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = (zero extended) byte MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LHU:
		// A-66: lhu rt, address
		i1 = state.iGetRegister(db.rs);
		i2 = state.loadHalfWord(db.offset + i1) & 0x0000FFFF;  // zero-extended
		state.setRegister(db.rt, i2);

		// status("REG["+db.rt+"] = halfword MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LWR:
		// A-66: lwr rt, address
		i1 = db.offset + state.iGetRegister(db.rs);   // address counter
		i3 = 0;                                      // shift counter
		i4 = 0x00000000;                             // holds final value
		do {
		  i2 = (state.loadByte(i1) & 0x000000FF) << i3;

		  i4 = i4 | i2;

		  if ( (i1 & 0x00000003) == 0)
			break;

		  i1 = i1 + 1;
		  i3 = i3 + 8;

		} while (true);

		state.setRegister(db.rt, i4);

		break;

	  case OP_SB:  // Store byte
		// A-67: sb rt, address
		// Store the low byte from register rt at address

		// sb t,addr
		//   *((char*)addr) = t;

		i1 = state.iGetRegister(db.rs);
		byte b = (byte)(state.iGetRegister(db.rt) & 0x000000FF);
		i1 = db.offset + i1;
		state.storeByte(i1, b);

		// status("MEM["+ Utility.sAsHexPadded(i1, 8) + "] = byte " + Utility.sAsHexPadded(b, 2));

		break;

	  case OP_SH:  // Store halfword
		// A-67: sh rt, address
		// Store the low halfword from register rt at address

		// sh t, addr
		//   *((short*)addr) = t;

		i1 = state.iGetRegister(db.rs);
		i2 = (state.iGetRegister(db.rt) & 0x0000FFFF);
		i1 = db.offset + i1;
		state.storeHalfWord(i1, i2);

		// status("MEM["+ Utility.sAsHexPadded(i1, 8) + "] = halfword " + Utility.sAsHexPadded(i2, 4));

		break;

//      case OP_SWL:
// **INCOMPLETE
		// A-68: swl rt, address
//        break;

	  case OP_SW:  // Store word
		// A-67: sw rt, address
		// Store the word from register rt at address
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i1 = db.offset + i1;
		state.storeWord(i1, i2);

		// status("MEM["+ Utility.sAsHexPadded(i1, 8) + "] = word " + Utility.sAsHexPadded(i2, 8));

		break;

//      case OP_SWR:
		// A-68: swr rt, address
// **INCOMPLETE
//        break;

	  case OP_LWC0:
		// load word @ address into register rt on CP0
		i1 = state.iGetRegister(db.rs);
		i2 = state.loadWord(db.offset + i1);
		state.setCP0Register(db.rt, i2);

		// status("CP0REG["+db.rt+"] = word MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

	  case OP_LWC1:
		// load word @ address into register rt on CP1
		i1 = state.iGetRegister(db.rs);
		i2 = state.loadWord(db.offset + i1);
		state.setCP1Register(db.rt, i2);

		// status("CP1REG["+db.rt+"] = word MEM["+ Utility.sAsHexPadded(db.offset+i1,8)+"] == " + Utility.sAsHexPadded(i2,8));

		break;

//      case OP_LWC2:
//        break;
//      case OP_LWC3:
//        break;

	  case OP_SWC0:
		// Store the word from CP0 register rt at address
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetCP0Register(db.rt);
		i1 = db.offset + i1;
		state.storeWord(i1, i2);

		// status("MEM["+ Utility.sAsHexPadded(i1, 8) + "] = CP0 word " + Utility.sAsHexPadded(i2, 8));

		break;

	  case OP_SWC1:
		// A-68: swcz rt, address
		// Store the word from CP1 register rt at address
		// MEM[address] = rt

		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetCP1Register(db.rt);

		// status("swc1 address == " + Utility.sAsHexPadded(db.offset, 4) + " (reg" + db.rs + ") " + Utility.sAsHexPadded(i1, 8));

		i1 = db.offset + i1;
		state.storeWord(i1, i2);

		// status("MEM["+ Utility.sAsHexPadded(i1, 8) + "] = CP1 word " + Utility.sAsHexPadded(i2, 8));

		break;

//      case OP_SWC2:
//        break;
//      case OP_SWC3:
//        break;

	  default:
		unknownOpcode();
		break;

	}

  }  // end method execute(InstructionDecodeBuffer)    

  private void executeMulti1Instruction(InstructionDecodeBuffer db) {

	int i1,i2,i3;
	long l1,l2,l3;

	switch (db.funct) {

	  case OP_SLL:  // Shift left logical
		// A-58: sll rd, rt, shamt
		// Shift register rt left by the distance indicated by
		//  immediate shamt and put the result
		//  in register rd.  Argument rs is ignored.
		i1 = state.iGetRegister(db.rt);
		i1 = i1 << db.shamt;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] << " + db.shamt + ")");
		break;

	  case OP_SRL:  // Shift right logical
		// A-58: srl rd, rt, shamt
		i1 = state.iGetRegister(db.rt);
		i1 = i1 >> db.shamt;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] >> " + db.shamt + ")");
		break;

	  case OP_SRA:  // Shift right arithmetic
		// A-58: sra rd, rt, shamt
		i1 = state.iGetRegister(db.rt);
		i1 = i1 >>> db.shamt;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] >>> " + db.shamt + ")");
		break;

	  case OP_SLLV:  // Shift left logical variable
		// A-58: sllv rd, rt, rs
		i1 = state.iGetRegister(db.rt);
		i2 = state.iGetRegister(db.rs);
		i1 = i1 << i2;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] << REG[" + db.rs + "])");
		break;

	  case OP_SRLV:  // Shift right logical variable
		// A-58: srlv rd, rt, rs
		i1 = state.iGetRegister(db.rt);
		i2 = state.iGetRegister(db.rs);
		i1 = i1 >> i2;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] >> REG[" + db.rs + "])");
		break;

	  case OP_SRAV:  // Shift right arithmetic variable
		// A-58: srav rd, rt, rs
		i1 = state.iGetRegister(db.rt);
		i2 = state.iGetRegister(db.rs);
		i1 = i1 >>> i2;
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = (REG["+db.rt+"] >>> REG[" + db.rs + "])");
		break;

	  case OP_JR:  // Jump register
		// A-65: jr rs
		// Unconditionally jump to the instruction whose
		//   address is in register rs
		i1 = state.iGetRegister(db.rs);
		state.setRegister(RegisterFile.R_PC, i1);

		// status("PC = REG["+db.rs+"] == " + Utility.sAsHexPadded(i1,8) );
		break;

	  case OP_JALR:  // Jump and link register
		// A-65: jalr rs, rd
		// Unconditionally jump to the instruction whose address
		//   is in register rs.  Save the address of the next
		//   instruction in register rd

		i1 = state.iGetRegister(RegisterFile.R_PC);  // get the return address
		state.setRegister(db.rd, i1);  // put in REG[rd]

		i2 = state.iGetRegister(db.rs);
		state.setRegister(RegisterFile.R_PC, i2);

		// status("REG["+db.rd+"] = PC == " + Utility.sAsHexPadded(i1,8));
		// status("PC = REG["+db.rs+"] == " + Utility.sAsHexPadded(i2,8));
		break;

	  case OP_SYSCALL:  // System call
		// A-74: syscall
		// Register $v0 contains the number of the system call provided
		//   by the simulator
		i1 = state.iGetRegister(RegisterFile.R_V0);
		performSystemCall(i1);

		// status("SYSTEM CALL SERVICE #" + i1);
		break;

	  case OP_BREAK:  // Break
		// A-75: break code
		// Cause exception code.  Exception 1 is reserved for the debugger.
		iExecuteResult = ERR_BREAK;
		break;

	  case OP_MFHI:  // Move from hi
		// A-68: mfhi rd

		// if integer mul/div is not done, we need to stall the instruction

		i1 = state.iGetRegister(RegisterFile.R_HI);
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = REG[HI] == " + Utility.sAsHexPadded(i1,8));
		break;

	  case OP_MTHI:  // Move to hi
		// A-69: mthi rs
		i1 = state.iGetRegister(db.rs);
		state.setRegister(RegisterFile.R_HI, i1);

		// status("REG[HI] = REG["+db.rs+"] == " + Utility.sAsHexPadded(i1,8));
		break;

	  case OP_MFLO:  // Move from lo
		// A-69: mflo rd

		// if integer mul/div is not done, we need to stall the instruction

		i1 = state.iGetRegister(RegisterFile.R_LO);
		state.setRegister(db.rd, i1);

		// status("REG["+db.rd+"] = REG[LO] == " + Utility.sAsHexPadded(i1,8));
		break;

	  case OP_MTLO:  // Move to lo
		// A-69: mtlo rs
		i1 = state.iGetRegister(db.rs);
		state.setRegister(RegisterFile.R_LO, i1);

		// status("REG[LO] = REG["+db.rs+"] == " + Utility.sAsHexPadded(i1,8));
		break;

	  case OP_MULT:  // Multiply
		// A-56: mult rs, rt
		// Multiply register rs and rt.  Leave the low-order word of
		//   the product in register lo and the high-order word
		//   in register hi.
		l1 = (long)state.iGetRegister(db.rs);  // Java sign extends
		l2 = (long)state.iGetRegister(db.rt);

		l1 = l1 * l2;

		i1 = (int)((l1 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l1 & 0x00000000FFFFFFFFL);

		state.setRegister(RegisterFile.R_HI, i1);
		state.setRegister(RegisterFile.R_LO, i2);

		// status("REG["+db.rs+"] * REG["+db.rt+"] = " + l1);
		break;

	  case OP_MULTU:
		// A-56: multu rs, rt
		l1 = (long)(state.iGetRegister(db.rs) & 0x00000000FFFFFFFFL);
		l2 = (long)(state.iGetRegister(db.rt) & 0x00000000FFFFFFFFL);

		l1 = l1 * l2;

		i1 = (int)((l1 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l1 & 0x00000000FFFFFFFFL);

		state.setRegister(RegisterFile.R_HI, i1);
		state.setRegister(RegisterFile.R_LO, i2);

		// status("REG["+db.rs+"] * REG["+db.rt+"] = " + l1);
		break;

	  case OP_DIV:  // Divide (signed)
		// A-56: div rs, rt
		// Divide register rs by register rt.  Leave the quotient in
		//   register lo and the remainder in register hi.  Note that if
		//   an operand is negative, the remainder is unspecified
		//   by the MIPS architecture and depends on the convension of
		//   the machine on which SPIM is run.
		l1 = (long)state.iGetRegister(db.rs);  // Java sign extends
		l2 = (long)state.iGetRegister(db.rt);

		if (l2 == 0) {
		  // no exception in to be thrown
		  //state.setRegister(RegisterFile.R_HI, UNDEFINED_DIVIDE);
		  //state.setRegister(RegisterFile.R_LO, UNDEFINED_DIVIDE);
		  //status("DIVID BY ZERO - IGNORED");
		} else {
		  state.setRegister(RegisterFile.R_LO, (int)(l1 / l2) );
		  state.setRegister(RegisterFile.R_HI, (int)(l1 % l2) );
		  //status("REG["+db.rs+"] / REG["+db.rt+"] == " + (l1/l2) );
		  //status("REG[HI] = " + Utility.sAsHexPadded( (int)(l1 % l2), 8) + ",  REG[LO] = " + Utility.sAsHexPadded( (int)(l1 / l2), 8) );
		}

		break;

	  case OP_DIVU:  // Divide (unsigned)
		// A-56: divu rs, rt
		l1 = (long)state.iGetRegister(db.rs) & 0x00000000FFFFFFFFL;
		l2 = (long)state.iGetRegister(db.rt) & 0x00000000FFFFFFFFL;

		if (l2 == 0) {
		  // no exception in to be thrown
		  //state.setRegister(RegisterFile.R_HI, UNDEFINED_DIVIDE);
		  //state.setRegister(RegisterFile.R_LO, UNDEFINED_DIVIDE);
		} else {
		  state.setRegister(RegisterFile.R_LO, (int)(l1 / l2) );
		  state.setRegister(RegisterFile.R_HI, (int)(l1 % l2) );
		}

		// status("REG["+db.rs+"] / REG["+db.rt+"] == " + (l1/l2) );
		// status("REG[HI] = " + Utility.sAsHexPadded( (int)(l1 % l2), 8) + ",  REG[LO] = " + Utility.sAsHexPadded( (int)(l1 / l2), 8) );
		break;

	  case OP_ADD:  // Addition (with overflow)
		// A-55: add rd, rs, rt
		// Put the sum of registers rs and rt into register rd
		// rd = rs + rt
		// Generate a trap if the result overflows into bit 31 (sign bit)
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 + i2;

		// Page 222, Fig 4.4: Overflow conditions for addition and subtraction.
		if (  ( (i1 >= 0) & (i2 >= 0) & (i3 < 0) )
			| ( (i1 < 0) & (i2 < 0) & (i3 >= 0) )
		   ) {
		  // report overflow exception
		  // status("OVERFLOW EXCEPTION");
		  iExecuteResult = ERR_OVERFLOW;

		} else {
		  state.setRegister(db.rd, i3);

		  // status("REG["+db.rd+"] = REG["+db.rs+"] + REG["+db.rt+"] == " + Utility.sAsHexPadded(i3,8));

		}

		break;

	  case OP_ADDU:  // Addition (without overflow)
		// A-55: addu rd, rs, rt
		// U means no-overflow test.
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 + i2;
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] + REG["+db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_SUB:  // Subtract (with overflow)
		// A-59: sub rd, rs, rt
		// Put the difference of registers rs and rt into register rd.
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 - i2;

		// Page 222, Fig 4.4: Overflow conditions for addition and subtraction.
		if (  ( (i1 >= 0) & (i2 < 0) & (i3 < 0) )
			| ( (i1 < 0) & (i2 >= 0) & (i3 >= 0) )
		   ) {
		  // report overflow exception
		  // status("OVERFLOW EXCEPTION");
		  iExecuteResult = ERR_OVERFLOW;

		} else {

		  state.setRegister(db.rd, i3);
		  // status("REG["+db.rd+"] = REG["+db.rs+"] - REG["+db.rt+"] == " + Utility.sAsHexPadded(i3,8));

		}

		break;

	  case OP_SUBU:  // Subtract (without overflow)
		// A-59: subu rd, rs, rt
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 - i2;
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] - REG["+db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_AND:  // AND
		// A-55: and rd, rs, rt
		// Put the logical AND of registers rs and rt into register rd
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 & i2;
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] & REG["+ db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_OR:  // OR
		// A-57: or rd, rs, rt
		// Put the logical OR of registers rs and rt into register rd
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 | i2;
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] | REG["+ db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_XOR:  // XOR
		// A-59: xor rd, rs, rt
		// Put the logical NOR of registers rs and rt into register rd
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = i1 ^ i2;
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] ^ REG["+ db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_NOR:  // NOR
		// A-57: nor rd, rs, rt
		// Put the logical NOR of registers rs and rt into register rd
		i1 = state.iGetRegister(db.rs);
		i2 = state.iGetRegister(db.rt);
		i3 = ~(i1 | i2);
		state.setRegister(db.rd, i3);

		// status("REG["+db.rd+"] = REG["+db.rs+"] ~| REG["+ db.rt+"] == " + Utility.sAsHexPadded(i3,8));
		break;

	  case OP_SLT:  // Set less than
		// A-60: slt rd, rs, rt
		// Set register rd to 1 if register rs is less than rt, and to 0 otherwise.

		// sltu d, s, t
		//  d = ((signed)s < (signed)t) ? 1 : 0;

		l1 = (long)state.iGetRegister(db.rs);  // Java sign extends
		l2 = (long)state.iGetRegister(db.rt);
		i3 = 0;

		// status("IS REG["+db.rs+"] < REG["+db.rt+"]?");
		if (l1 < l2) {
		  i3 = 1;
		}

		// status("REG["+db.rd+"] == " + i3);

		state.setRegister(db.rd, i3);

		break;

	  case OP_SLTU:  // Set less than unsigned
		// A-60: sltu rd, rs, rt

		// sltu d, s, t
		//  d = ((unsigned)s < (unsigned)t) ? 1 : 0;

		l1 = (long)(state.iGetRegister(db.rs) & 0x00000000FFFFFFFFL);
		l2 = (long)(state.iGetRegister(db.rt) & 0x00000000FFFFFFFFL);
		i3 = 0;
		if (l1 < l2)
		  i3 = 1;

		// status("REG["+db.rd+"] == " + i3);

		state.setRegister(db.rd, i3);

		break;

	  default:
		// unknown opcode
		unknownOpcode();

	}

  }  // end method executeMulti1Instruction(InstructionDecodeBuffer)    

  private void executeMulti2Instruction(InstructionDecodeBuffer db) {

	int i1,i2,i3;

	switch (db.rt) {

	  case OP_BLTZ:  // Branch on less than zero
		// A-63: bltz rs, label
		// Conditionally branch the number of instructions specified
		//   by the offset if register rs is less than 0.
		i1 = state.iGetRegister(db.rs);

		// status("REG["+db.rs+"] < 0?");

		if (i1 < 0) {
		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);
		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1,8));
		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_BGEZ:  // Branch on greater than equal zero
		// A-62: bgez rs, label
		// Conditionally branch the number of instructions specified by
		//   the offset if register rs is greater than or equal to 0
		i1 = state.iGetRegister(db.rs);

		// status("REG["+db.rs+"] >= 0?");

		if (i1 >= 0) {
		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);
		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1, 8));
		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  case OP_BLTZAL:  // Branch on less than and link
		// A-63: bltzal rs, label
		// Conditionally branch the number of instructions specified by
		//   the offset if register rs is less than 0.  Save the address
		//   of the next instruction in register 31.
		i1 = state.iGetRegister(db.rs);
		// status("REG["+db.rs+"] < 0?");
		if (i1 < 0) {

		  i1 = state.iGetRegister(RegisterFile.R_PC);
		  state.setRegister(RegisterFile.R_RA, i1);
		  // status("BRANCH TAKEN, REG[RA] = " + Utility.sAsHexPadded(i1,8));

		  i1 = state.iGetRegister(RegisterFile.R_PC) - 4;
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);

		  // status("PC = " + Utility.sAsHexPadded(i1,8));
		}
		break;

	  case OP_BGEZAL:  // Branch on greater than equal zero and link
		// A-62: bgezal rs, label
		i1 = state.iGetRegister(db.rs);
		// status("REG["+db.rs+"] >= 0?");
		if (i1 >= 0) {

		  i1 = state.iGetRegister(RegisterFile.R_PC);
		  state.setRegister(RegisterFile.R_RA, i1);

		  // status("BRANCH TAKEN, REG[RA] = " + Utility.sAsHexPadded(i1, 8));

		  i1 = i1 - 4;  // get the curent PC
		  i1 = i1 + (db.offset * 4);
		  state.setRegister(RegisterFile.R_PC, i1);

		  // status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i1, 8));
		} else {
		  // status("BRANCH NOT TAKEN");
		}
		break;

	  default:
		unknownOpcode();
		break;

	}
  }  // end method executeMulti2Instruction    

  private void executeMultiZInstruction(InstructionDecodeBuffer db, int z) {
	int i1, i2, i3;
	switch (db.rs) {
	  case OP_MFCZ:
		// A-69: mtcz rd, rt
		// Move contents from coprocessor Z register cs into gpr t
		// mfc0 t, cs
		switch (z) {
		  case 0:
			i1 = state.iGetCP0Register(db.rt);
			state.setRegister(db.rd, i1);
			break;
		  case 1:
			i1 = state.iGetCP1Register(db.rt);
			state.setRegister(db.rd, i1);
			break;
		  default:
			unknownOpcode();
			break;
		}
		break;

	  case OP_CFCZ:
		break;

	  case OP_MTCZ:
		// A-69: mtcz rd, rt
		// Move CPU register rt to coprocessor z's register rd
		switch (z) {
		  case 0:
			i1 = state.iGetRegister(db.rt);
			state.setCP0Register(db.rd, i1);
			break;
		  case 1:
			i1 = state.iGetRegister(db.rt);
			state.setCP1Register(db.rd, i1);
			break;
		  default:
			unknownOpcode();
			break;
		}
		break;

//      case OP_CTCZ:
// **INCOMPLETE
//        break;

	  case OP_MULTI_7:
		// check bit 16 to see if BCZF or BCZT instruction
		// BC0T BC1T BC0F BC1F
		if (Utility.bBitSet(db.rawInstruction, 16, false)) {
		  // execute BCZF
		  // Test if Cond = FALSE in CoprocessorZ
		  // If so, Branch to specified label
		  switch (z) {
			case 1:
			  i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
			  //status("Is Z1 cond false?");
			  if (Utility.bBitSet(i1, Coprocessor1.CONDITION_BIT, false)) {
				i2 = state.iGetRegister(RegisterFile.R_PC) - 4;  // get current PC
				i2 = i2 + (db.offset * 4);
				//status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i2,8) );
				state.setRegister(RegisterFile.R_PC, i2);
			  } else {
				//status("BRANCH NOT TAKEN");
			  }
			  break;
			default:
			  unknownOpcode();
		  }
		} else {
		  // execute BCZT
		  // Test if Cond = TRUE in CoprocessorZ
		  // If so, Branch to specified label
		  switch (z) {
			case 1:
			  i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
			  // status("Is Z1 cond true?");
			  if (Utility.bBitSet(i1, Coprocessor1.CONDITION_BIT, true)) {
				i2 = state.iGetRegister(RegisterFile.R_PC) - 4;  // get current PC
				i2 = i2 + (db.offset * 4);
				// status("BRANCH TAKEN, PC = " + Utility.sAsHexPadded(i2,8) );
				state.setRegister(RegisterFile.R_PC, i2);
			  } else {
				// status("BRANCH NOT TAKEN");
			  }

			  break;
			default:
			  unknownOpcode();
		  }
		}
		break;

	  case OP_COPZ_1:
		switch (z) {
		  case 0:
			executeMiscInstruction(db);
			break;
		  case 1:
			executeFPInstruction(db, FP_S);
			break;
		}
		break;
	  case OP_COPZ_2:
		switch (z) {
		  case 1:
			executeFPInstruction(db, FP_D);
			break;
		  default:
			unknownOpcode();
		}
		break;

	  default:
		unknownOpcode();
		break;

	}
  }  // executeMultiZInstruction(InstructionDecodeBuffer, int)    

  private void executeMiscInstruction(InstructionDecodeBuffer db) {

	switch (db.funct) {
// **INCOMPLETE
//      case OP_TLBR:
//        break;

//      case OP_TLBWL:
//        break;

//      case OP_TLBWR:
//        break;

//      case OP_TLBP:
//        break;

//      case OP_RFE:  // Return from exception
//        break;

	  default:
		unknownOpcode();
		break;
	}

  }  // end method executeMiscInstruction(InstructionDecodeBuffer)    

  private void executeFPInstruction(InstructionDecodeBuffer db, int fpType) {

	// Make sure the FPA is available.
	if (!simulator.bHasFPA()) {
	  iExecuteResult = ERR_FP_HARDWARE_REQUIRED;
	  return;
	}

	// fpType == FP_S or FP_D
	switch (fpType) {
	  case FP_S:
		executeFPSInstruction(db);
		break;

	  case FP_D:
		executeFPDInstruction(db);
		break;

	  default:
		unknownOpcode();
		break;
	}

  }  // end method executeFPInstruction(InstructionDecodeBuffer, int)    

  private void executeFPSInstruction(InstructionDecodeBuffer db) {
	// Single-precision arithmetic

	int i1, i2, i3;
	float f1, f2, f3 = 0;
	double d1;
	long l1;

	switch (db.funct) {
	  case OP_ADD_F:
		// A-70: add.s fd, fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		f3 = f1 + f2;

		i3 = Float.floatToIntBits(f3);

		state.setCP1Register(db.fd, i3);

		// status("CP1REG["+db.fd+"] = CP1REG["+db.fs+"] + CP1REG["+db.ft+"] == " + f3);
		break;

	  case OP_SUB_F:
		// A-74: sub.s fd, fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		f3 = f1 - f2;

		i3 = Float.floatToIntBits(f3);

		state.setCP1Register(db.fd, i3);

		// status("CP1REG["+db.fd+"] = CP1REG["+db.fs+"] - CP1REG["+db.ft+"] == " + f3);
		break;

	  case OP_MUL_F:
		// A-73: mul.s fd, fs, ft

		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		f3 = f1 * f2;

		i3 = Float.floatToIntBits(f3);

		state.setCP1Register(db.fd, i3);

		// status("CP1REG["+db.fd+"] = CP1REG["+db.fs+"] * CP1REG["+db.ft+"] == " + f3);
		break;

	  case OP_DIV_F:
		// A-73: mul.s fd, fs, ft

		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		if (f2 == 0.0) {
		  iExecuteResult = ERR_FP_EXCEPTION;
		} else {

		  f3 = f1 / f2;

		  i3 = Float.floatToIntBits(f3);

		  state.setCP1Register(db.fd, i3);
		}
		//status("CP1REG["+db.fd+"] = CP1REG["+db.fs+"] / CP1REG["+db.ft+"] == " + f3);
		break;

	  case OP_ABS_F:
		// A-70: abs.s fd, fs
		i1 = state.iGetCP1Register(db.fs);

		f1 = Float.intBitsToFloat(i1);

		f2 = Math.abs(f1);

		i2 = Float.floatToIntBits(f2);

		state.setCP1Register(db.fd, i2);

		// status("CP1REG["+db.fd+"] = abs(CP1REG["+db.fs+"]) == " + f2);
		break;

	  case OP_MOV_F:
		// A-73: mov.s fd, fs
		i1 = state.iGetCP1Register(db.fs);
		state.setCP1Register(db.fd, i1);

		// status("CP1REG["+db.fd+"] = CP1REG["+db.fs+"]");
		break;

	  case OP_NEG_F:
		// A-73: neg.s fd, fs
		i1 = state.iGetCP1Register(db.fs);
		f1 = Float.intBitsToFloat(i1);
		f2 = -f1;
		i2 = Float.floatToIntBits(f2);
		state.setCP1Register(db.fd, i2);

		// status("CP1REG["+db.fd+"] = -CP1REG["+db.fs+"] == " + f2);
		break;

	  case OP_CVT_S_F:
		// A-72: cvt.s.w fd, fs
		// Convert integer to single
		i1 = state.iGetRegister(db.fs);
		f1 = (float)i1;  // convert integer to float
		i2 = Float.floatToIntBits(f1);
		state.setCP1Register(db.fd, i2);

		// status("CP1REG["+db.fd+"] = (float)REG["+db.fs+"]");
		break;

	  case OP_CVT_D_F:
		// A-72: cvt.d.s
		// Convert single-precision FP to double prec, store in fd
		i1 = state.iGetRegister(db.fs);

		f1 = Float.intBitsToFloat(i1);

		d1 = (double)f1;  // convert single to double
		l1 = Double.doubleToLongBits(d1);

		i1 = (int)((l1 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l1 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd, i1);
		state.setCP1Register(db.fd+1, i2);

		// status("CP1REG["+db.fd+"] = (double)CP1REG["+db.fs+"]");
		break;

	  case OP_CVT_W_F:
		// A-72: cvt.w.s fd, fs
		// Convert single to integer
		i1 = state.iGetCP1Register(db.fs);
		f1 = Float.intBitsToFloat(i1);

		i1 = (int)f1;  // convert single to integer

		state.setRegister(db.fd, i1);

		// status("REG["+db.fd+"] = (int)CP1REG["+db.fs+"]");
		break;

//      case OP_C_F_F:
// **INCOMPLETE
//        break;

//      case OP_C_UN_F:
// **INCOMPLETE
//        break;

	  case OP_C_EQ_F:
		// A-71: c.eq.s fs, ft
		// Compare fs and ft, and set FP cond flag to true if equal
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
		// status("CP1REG["+db.fs+"] == CP1REG["+db.fs+"]?");
		if (f2 == f1) {
		  // set the flag
		  // status("TRUE, set CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  // status("FALSE, clear CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}

		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		break;

	  case OP_C_UEQ_F:
// **INCOMPLETE
//        break;

//      case OP_C_OLT_F:
// **INCOMPLETE
//        break;

//      case OP_C_ULT_F:
// **INCOMPLETE
//        break;

//      case OP_C_OLE_F:
// **INCOMPLETE
//        break;

//      case OP_C_ULE_F:
// **INCOMPLETE
//        break;

//      case OP_C_ST_F:
// **INCOMPLETE
//        break;

//      case OP_C_NGLE_F:
// **INCOMPLETE
//        break;

//      case OP_C_SEQ_F:
// **INCOMPLETE
//        break;

//      case OP_C_NGL_F:
// **INCOMPLETE
//        break;

	  case OP_C_LT_F:
		// A-71: c.lt.s fs, ft
		// Compare FP single fs < ft, and set COND if true
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
		if (f1 < f2) {
		  // set the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}
		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		// status("CP1REG["+db.fs+"] < CP1REG["+db.fs+"]? If so, set COND");
		break;

//      case OP_C_NGE_F:
// **INCOMPLETE
//        break;

	  case OP_C_LE_F:
		// A-71: c.le.s fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.ft);

		f1 = Float.intBitsToFloat(i1);
		f2 = Float.intBitsToFloat(i2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);

		// status("CP1REG["+db.fs+"] <= CP1REG["+db.ft+"]?");
		if (f1 <= f2) {
		  // set the flag
		  // status("TRUE, set CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  // status("FALSE, clear CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}
		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		break;

//      case OP_C_NGT_F:
// **INCOMPLETE
//        break;

	  default:
		unknownOpcode();
		break;

	}
  }  // end method executeFPSInstruction()    

  private void executeFPDInstruction(InstructionDecodeBuffer db) {
	// Double-precision arithmetic

	int i1, i2, i3, i4;
	double d1, d2, d3 = 0;
	float f1;
	long l1, l2, l3;

	switch (db.funct) {

	  case OP_ADD_F:
		// A-70: add.d fd, fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		d3 = d1 + d2;
		l3 = Double.doubleToLongBits(d3);

		i1 = (int)((l3 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l3 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = dbl CP1REG["+db.fs+"] + dbl CP1REG["+db.ft+"] == " + d3);
		break;

	  case OP_SUB_F:
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		d3 = d1 - d2;
		l3 = Double.doubleToLongBits(d3);

		i1 = (int)((l3 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l3 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = dbl CP1REG["+db.fs+"] - dbl CP1REG["+db.ft+"] == " + d3);
		break;

	  case OP_MUL_F:
		// A-73: mul.d fd, fs, ft

		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		d3 = d1 * d2;
		l3 = Double.doubleToLongBits(d3);

		i1 = (int)((l3 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l3 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = dbl CP1REG["+db.fs+"] * dbl CP1REG["+db.ft+"] == " + d3);
		break;

	  case OP_DIV_F:
		// A-73: div.d fd, fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		if (d2 == 0.0) {
		  iExecuteResult = ERR_FP_EXCEPTION;
		} else {
		  d3 = d1 / d2;
		  l3 = Double.doubleToLongBits(d3);

		  i1 = (int)((l3 & 0xFFFFFFFF00000000L) >>> 32);
		  i2 = (int)(l3 & 0x00000000FFFFFFFFL);

		  state.setCP1Register(db.fd+1, i1);
		  state.setCP1Register(db.fd, i2);

		}
		//status("dbl CP1REG["+db.fd+"] = dbl CP1REG["+db.fs+"] / dbl CP1REG["+db.ft+"] == " + d3);
		break;

	  case OP_ABS_F:
		// A-70: abs.d fd, fs
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		d2 = Math.abs(d1);
		l2 = Double.doubleToLongBits(d2);

		i1 = (int)((l2 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l2 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = abs(CP1REG["+db.fs+"]) == " + d2);
		break;

	  case OP_MOV_F:
		// A-73: mov.d fd, fs
		// Move the floating-point double from register fs to register fd.
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);

		state.setCP1Register(db.fd, i1);
		state.setCP1Register(db.fd+1, i2);

		// status("dbl CP1REG["+db.fd+"] = CP1REG["+db.fs+"]");
		break;

	  case OP_NEG_F:
		// A-73: neg.d fd, fs
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		d2 = -d1;
		l2 = Double.doubleToLongBits(d2);

		i1 = (int)((l2 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l2 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = -CP1REG["+db.fs+"] == " + d2);
		break;

	  case OP_CVT_S_F:
		// A-72: cvt.s.d fd, fs
		// Convert double to single
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		f1 = (float)d1;  // Let Java do the conversion
		i1 = Float.floatToIntBits(f1);
		state.setCP1Register(db.fd, i1);

		// status("CP1REG["+db.fd+"] = (float)dbl CP1REG["+db.fs+"]");
		break;

	  case OP_CVT_D_F:
		// A-72: cvt.d.w fd, fs
		// Convert integer in register fs to
		//   double number and put in fd
		i1 = state.iGetRegister(db.fs);

		d1 = (double)i1;  // Convert integer to double
		l2 = Double.doubleToLongBits(d1);

		i1 = (int)((l2 & 0xFFFFFFFF00000000L) >>> 32);
		i2 = (int)(l2 & 0x00000000FFFFFFFFL);

		state.setCP1Register(db.fd+1, i1);
		state.setCP1Register(db.fd, i2);

		// status("dbl CP1REG["+db.fd+"] = REG["+db.fs+"]");
		break;

	  case OP_CVT_W_F:
		// A-72: cvt.w.d fd, fs
		// Convert double to integer
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i1 = (int)d1;  // Convert double to integer
		state.setRegister(db.fd, i1);

		// status("REG["+db.fd+"] = dbl CP1REG["+db.fs+"]");
		break;

//      case OP_C_F_F:
// **INCOMPLETE
//        break;

//      case OP_C_UN_F:
// **INCOMPLETE
//        break;

	  case OP_C_EQ_F:
		// A-71: c.eq.d fs, ft
		// if FPREG[fs] == FPREG[ft], set Cond
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
		if (d2 == d1) {
		  // set the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}
		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		// status("dbl CP1REG["+db.fs+"] == dbl CP1REG["+db.fs+"]? If so, set COND");
		break;

//      case OP_C_UEQ_F:
// **INCOMPLETE
//        break;

//      case OP_C_OLT_F:
// **INCOMPLETE
//        break;

//      case OP_C_ULT_F:
// **INCOMPLETE
//        break;

//      case OP_C_OLE_F:
// **INCOMPLETE
//        break;

//      case OP_C_ULE_F:
// **INCOMPLETE
//        break;

//      case OP_C_ST_F:
// **INCOMPLETE
//        break;

//      case OP_C_NGLE_F:
// **INCOMPLETE
//        break;

//      case OP_C_SEQ_F:
// **INCOMPLETE
//        break;

//      case OP_C_NGL_F:
// **INCOMPLETE
//        break;

	  case OP_C_LT_F:
		// A-71: c.lt.d fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);
		if (d2 < d1) {
		  // set the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}
		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		// status("dbl CP1REG["+db.fs+"] < dbl CP1REG["+db.fs+"]? If so, set COND");
		break;

//      case OP_C_NGE_F:
// **INCOMPLETE
//        break;

	  case OP_C_LE_F:
		// A-71: c.le.d fs, ft
		i1 = state.iGetCP1Register(db.fs);
		i2 = state.iGetCP1Register(db.fs+1);
		l1 = (long)i2 << 32;
		l1 = l1 | (i1 & 0x00000000FFFFFFFFL);
		d1 = Double.longBitsToDouble(l1);

		i3 = state.iGetCP1Register(db.ft);
		i4 = state.iGetCP1Register(db.ft+1);
		l2 = (long)i4 << 32;
		l2 = l2 | (i3 & 0x00000000FFFFFFFFL);
		d2 = Double.longBitsToDouble(l2);

		i1 = state.iGetCP1Register(Coprocessor1.R_FCR1);

		//status("dbl CP1REG["+db.fs+"] <= dbl CP1REG["+db.fs+"]?");

		if (d1 <= d2) {
		  // set the flag
		  //status("TRUE, set CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, true);
		} else {
		  // clear the flag
		  //status("FALSE, clear CP1 COND");
		  i1 = Utility.iSetBit(i1, Coprocessor1.CONDITION_BIT, false);
		}
		state.setCP1Register(Coprocessor1.R_FCR1, i1);

		break;

//      case OP_C_NGT_F:
// **INCOMPLETE
//        break;

	  default:
		unknownOpcode();
		break;

	}
  }  // end method executeFPSInstruction()    

  private void performSystemCall(int iService) {

	int i1,i2,i3;
	String s = "";

	iExecuteResult = ERR_SYSCALL;

	switch (iService) {
	  case SC_PRINT_INT:
		state.setFlag(State.FLAG_SC_PRINT_INT, true);
		break;

	  case SC_PRINT_FLOAT:
		state.setFlag(State.FLAG_SC_PRINT_FLOAT, true);
		break;

	  case SC_PRINT_DOUBLE:
		state.setFlag(State.FLAG_SC_PRINT_DOUBLE, true);
		break;

	  case SC_PRINT_STRING:       
		state.setFlag(State.FLAG_SC_PRINT_STRING, true);
		break;

	  case SC_READ_INT:
		state.setFlag(State.FLAG_SC_READ_INT, true);
		break;

	  case SC_READ_FLOAT:
		state.setFlag(State.FLAG_SC_READ_FLOAT, true);
		break;

	  case SC_READ_DOUBLE:
		state.setFlag(State.FLAG_SC_READ_DOUBLE, true);
		break;

	  case SC_READ_STRING:
		state.setFlag(State.FLAG_SC_READ_STRING, true);
		break;

//      case SC_SBRK:
// **INCOMPLETE
		// $a0 = amount
		// store address into $v0
//        break;

	  case SC_EXIT:
		// terminate the program
		state.setFlag(State.FLAG_SC_EXIT, true);
		break;

	  default:
		unknownOpcode();
		break;

	}
  }  // end method performSystemCall(int)    

}