package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    DirectiveProcessor.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/
import java.util.Vector;
import java.util.StringTokenizer;

public class DirectiveProcessor {

  public static final int SIZE_BYTE = 1;
  public static final int SIZE_HALF_WORD = 2;
  public static final int SIZE_WORD = 4;
  public static final int SIZE_FLOAT = 4;
  public static final int SIZE_DOUBLE = 8;

  public static final int DEFAULT_TEXT_ADDRESS = 0x00400000;
  public static final int DEFAULT_DATA_ADDRESS = 0x10000000;

  public static final int DEFAULT_KTEXT_ADDRESS = 0x80000080;
  public static final int DEFAULT_KDATA_ADDRESS = 0x90000000;


  public static boolean bOldWarnAboutAT;
  public static boolean bWarnAboutAT = true;
  // bWarnAboutAT is TRUE if the assembler should complain
  //   about the use of $at in an instruction.  This can be manually
  //   set to false by using ".set noat", or it is automatically
  //   forced to false when a pseudo instruction is being processed.

  public int iLastTextAddress;
  public int iLastDataAddress;

  public int iLastKTextAddress;
  public int iLastKDataAddress;

  public int iCurrentAddress;

  private Assembler assembler;

  public DirectiveProcessor(Assembler assembler) {

	this.assembler = assembler;

	iLastTextAddress = DEFAULT_TEXT_ADDRESS;
	iLastDataAddress = DEFAULT_DATA_ADDRESS;

	iLastKTextAddress = DEFAULT_KTEXT_ADDRESS;
	iLastKDataAddress = DEFAULT_KDATA_ADDRESS;

  }  // end constructor DirectiveProcessor(Assembler)    

  public int iProcessDirective(String sInput, int iCurrLine) {
  /*
  RETURNS
	0 = SUCCESS
   -1 = Mode is not data when using a data directive
   -2 = Invalid special character token used in ascii directive
   -3 = Invalid or unrecongized directive specified
  */

	String sDirective = new StringTokenizer(sInput).nextToken();
	// Let sDirective be the first token in the string, which should
	//   be the directive identifier (e.g. ".word", ".text").

	String[] saParam = Assembler.saGetParams(sInput.substring(sDirective.length()).trim());
	// Let saParam be a string array containing the directive
	//   parameters (e.g. "x1,x2,x3" => [x1, x2, x3])

	sDirective = sDirective.substring(1).toUpperCase();
	// We know directives start with ".", so we can trim the
	//   first character off.  And for easier string matching,
	//   force the directive string to uppercase.

	int i;  // Used in parameter processing for most directives.

	if (sDirective.equals("ALIGN")) {
	  // ***INCOMPLETE

	} else 
	if (sDirective.equals("ASCII") | sDirective.equals("ASCIIZ")) {
	  // "str1", ..., "strN"

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  boolean bAppendNull = (sDirective.equals("ASCIIZ"));

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";

	  for (i = 0; i < saParam.length; i++) {
		// Process each string parameter, saParam[i].

		// Convert the string paramter "xy...z" into XX YY ... ZZ
		//   Where XX is the 2-digit hex ASCII code for the first
		//   character, YY is the 2-digit hex ASCII for the second
		//   character, ..., ZZ is the 2-digit hex ASCII for the
		//   last character.
		String s = saParam[i];
		boolean bSpecialCode = false;
		for (int x = 1; x < s.length()-1; x++) {
		  char c = s.charAt(x);
		  if (bSpecialCode) {
			// The last character was a special code token, so
			//   this character is used to determine what special
			//   code was specified.
			switch (c) {
			  case '\\':
				c = '\\';
				break;
			  case 'n':
				c = '\n';
				break;
			  case 't':
				c = '\t';
				break;
			  case 'r':
				c = '\r';
				break;
			  case '0':
				c = '\0';
				break;
			  case '"':
				c = '"';
			  default:
				return -2;
			}
			bSpecialCode = false;
		  } else if (c == '\\') {
			// This character marks the beginning of a special character
			//   token, which is defined by the next character.
			bSpecialCode = true;
			continue;
		  }
		  dataString += Utility.sAsHexPadded( (int)c,2 ) + " ";
		  iCurrentAddress += SIZE_BYTE;
		}  // end string processing

		if (bAppendNull) {
		  // Append null-terminater character after each string.
		  dataString += Utility.sAsHexPadded( (int)'\0',2 ) + " ";
		  iCurrentAddress += SIZE_BYTE;
		}

		// ***INCOMPLETE: We should probably add padding so that
		//   the next reference to iCurrentAddress is on an
		//   aligned address.  This applies to all of the data
		//   directives.

	  } // end parameter processing

	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else
	if (sDirective.equals("BYTE")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";
	  for (i = 0; i < saParam.length; i++) {
		byte b = (byte)Utility.IntegerLiteralToLong(saParam[i]);
		dataString += Utility.sAsHexPadded(b,2) + " ";
		iCurrentAddress += SIZE_BYTE;
	  }
	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else
	if (sDirective.equals("DATA")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_TEXT:
		  iLastTextAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KDATA:
		  iLastKDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KTEXT:
		  iLastKTextAddress = iCurrentAddress;
		  break;
	  }
	  assembler.iCurrentMode = Assembler.MODE_DATA;

	  if (saParam.length == 0) {
		iCurrentAddress = iLastDataAddress;
	  } else {
		iCurrentAddress = (int)Utility.IntegerLiteralToLong(saParam[0]);
	  }

	  if (assembler.vDataSection == null) {
		assembler.vDataSection = new Vector();
	  }
	  assembler.currentDataSection = new DataSection(iCurrentAddress, DataSection.TYPE_DATA);
	  assembler.vDataSection.add(assembler.currentDataSection);
	  assembler.iCurrentDataSectionIndex++;
	  assembler.iCurrentDataIndex = 0;

	} else
	if (sDirective.equals("DOUBLE")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";
	  for (i = 0; i < saParam.length; i++) {
		// LO is the lower 32 bits of the double, stored in EVEN regs
		// HI is the upper 32 bits of the double, stored in ODD regs
		// e.g. a single is stored at $f0 - $f31
		// e.g. a double is stored at $f3 and $f4,
		//      $f3 has the upper 32 bits, $f4 has the lower 32-bits
		double d = Utility.FloatingPointLiteralToDouble(saParam[i]);
		long l = Double.doubleToLongBits(d);
		int iHI = (int)((l & 0xFFFFFFFF00000000L) >>> 32);
		int iLO = (int)(l &  0x00000000FFFFFFFFL);
		dataString += Utility.sAsHexPadded(iLO,8);
		dataString += Utility.sAsHexPadded(iHI,8);
		iCurrentAddress += SIZE_DOUBLE;
	  }
	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else
//    if (sDirective.equals("EXTERN")) {
	  // ***INCOMPLETE
//    } else
	if (sDirective.equals("FLOAT")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";
	  for (i = 0; i < saParam.length; i++) {
		float f = Utility.FloatingPointLiteralToFloat(saParam[i]);
		dataString += Utility.sAsHexPadded(f,8) + " ";
		iCurrentAddress += SIZE_FLOAT;
	  }
	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else
	if (sDirective.equals("GLOBL")) {
	  // ***INCOMPLETE
	} else
	if (sDirective.equals("HALF")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";
	  for (i = 0; i < saParam.length; i++) {
		short x = (short)Utility.IntegerLiteralToLong(saParam[i]);
		dataString += Utility.sAsHexPadded((int)x,4) + " ";
		iCurrentAddress += SIZE_WORD;
	  }
	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else
	if (sDirective.equals("KDATA")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_TEXT:
		  iLastTextAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_DATA:
		  iLastDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KTEXT:
		  iLastKTextAddress = iCurrentAddress;
		  break;
	  }
	  assembler.iCurrentMode = Assembler.MODE_KDATA;

	  if (saParam.length == 0) {
		iCurrentAddress = iLastKDataAddress;
	  } else {
		iCurrentAddress = (int)Utility.IntegerLiteralToLong(saParam[0]);
	  }

	  if (assembler.vDataSection == null) {
		assembler.vDataSection = new Vector();
	  }
	  assembler.currentDataSection = new DataSection(iCurrentAddress, DataSection.TYPE_KDATA);
	  assembler.vDataSection.add(assembler.currentDataSection);
	  assembler.iCurrentDataSectionIndex++;
	  assembler.iCurrentDataIndex = 0;

	} else
	if (sDirective.equals("KTEXT")) {
	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_DATA:
		  iLastDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KDATA:
		  iLastKDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_TEXT:
		  iLastTextAddress = iCurrentAddress;
		  break;
	  }
	  assembler.iCurrentMode = Assembler.MODE_KTEXT;

	  if (saParam.length == 0) {
		iCurrentAddress = iLastKTextAddress;
	  } else {
		iCurrentAddress = (int)Utility.IntegerLiteralToLong(saParam[0]);
	  }

	  if (assembler.vTextSection == null) {
		assembler.vTextSection = new Vector();
	  }
	  assembler.currentTextSection = new TextSection(iCurrentAddress, TextSection.TYPE_KTEXT);
	  assembler.vTextSection.add(assembler.currentTextSection);
	  assembler.iCurrentTextSectionIndex++;
	  assembler.iCurrentTextIndex = 0;

	} else 
	if (sDirective.equals("SET")) {
	  if (saParam[0].toUpperCase().equals("NOAT")) {
		bWarnAboutAT = false;
	  } else if (saParam[0].toUpperCase().equals("AT")) {
		bWarnAboutAT = true;
	  } else {
		System.err.println("WARNING: Invalid parameter for SET directive (L" + iCurrLine + ")");
	  }
	} else
	if (sDirective.equals("SPACE")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);

	  String dataString = "";
	  int n = (int)Utility.IntegerLiteralToLong(saParam[0]);
	  for (i = 0; i < n; i++) {
		dataString += "00 ";
		iCurrentAddress += SIZE_BYTE;
	  }

	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else 
	if (sDirective.equals("TEXT")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_DATA:
		  iLastDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KDATA:
		  iLastKDataAddress = iCurrentAddress;
		  break;
		case Assembler.MODE_KTEXT:
		  iLastKTextAddress = iCurrentAddress;
		  break;
	  }
	  assembler.iCurrentMode = Assembler.MODE_TEXT;

	  if (saParam.length == 0) {
		iCurrentAddress = iLastTextAddress;
	  } else {
		iCurrentAddress = (int)Utility.IntegerLiteralToLong(saParam[0]);
	  }

	  if (assembler.vTextSection == null) {
		assembler.vTextSection = new Vector();
	  }
	  assembler.currentTextSection = new TextSection(iCurrentAddress, TextSection.TYPE_TEXT);
	  assembler.vTextSection.add(assembler.currentTextSection);
	  assembler.iCurrentTextSectionIndex++;
	  assembler.iCurrentTextIndex = 0;

	} else
	if (sDirective.equals("WORD")) {

	  switch (assembler.iCurrentMode) {
		case Assembler.MODE_KDATA:
		case Assembler.MODE_DATA:
		  break;
		default:
		  return -1;
	  }

	  DataEntry dataEntry = new DataEntry(iCurrentAddress);
	  String dataString = "";
	  for (i = 0; i < saParam.length; i++) {
		try {
		  int x = (int)Utility.IntegerLiteralToLong(saParam[i]);
		  dataString += Utility.sAsHexPadded(x,8) + " ";
		} catch (NumberFormatException e) {
		  dataString += "addr("+saParam[i]+") ";
		}

		iCurrentAddress += SIZE_WORD;
	  }
	  dataEntry.sData = dataString.trim();
	  assembler.addToDataVector(dataEntry);

	} else {
	  return -3;
	}

	return 0;

  }    

}