package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandSet.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: SET ( M* (range|single) value ) | ( R* (GPR|CP0|CP1) (range | x) value )

InvalidCommandUseException Messages:
  Invalid range or value specified.
  Invalid memory value specified.
  Memory value must be 8-bit number.
  Invalid register set specified in SET command.
  Invalid range or value or incorrect use of SET command.
  Invalid use of SET command.

Command Buffer Results:
  * SET DONE

*/
import java.util.StringTokenizer;

/**
  * This is the CommandSet class.
  *
  * @author Steve Lewis
*/
public class CommandSet extends Command {

  private Simulator simulator;

  public CommandSet(StringTokenizer st, Simulator simulator) {
	super(st);
	this.simulator = simulator;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	State stateSim = simulator.getState();

	String s = nextToken();  // s == which area to set (memory or registers)

	if (s.startsWith("M")) {  // MEMORY keyword

	  s = nextToken();  // s == memory range or specified address to be set
	  int[] ia = iaGetMemoryRangeOrValue(s);

	  if (ia == null)
		throw new InvalidCommandUseException("Invalid range or value specified.");

	  s = nextToken();  // s == value to set memory to
	  int value = 0;
	  try {
		value = iGetNumber(s);
	  } catch (NumberFormatException e) {
		throw new InvalidCommandUseException("Invalid memory value specified.");
	  }

	  if ( (value < -128) || (value > 255) )
		throw new InvalidCommandUseException("Memory value must be 8-bit number.");

	  for (int i = ia[0]; i <= ia[1]; i++)
		stateSim.storeByte(i, (byte)value);

	} else if (s.startsWith("R")) {  // REGISTER keyword

	  s = nextToken();  // s == which set of registers to be set

	  String sWhichReg = nextToken();
	  // sWhichReg == range of or specific reg to be set in the selected register set

	  String sValue = nextToken();
	  // sValue == desired value to set register(s) to

	  try {

		int[] ia = null;
		if (s.equals("GPR")) {

		  ia = iaGetRegisterRangeOrValue(sWhichReg, 0);
		  int iValue = 0;
		  iValue = iGetNumber(sValue);
		  for (int i = ia[0]; i <= ia[1]; i++)
			stateSim.setRegister(i, iValue);

		}
		else if (s.equals("CP0")) {

		  ia = iaGetRegisterRangeOrValue(sWhichReg, 1);
		  int iValue = 0;
		  iValue = iGetNumber(sValue);
		  for (int i = ia[0]; i <= ia[1]; i++)
			stateSim.setCP0Register(i, iValue);

		}
		else if (s.equals("CP1")) {

		  ia = iaGetRegisterRangeOrValue(sWhichReg, 2);
		  int iValue = 0;
		  iValue = iGetNumber(sValue);
		  for (int i = ia[0]; i <= ia[1]; i++)
			stateSim.setCP1Register(i, iValue);

		}
		else {
		  throw new InvalidCommandUseException("Invalid register set specified in SET command.");
		}

	  } catch (Exception e) {
		throw new InvalidCommandUseException("Invalid range or value or incorrect use of SET command.");
	  }

	} else {
	  throw new InvalidCommandUseException("Invalid use of SET command.");
	}

	crb.append(RESPONSE + " SET " + DONE);
  }    

}