package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandRun.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: RUN

InvalidCommandUseException Messages:
  <none>

Command Buffer Results:
  * RUN ERR ERR_INVALID_RESPONSE
  * RUN ERR ERR_NOT_STEP_RESPONSE
  * RUN BREAKPOINT index
  * RUN GUARD index
  * RUN EXCEPTION x
  * RUN ERR ERR_STEP_ERROR stepResponse
  * RUN DONE

*/
import java.util.StringTokenizer;

/**
  * This is the CommandRun class.
  *
  * @author Steve Lewis
*/
public class CommandRun extends Command {

  public static final String ERR_INVALID_RESPONSE = "ERR_INVALID_RESPONSE";
  public static final String ERR_NOT_STEP_RESPONSE = "ERR_NOT_STEP_RESPONSE";
  public static final String ERR_STEP_ERROR = "ERR_STEP_ERROR";

  private CommandProcessor commandProcessor;

  public CommandRun(CommandProcessor commandProcessor) {
	super();
	this.commandProcessor = commandProcessor;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	boolean bRunning = true;

	do {

	  // Execute a STEP 1 command
	  CommandResultBuffer crbStep = commandProcessor.processCommand("STEP " + Long.MAX_VALUE);

	  // Process the response from the step,
	  //   to see if we can stop running yet...

	  String sStepResponse = crbStep.sGetNextResponse();

	  StringTokenizer st = new StringTokenizer(sStepResponse);

	  String s = st.nextToken();
	  if (!s.equals(RESPONSE)) {
		crb.append(RESPONSE + " RUN ERR " + ERR_INVALID_RESPONSE);
		break;
	  }

	  s = st.nextToken();
	  if (!s.equals("STEP")) {
		crb.append(RESPONSE + " RUN ERR " + ERR_NOT_STEP_RESPONSE);
		break;
	  }

	  s = st.nextToken();

	  if (s.equals("BREAKPOINT")) {
		// we encountered a breakpoint, stop running
		s = st.nextToken();  // breakpoint index
		crb.append(RESPONSE + " RUN BREAKPOINT " + s);
		bRunning = false;
	  } else if (s.equals("GUARD")) {
		// we encountered a satisfied guard, stop running
		s = st.nextToken();  // guard index
		crb.append(RESPONSE + " RUN GUARD " + s);
		bRunning = false;
	  } else if (s.equals("EXCEPTION")) {
		// an exception triggers
		s = st.nextToken();
		crb.append(RESPONSE + " RUN EXCEPTION " + s);
		bRunning = false;
	  } else if (s.equals("ERR")) {
		// Some error occured
		crb.append(RESPONSE + " RUN ERR " + ERR_STEP_ERROR + " " + sStepResponse);
		bRunning = false;
	  } else {
		// We should check if we have entered an infinite loop.
		//   For now, we leave that responsibility to the user.
	  }

	} while (bRunning);

	crb.append(RESPONSE + " RUN " + DONE);
	
  } // end method execute    

}