package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandAdd.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: ADD ( (B* address) | (G* expression) )

InvalidCommandUseException Messages:
  Invalid address specified for breakpoint.
  Unable to initialize breakpoint list.
  Unable to add breakpoint.
  Breakpoint already exists at that address.
  Invalid guard expression.
  Unable to add guard expression.
  Incorrect use of ADD command.
Command Buffer Results:
  * ADD DONE

*/
import java.util.StringTokenizer;

/**
  * This is the CommandAdd class.
  *
  * @author Steve Lewis
*/
public class CommandAdd extends Command {

  private Simulator simulator;

  public CommandAdd(StringTokenizer st, Simulator simulator) {
	super(st);
	this.simulator = simulator;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	String s = nextToken();

	if (s.startsWith("B")) {  // BREAKPOINT keyword

	  s = nextToken();  // s == should equal the breakpoint address

	  int iAddress = 0;
	  try {
		iAddress = iGetNumber(s);
	  } catch (NumberFormatException e) {
		throw new InvalidCommandUseException("Invalid address specified for breakpoint.");
	  }

	  BreakpointList breakpointList = simulator.getBreakpointList();

	  int i = 1;
	  try {
		i = breakpointList.iAddBreakpoint(iAddress);
	  } catch (Exception e) {

	  }

	  switch (i) {
		case 1:
		  throw new InvalidCommandUseException("Unable to initialize breakpoint list.");

		case BreakpointList.ERR_UNKNOWN:
		  throw new InvalidCommandUseException("Unable to add breakpoint.");

		case BreakpointList.ERR_DUPLICATE_BREAKPOINT:
		  throw new InvalidCommandUseException("Breakpoint already exists at that address.");
	  }

	} else if (s.startsWith("G")) {  // GUARD keyword

	  s = nextToken();  // s == should equal the guard expression

	  GuardExpression g = null;
	  try {
		g = GuardExpression.translateGuardExpression(s);
	  } catch (Exception e) {

	  }

	  if (g == null)
		throw new InvalidCommandUseException("Invalid guard expression.");

	  GuardList guardList = simulator.getGuardList();

	  try {
		guardList.addGuard(g);
	  } catch (Exception e) {
		throw new InvalidCommandUseException("Unable to add guard expression.");
	  }

	} else {

	  throw new InvalidCommandUseException("Incorrect use of ADD command.");

	}

	crb.append(RESPONSE + " ADD " + DONE);

  }  // end method execute()    

}