package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    BreakpointList.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/

/**
  * This is the BreakpointList class.
  *
  * @author Steve Lewis
*/
public class BreakpointList {

  public static final int SUCCESS = 0;
  public static final int ERR_DUPLICATE_BREAKPOINT = -1;
  public static final int ERR_UNKNOWN = -2;
  public static final int ERR_INVALID_INDEX = -3;

  private int[] iaBreakpoint;
  private int iArrayTop;  // Represents number of breakpoints-1
  private int iArrayCapacity;  // -1 if list is empty

  public BreakpointList() {
	iArrayCapacity = 2;
	iaBreakpoint = new int[iArrayCapacity];
	iArrayTop = -1;
  }    

  private int iNumBreakpoints() {
	return (iArrayTop+1);
  }    

  private void makeBreakpointArrayLarger() {
	// By default, double the size of the breakpoint array.
	makeBreakpointArrayLarger(2);
  }    

  private void makeBreakpointArrayLarger(int iFactor) {
	// Expand the capacity of breakpoint array by iFactor
	// NOTE: This could cause an OutOfMemoryException!
	iArrayCapacity = iArrayCapacity * iFactor;
	int[] temp = new int[iArrayCapacity];
	int n = iNumBreakpoints();
	for (int i = 0; i < n; i++)
	  temp[i] = iaBreakpoint[i];
	iaBreakpoint = temp;
  }  // end method makeBreakpointArrayLarger()    

  public boolean isBreakpoint(int iAddress) {
	// Return TRUE if argument iAddress is a previosly specified breakpoint
	// Otherwise return FALSE.
	int n = iNumBreakpoints();
	for (int i = 0; i < n; i++) {
	  if (iaBreakpoint[i] == iAddress)
		return true;
	}
	return false;
  }  // end method isBreakpoint(int)    

  public int iRemoveBreakpoint(int iIndex) {
	// Remove the breakpoint at position iIndex of the
	//   breakpoint integer array.
	// Returns either:
	//   SUCCESS
	//   ERR_INVALID_INDEX

	if ((iIndex >= 0) && (iIndex <= iArrayTop)) {
	  if (iIndex < iArrayTop) {
		// Since we are deleting a breakpoint somewhere inside
		//   the array, we need to move all the later elements
		//   of the array down one slot.
		for (int i = iIndex+1; i <= iArrayTop; i++) {
		  iaBreakpoint[i-1] = iaBreakpoint[i];
		}
	  }
	  iArrayTop--;
	  return SUCCESS;  // success
	}
	return ERR_INVALID_INDEX;  // invalid index
  }  // end method iRemoveBreakpoint(int)    

  public int iAddBreakpoint(int iAddress) {

	// We want to add the breakpoint iAddress to the
	//   breakpoint list.
	// Returns either:
	//   SUCCESS
	//   ERR_DUPLICATE_BREAKPOINT
	//   ERR_UNKNOWN

	// See of the breakpoint has already been specified
	if (isBreakpoint(iAddress))
	  return ERR_DUPLICATE_BREAKPOINT;  // Breakpoint already specified

	int n = iNumBreakpoints();

	if (iArrayTop+1 == iArrayCapacity) {
	  // We need to increase the capacity of the breaklist array
	  try {
		makeBreakpointArrayLarger();
	  } catch (Exception e) {
		return ERR_UNKNOWN;  // Out of memory?
	  }
	}

	// Add the new breakpoint to the array top+1
	iArrayTop++;
	iaBreakpoint[iArrayTop] = iAddress;

	return SUCCESS;

  }  // end method iAddBreakpoint(int)    

  public int[] iaGetBreakpoints() {
	// Return a new copy of the breakpoint array.
	int[] temp = null;
	int n = iNumBreakpoints();
	if (n > 0) {
	  temp = new int[n];
	  for (int i = 0; i < n; i++)
		temp[i] = iaBreakpoint[i];
	}
	return temp;
  }  // end method iaGetBreakpoints()    

}