package JIMSwingGUI;

import java.applet.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.plaf.metal.MetalLookAndFeel;
import JIMSCore.*;

// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * A class representing the main window container for the application.
  *
  * @author Fred Williams
*/
public class TopLevelWindow extends JFrame implements ActionListener, WindowListener, ComponentListener, InternalFrameListener
{
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Constructs a new application window.
  *
*/
	public TopLevelWindow()
	{
		Splashscreen ss = new Splashscreen();
		shouldRun = false;
		fileLoaded = false;

		menuBar = new JMenuBar();
		setUpMenus();
		toolBar = new JToolBar();
		setUpToolBar();

		contentPane = new JPanel();
		contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));
		topInsets = getInsets();
		setContentPane(contentPane);

		setJMenuBar(menuBar);
		contentPane.add(toolBar);
		pack();

		xCorrection = topInsets.left + topInsets.right;
		yCorrection = topInsets.top + topInsets.bottom + menuBar.getHeight() + toolBar.getHeight();

		loadConfig();

		setUpMenuItems();

		desktopPane = new JDesktopPane();
		desktopPane.setBackground(Color.getColor("gray"));
		desktopPane.setLayout(null);
		desktopPane.setBorder(BorderFactory.createLineBorder(Color.getColor("black")));
		desktopInsets = desktopPane.getInsets();

		contentPane.add(desktopPane);

		registerWindow = new JInternalFrame("Registers", true, false, true, true);
		registerWindow.setBounds(registersX, registersY, registersWidth, registersHeight);
		registerList = new JList();
		registerList.setSelectionModel(new SourceListSelectionModel());
		registerList.setCellRenderer(new SourceListCellRenderer());
		registerPane = new JScrollPane(registerList, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		registerPane.setSize(registerWindow.getContentPane().getWidth(), registerWindow.getContentPane().getHeight());
		registerPane.getViewport().setScrollMode(JViewport.SIMPLE_SCROLL_MODE);
		registerWindow.getContentPane().add(registerPane);
		registerWindow.addComponentListener(this);
		registerWindow.addInternalFrameListener(this);

		memoryWindow = new JInternalFrame("Memory", true, false, true, true);
		memoryWindow.setBounds(memoryX, memoryY, memoryWidth, memoryHeight);
		memoryList = new JList();
		memoryList.setSelectionModel(new SourceListSelectionModel());
		memoryList.setCellRenderer(new SourceListCellRenderer());
		memoryPane = new JScrollPane(memoryList, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		memoryPane.getViewport().setScrollMode(JViewport.SIMPLE_SCROLL_MODE);
		memoryWindow.getContentPane().add(memoryPane);
		memoryWindow.addComponentListener(this);
		memoryWindow.addInternalFrameListener(this);

		sourceWindow = new JInternalFrame("Source", true, false, true, true);
		sourceWindow.setBounds(sourceX, sourceY, sourceWidth, sourceHeight);
		sourceList = new JList();
		sourceList.setSelectionModel(new SourceListSelectionModel());
		sourceList.setCellRenderer(new SourceListCellRenderer());
		sourcePane = new JScrollPane(sourceList, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		sourcePane.setSize(sourceWindow.getContentPane().getWidth(), sourceWindow.getContentPane().getHeight());
		sourcePane.getViewport().setScrollMode(JViewport.SIMPLE_SCROLL_MODE);
		sourceWindow.getContentPane().add(sourcePane);
		sourceWindow.addComponentListener(this);
		sourceWindow.addInternalFrameListener(this);

		consoleWindow = new JInternalFrame("Console", true, false, true, true);
		consoleWindow.setBounds(consoleX, consoleY, consoleWidth, consoleHeight);
		consoleText = new JTextArea();
		consoleText.setEditable(false);
		consoleText.setFont(new Font("monospaced", 0, 12));
		consolePane = new JScrollPane(consoleText, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		consolePane.setSize(consoleWindow.getContentPane().getWidth(), consoleWindow.getContentPane().getHeight());
		consolePane.getViewport().setScrollMode(JViewport.SIMPLE_SCROLL_MODE);
		consoleWindow.getContentPane().add(consolePane);
		consoleWindow.addComponentListener(this);
		consoleWindow.addInternalFrameListener(this);

		desktopPane.add(registerWindow);
		desktopPane.add(memoryWindow);
		desktopPane.add(sourceWindow);
		desktopPane.add(consoleWindow);
		try
		{
			registerWindow.setIcon(registerWindowIsIcon);
			memoryWindow.setIcon(memoryWindowIsIcon);
			sourceWindow.setIcon(sourceWindowIsIcon);
			consoleWindow.setIcon(consoleWindowIsIcon);
		}
		catch (java.beans.PropertyVetoException pve)
		{
			System.err.println("Error setting iconified/deiconified state of internal frames in TopLevelWindow constructor.");
		}

		registerWindow.setVisible(true);
		memoryWindow.setVisible(true);
		sourceWindow.setVisible(true);
		consoleWindow.setVisible(true);

		performReset();

		setLocation(applicationX, applicationY);
		setSize(applicationWidth, applicationHeight);
		setResizable(true);
		setTitle("JIMSwing Release 1.0");
		ss.dispose();
		setVisible(true);
		try
		{
			registerWindow.setSelected(true);
			memoryWindow.setSelected(false);
			sourceWindow.setSelected(false);
			consoleWindow.setSelected(false);
		}
		catch (java.beans.PropertyVetoException pve)
		{
			System.err.println("Error setting selected internal frame in TopLevelWindow constructor.");
		}
		desktopPaneWidth = desktopPane.getWidth();
		desktopPaneHeight = desktopPane.getHeight();

		addWindowListener(this);
		addComponentListener(this);
	}

// ===================================================================================
// METHOD:		performBackStep
// FUNCTION:	Executes the next instruction
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Steps backward by one instruction.
  * 
  * @param updateDisplay Defines whether the display should be updated after the step.
*/
	private void performBackStep(boolean updateDisplay)
	{
		commandResultBuffer = commandProcessor.processCommand("UNDO 1");
		String s = commandResultBuffer.sGetNextResponse();
		if (s.startsWith("* UNDO OUTPUT"))
		{
			commandResultBuffer = commandProcessor.processCommand("OUTPUT");
			s = commandResultBuffer.sGetNextResponse();
			if (s.equals("* OUTPUT OK"))
			{
				CommandOutput commandOutput = (CommandOutput)commandProcessor.getLastCommand();
				StringBuffer output = commandOutput.sbGetOutput();
				consoleText.append(output.toString());
			}
			validate();
		}
		if (s.startsWith("* UNDO EXCEPTION"))
		{
			JOptionPane.showMessageDialog(this, "Error encountered.", "Error", JOptionPane.ERROR_MESSAGE);
			shouldRun = false;
			updateDisplay = true;
		}
		if (s.startsWith("* UNDO ERR"))
		{
			JOptionPane.showMessageDialog(this, "Error encountered.", "Error", JOptionPane.ERROR_MESSAGE);
			shouldRun = false;
			updateDisplay = true;
		}
		updateConsole();
		if (updateDisplay)
		{
			updateMemory();
			updateRegisters();
			highlightSourceLine(getRegisterValue("GPR", "$PC"));
		}
		validate();
	}

// ===================================================================================
// METHOD:		performForwardStep
// FUNCTION:	Executes the next instruction
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Steps forward by one instruction.
  * 
  * @param updateDisplay Defines whether the display should be updated after the step.
*/
	private void performForwardStep(boolean updateDisplay)
	{
		commandResultBuffer = commandProcessor.processCommand("STEP 1");
		String s = commandResultBuffer.sGetNextResponse();
		if (s.startsWith("* STEP EXCEPTION 4"))
		{
			commandResultBuffer = commandProcessor.processCommand("OUTPUT");
			s = commandResultBuffer.sGetNextResponse();
			if (s.equals("* OUTPUT OK"))
			{
				CommandOutput commandOutput = (CommandOutput)commandProcessor.getLastCommand();
				StringBuffer output = commandOutput.sbGetOutput();
				consoleText.append(output.toString());
			}
			validate();
		}
		if (s.startsWith("* STEP EXCEPTION 5"))
		{
			shouldRun = false;
			highlightSourceLine(getRegisterValue("GPR", "$PC"));
			validate();
			InputDialog newDialog = new InputDialog(this, commandProcessor);
			s = newDialog.getReturnValue();
			if (s != null)
			{
				if (s.equals("* INPUT DONE"))
				{
					consoleText.append(newDialog.getText() + "\n");
					shouldRun = true;
				}
			}
		}
		if (s.startsWith("* STEP EXCEPTION 6"))
		{
			shouldRun = false;
			updateDisplay = true;
			performBackStep(true);
			JOptionPane.showMessageDialog(this, "Exit system call encountered.", "System call exit", JOptionPane.INFORMATION_MESSAGE);
		}
		if (s.startsWith("* STEP BREAKPOINT"))
		{
			shouldRun = false;
			updateDisplay = true;
			JOptionPane.showMessageDialog(this, "Breakpoint encountered.", "Breakpoint", JOptionPane.INFORMATION_MESSAGE);
		}
		if (s.startsWith("* STEP ERR"))
		{
			shouldRun = false;
			updateDisplay = true;
			JOptionPane.showMessageDialog(this, s, "Error", JOptionPane.ERROR_MESSAGE);
		}
		if (s.startsWith("* STEP GUARD"))
		{
			shouldRun = false;
			updateDisplay = true;
			JOptionPane.showMessageDialog(this, "Guard expression encountered.", "Guard", JOptionPane.INFORMATION_MESSAGE);
		}
		updateConsole();
		if (updateDisplay)
		{
			updateMemory();
			updateRegisters();
			highlightSourceLine(getRegisterValue("GPR", "$PC"));
		}
		validate();
	}

// ===================================================================================
// METHOD:		actionPerformed
// FUNCTION:	handles ActionEvent events
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Handles ActionEvents generated by menu items and buttons.
  * 
  * @param ae The ActionEvent object generated by a menu item or button.
*/
	public void actionPerformed(ActionEvent ae)
	{
		Object source = ae.getSource();
		if (source == exit)
		{
			performExit();
		}
		if (source == resetAllWindows)
		{
			int confirm = JOptionPane.showConfirmDialog(this, "Reset all windows?", "Confirm", JOptionPane.YES_NO_OPTION);
			if (confirm == JOptionPane.YES_OPTION)
			{
				restoreWindows();
			}
		}
		if ((source == settings) || (source == settingsButton))
		{
			SettingsDialog sd = new SettingsDialog(this, GPRHex, SFPHex, DFPHex, ignoreInvalidAT);
		}
		if (source == showRegisterWindow)
		{
			try
			{
				registerWindow.setIcon(!(showRegisterWindow.isSelected()));
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.err.println("Error iconifying/deiconifying window in TopLevelWindow.actionPerformed.");
			}
		}
		if (source == showMemoryWindow)
		{
			try
			{
				memoryWindow.setIcon(!(showMemoryWindow.isSelected()));
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.err.println("Error iconifying/deiconifying window in TopLevelWindow.actionPerformed.");
			}
		}
		if (source == showSourceWindow)
		{
			try
			{
				sourceWindow.setIcon(!(showSourceWindow.isSelected()));
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.err.println("Error iconifying/deiconifying window in TopLevelWindow.actionPerformed.");
			}
		}
		if (source == showConsoleWindow)
		{
			try
			{
				consoleWindow.setIcon(!(showConsoleWindow.isSelected()));
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.err.println("Error iconifying/deiconifying window in TopLevelWindow.actionPerformed.");
			}
		}
		if (source == about)
		{
			JOptionPane.showMessageDialog
			(this,
				"JIMSwing Release 1.0\n" +
				"Fred Williams, 2001\n" + "University of Florida\n\n" +
				"An extension of JIMS Release 1\n" +
				"Steve Lewis, 2001\n" + "University of Florida\n",
				"About JIMSwing", JOptionPane.INFORMATION_MESSAGE
			);
		}
		if (source == helpContents)
		{
			HelpWindow hw = new HelpWindow(this);
		}
		if (source == commandConsole)
		{
			CommandConsoleWindow ccw = new CommandConsoleWindow(this, commandProcessor);
		}
		if (source == status)
		{
			commandResultBuffer = commandProcessor.processCommand("STATUS");
			String s = new String();
			StringBuffer statusText = new StringBuffer();
			while (!(s = commandResultBuffer.sGetNextResponse()).equals("* STATUS DONE"))
			{
				statusText.append(s + "\n");
			}
			JOptionPane.showMessageDialog(this, statusText.toString(), "Simulator status", JOptionPane.INFORMATION_MESSAGE);
		}
		if (source == reset)
		{
			if (performReset())
			{
				JOptionPane.showMessageDialog(this, "Simulator reset.", "Reset status", JOptionPane.INFORMATION_MESSAGE);
			}
		}
		if ((source == runF) || (source == goFButton))
		{
			if (!(shouldRun))
			{
				shouldRun = true;
				simExecutionThread = new runThread("JIMSwing running", true);
				simExecutionThread.start();
			}
		}
		if ((source == runB) || (source == goBButton))
		{
			if (!(shouldRun))
			{
				shouldRun = true;
				simExecutionThread = new runThread("JIMSwing running", false);
				simExecutionThread.start();
			}
		}
		if ((source == stop) || (source == stopButton))
		{
			if (shouldRun)
			{
				shouldRun = false;
				updateMemory();
				updateRegisters();
				highlightSourceLine(getRegisterValue("GPR", "$PC"));		
			}
			else
			{
				JOptionPane.showMessageDialog(this, "Simulator not running.", "Error", JOptionPane.ERROR_MESSAGE);					
			}
		}
		if ((source == multipleStepF) || (source == multiStepForwardButton))
		{
			if (!(shouldRun))
			{
				MultistepForwardDialog newDialog = new MultistepForwardDialog(this, commandProcessor);
				validate();
				int response = newDialog.getReturnValue();
				int current = 0;
				if (response != 0)
				{
					while (current < response)
					{
						performForwardStep(false);
						current++;
					}
					performForwardStep(true);
				}
			}
		}
		if ((source == multipleStepB) || (source == multiStepBackButton))
		{
			if (!(shouldRun))
			{
				MultistepBackwardDialog newDialog = new MultistepBackwardDialog(this, commandProcessor);
				validate();
				int response = newDialog.getReturnValue();
				int current = 0;
				if (response != 0)
				{
					while (current < response)
					{
						performBackStep(false);	
						current++;
					}
					performBackStep(true);
				}
			}
		}
		if ((source == singleStepF) || (source == stepForwardButton)) 
		{
			if (!(shouldRun))
			{
				performForwardStep(true);
			}
		}
		if ((source == singleStepB) || (source == stepBackButton))
		{
			if (!(shouldRun))
			{
				performBackStep(true);
			}
		}
		if ((source == breakpoints))
		{
			BreakpointEditDialog newDialog = new BreakpointEditDialog(this, commandProcessor);
		}
		if ((source == memoryModify))
		{
			MemoryEditDialog newDialog = new MemoryEditDialog(this, commandProcessor);
		}
		if ((source == registerModify))
		{
			RegisterEditDialog newDialog = new RegisterEditDialog(this, commandProcessor);
		}
		if ((source == open) || (source == openButton))
		{
			final JFileChooser fc;
			if (!(openDir.equals("nonespecified")))
			{
				fc = new JFileChooser(openDir);
			}
			else
			{
				fc = new JFileChooser();
			}
			fc.setFileHidingEnabled(true);
			fc.setMultiSelectionEnabled(false);
			fc.setFileFilter(new SourceFileFilter());
			int returnVal = fc.showOpenDialog(contentPane);
			if (returnVal == JFileChooser.APPROVE_OPTION)
			{
				performReset();
				File file = fc.getSelectedFile();
				openDir = file.getParent();
				String fileName = file.getName();
				int x = fileName.indexOf('.');
				if (x >= 0)
				{
					fileName = fileName.substring(0, x);
				}
				String fileFullPath = new String();
				try
				{
					fileFullPath = file.getCanonicalPath();
				}
				catch (IOException ioe)
				{
					System.err.println("Error getting canonical path for source file in TopLevelWindow.actionPerformed");
					JOptionPane.showMessageDialog(this, "Error opening file.", "Error", JOptionPane.INFORMATION_MESSAGE);
				}

				if (assemblyPhase1(fileFullPath, openDir, fileName, ignoreInvalidAT) > 0)
				{
					performReset();
					return;
				}
				if (assemblyPhase2(openDir + File.separator + fileName + ".modified", openDir, fileName, ignoreInvalidAT) > 0)
				{
					performReset();
					return;
				}

				Vector vCurrentProgramMachineCode = Utility.vLoadFileAsVector(openDir + File.separator + fileName + ".code");
				Enumeration e = vCurrentProgramMachineCode.elements();
				sourceLinks = new Vector();
				while (e.hasMoreElements())
				{
					String s = (String) e.nextElement();
					if (s.startsWith("T "))
					{
						commandResultBuffer = commandProcessor.processCommand("LOAD " + s);
						StringTokenizer st = new StringTokenizer(s);
						st.nextToken();
						String sAddress = st.nextToken();
						st.nextToken();
						st.nextToken();
						st.nextToken();
						st.nextToken();
						String sLine = st.nextToken().substring(1);
						long lAddress = Long.parseLong(sAddress, 16);
						int iLine = Integer.parseInt(sLine);
						sourceLinks.add(new sourceLink(sAddress, iLine));
					}
					if (s.startsWith("D "))
					{
						commandResultBuffer = commandProcessor.processCommand("LOAD " + s);
					}
				}
				vCurrentProgramMachineCode = null;
				Vector vCurrentProgramSourceCode = Utility.vLoadFileAsVector(openDir + File.separator + fileName + ".modified");
				sourceCode = new Vector();
				Enumeration e1 = vCurrentProgramSourceCode.elements();
				int lineCounter = 1;
				int instrCounter = 0;
				while (e1.hasMoreElements())
				{
					StringBuffer nextLine = new StringBuffer();
					if ((instrCounter < sourceLinks.size()) && (lineCounter == ((sourceLink)sourceLinks.elementAt(instrCounter)).line))
					{
							nextLine.append("[0x");
							nextLine.append((String)(((sourceLink)sourceLinks.elementAt(instrCounter)).pc));
							nextLine.append("]");
							nextLine.append((String)e1.nextElement());
							instrCounter++;
					}
					else
					{
						nextLine.append((String)e1.nextElement());
					}
					lineCounter++;
					sourceCode.add(nextLine.toString());
				}
				Vector machineCode = Utility.vLoadFileAsVector(openDir + File.separator + fileName + ".code");
				while (e.hasMoreElements())
				{
					String s = (String) e.nextElement();
					if (s.startsWith("T "))
					{
						commandResultBuffer = commandProcessor.processCommand("LOAD " + s);
					}
				}
				updateRegisters();
				updateMemory();
				sourceList.setListData(sourceCode);
				highlightSourceLine(getRegisterValue("GPR", "$PC"));
				fileLoaded = true;
				validate();
			}
		}
	}

// ===================================================================================
// METHOD:		setUpMenus
// FUNCTION:
// RETURNS:
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Sets up the application menus.
  * 
*/
	private void setUpMenus()
	{
		fileMenu = new JMenu("File");
		fileMenu.setMnemonic(KeyEvent.VK_F);

		simulatorMenu = new JMenu("Simulator");
		simulatorMenu.setMnemonic(KeyEvent.VK_S);

		windowMenu = new JMenu("Window");
		windowMenu.setMnemonic(KeyEvent.VK_W);

		helpMenu = new JMenu("Help");
		helpMenu.setMnemonic(KeyEvent.VK_H);

		menuBar.add(fileMenu);
		menuBar.add(simulatorMenu);
		menuBar.add(windowMenu);
		menuBar.add(helpMenu);
	}

// ===================================================================================
// METHOD:		setUpMenuItems
// FUNCTION:
// RETURNS:
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Sets up the application menu items.
  * 
*/
	public void setUpMenuItems()
	{
		open = new JMenuItem("Open");
		open.setMnemonic(KeyEvent.VK_O);
		open.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.CTRL_MASK));
		open.addActionListener(this);
		
		exit = new JMenuItem("Exit");
		exit.setMnemonic(KeyEvent.VK_X);
		exit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, ActionEvent.ALT_MASK));
		exit.addActionListener(this);

		fileMenu.add(open);
		fileMenu.addSeparator();
		fileMenu.add(exit);

		singleStepB = new JMenuItem("Step backward");
		singleStepB.setMnemonic(KeyEvent.VK_S);
		singleStepB.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F6, 0));
		singleStepB.addActionListener(this);

		multipleStepB = new JMenuItem("Multiple step backward");
		multipleStepB.setMnemonic(KeyEvent.VK_M);
		multipleStepB.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F5, 0));
		multipleStepB.addActionListener(this);

		runB = new JMenuItem("Run backward");
		runB.setMnemonic(KeyEvent.VK_B);
		runB.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F3, 0));
		runB.addActionListener(this);
	
		singleStepF = new JMenuItem("Step forward");
		singleStepF.setMnemonic(KeyEvent.VK_S);
		singleStepF.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0));
		singleStepF.addActionListener(this);

		multipleStepF = new JMenuItem("Multiple step forward");
		multipleStepF.setMnemonic(KeyEvent.VK_M);
		multipleStepF.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F8, 0));
		multipleStepF.addActionListener(this);

		runF = new JMenuItem("Run forward");
		runF.setMnemonic(KeyEvent.VK_F);
		runF.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0));
		runF.addActionListener(this);

		stop = new JMenuItem("Stop");
		stop.setMnemonic(KeyEvent.VK_P);
		stop.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F2, 0));
		stop.addActionListener(this);
		
		reset = new JMenuItem("Reset");
		reset.setMnemonic(KeyEvent.VK_R);
		reset.addActionListener(this);

		breakpoints = new JMenuItem("Breakpoints");
		breakpoints.setMnemonic(KeyEvent.VK_K);
		breakpoints.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, ActionEvent.CTRL_MASK));
		breakpoints.addActionListener(this);
		
		registerModify = new JMenuItem("Register Modify");
		registerModify.setMnemonic(KeyEvent.VK_G);
		registerModify.addActionListener(this);

		memoryModify = new JMenuItem("Memory Modify");
		memoryModify.setMnemonic(KeyEvent.VK_E);
		memoryModify.addActionListener(this);

		status = new JMenuItem("Status");
		status.addActionListener(this);
		
		commandConsole = new JMenuItem("Command console");
		commandConsole.addActionListener(this);

		settings = new JMenuItem("Settings");
		settings.addActionListener(this);

		simulatorMenu.add(reset);
		simulatorMenu.addSeparator();
		simulatorMenu.add(runF);
		simulatorMenu.add(stop);
		simulatorMenu.add(runB);
		simulatorMenu.addSeparator();		
		simulatorMenu.add(singleStepF);
		simulatorMenu.add(multipleStepF);
		simulatorMenu.add(singleStepB);
		simulatorMenu.add(multipleStepB);
		simulatorMenu.addSeparator();
		simulatorMenu.add(registerModify);
		simulatorMenu.add(memoryModify);
		simulatorMenu.add(breakpoints);
		simulatorMenu.add(status);
		simulatorMenu.addSeparator();
		simulatorMenu.add(commandConsole);
		simulatorMenu.addSeparator();
		simulatorMenu.add(settings);
		
		showRegisterWindow = new JCheckBoxMenuItem("Registers", !(registerWindowIsIcon));
		showRegisterWindow.addActionListener(this);

		showMemoryWindow = new JCheckBoxMenuItem("Memory", !(memoryWindowIsIcon));
		showMemoryWindow.addActionListener(this);

		showSourceWindow = new JCheckBoxMenuItem("Source", !(sourceWindowIsIcon));
		showSourceWindow.addActionListener(this);

		showConsoleWindow = new JCheckBoxMenuItem("Console", !(consoleWindowIsIcon));
		showConsoleWindow.addActionListener(this);

		resetAllWindows = new JMenuItem("Reset all");
		resetAllWindows.addActionListener(this);

		windowMenu.add(showRegisterWindow);
		windowMenu.add(showMemoryWindow);
		windowMenu.add(showSourceWindow);
		windowMenu.add(showConsoleWindow);
		windowMenu.addSeparator();
		windowMenu.add(resetAllWindows);
		
		helpContents = new JMenuItem("Contents");
		helpContents.addActionListener(this);

		about = new JMenuItem("About");
		about.setMnemonic(KeyEvent.VK_A);
		about.addActionListener(this);
		
		helpMenu.add(helpContents);
		helpMenu.add(about);	
	}

// ===================================================================================
// METHOD:		setUpToolBar
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Sets up the application toolbar.
  * 
*/
	public void setUpToolBar()
	{
		toolBar.setFloatable(false);
		
		openButton = new JButton(new ImageIcon("./toolBarGraphics/Open16.gif"));
		openButton.addActionListener(this);
		openButton.setToolTipText("Open");

		goBButton = new JButton(new ImageIcon("./toolBarGraphics/StepBack16.gif"));
		goBButton.addActionListener(this);
		goBButton.setToolTipText("Run backward");

		multiStepBackButton = new JButton(new ImageIcon("./toolBarGraphics/Rewind16.gif"));
		multiStepBackButton.addActionListener(this);
		multiStepBackButton.setToolTipText("Multi-step backward");

		stepBackButton = new JButton(new ImageIcon("./toolBarGraphics/PlayBack16.gif"));
		stepBackButton.addActionListener(this);
		stepBackButton.setToolTipText("Step backward");

		stopButton = new JButton(new ImageIcon("./toolBarGraphics/Stop16.gif"));
		stopButton.addActionListener(this);
		stopButton.setToolTipText("Stop");

		stepForwardButton = new JButton(new ImageIcon("./toolBarGraphics/Play16.gif"));
		stepForwardButton.addActionListener(this);
		stepForwardButton.setToolTipText("Step forward");

		multiStepForwardButton = new JButton(new ImageIcon("./toolBarGraphics/FastForward16.gif"));
		multiStepForwardButton.addActionListener(this);
		multiStepForwardButton.setToolTipText("Multi-step forward");		

		goFButton = new JButton(new ImageIcon("./toolBarGraphics/StepForward16.gif"));
		goFButton.addActionListener(this);
		goFButton.setToolTipText("Run forward");

		settingsButton = new JButton(new ImageIcon("./toolBarGraphics/Preferences16.gif"));
		settingsButton.addActionListener(this);
		settingsButton.setToolTipText("Settings");
		            
		toolBar.add(openButton);
		toolBar.addSeparator();
		toolBar.add(goBButton);
		toolBar.add(multiStepBackButton);
		toolBar.add(stepBackButton);
		toolBar.add(stopButton);
		toolBar.add(stepForwardButton);
		toolBar.add(multiStepForwardButton);
		toolBar.add(goFButton);
		toolBar.addSeparator();
		toolBar.add(settingsButton);
		toolBar.setBorderPainted(false);
		toolBar.setVisible(true);
		toolBar.setAlignmentX(Component.LEFT_ALIGNMENT);
	}

// ===================================================================================
// METHOD:		windowXXXXX
// FUNCTION:	handles WindowEvent events
// ===================================================================================		
	public void windowOpened(WindowEvent we) { }
	public void windowClosing(WindowEvent we)
	{
		performExit();
	}
	public void windowClosed(WindowEvent we)
	{
		performExit();
	}
	public void windowActivated(WindowEvent we) { }
	public void windowDeactivated(WindowEvent we) { }
	public void windowIconified(WindowEvent we) { }
	public void windowDeiconified(WindowEvent we) { }


// ===================================================================================
// METHOD:		componentXXXXX
// FUNCTION:	handles the case when the internal frames
//			  or top-level frame are moved or resized
// ===================================================================================		
	public void componentMoved(ComponentEvent ce)
	{
		if (ce.getSource() == registerWindow)
		{
			registersX = registerWindow.getX();
			registersY = registerWindow.getY();
		}
		if (ce.getSource() == memoryWindow)
		{
			memoryX = memoryWindow.getX();
			memoryY = memoryWindow.getY();
		}
		if (ce.getSource() == sourceWindow)
		{
			sourceX = sourceWindow.getX();
			sourceY = sourceWindow.getY();
		}
		if (ce.getSource() == consoleWindow)
		{
			consoleX = consoleWindow.getX();
			consoleY = consoleWindow.getY();
		}
	}

	public void componentResized(ComponentEvent ce)
	{
		if (ce.getSource() == registerWindow)
		{
			registersWidth = registerWindow.getWidth();
			registersHeight = registerWindow.getHeight();
			validate();
		}
		if (ce.getSource() == memoryWindow)
		{
			memoryWidth = memoryWindow.getWidth();
			memoryHeight = memoryWindow.getHeight();
			validate();
		}
		if (ce.getSource() == sourceWindow)
		{
			sourceWidth = sourceWindow.getWidth();
			sourceHeight = sourceWindow.getHeight();
			sourcePane.setSize(sourceWindow.getContentPane().getWidth(), sourceWindow.getContentPane().getHeight());
			validate();
		}
		if (ce.getSource() == consoleWindow)
		{
			consoleWidth = consoleWindow.getWidth();
			consoleHeight = consoleWindow.getHeight();
			validate();
		}
		if (ce.getSource() == TopLevelWindow.this)
		{
			int newDeskWidth = desktopPane.getWidth();
			int newDeskHeight = desktopPane.getHeight();
			int temp;
			registersX = (int)(Math.rint(((double)(registersX * newDeskWidth)) / desktopPaneWidth));
			registersY = (int)(Math.rint(((double)(registersY * newDeskHeight)) / desktopPaneHeight));
			temp = (int)(Math.rint(((double)(registersWidth * newDeskWidth)) / desktopPaneWidth));
			if (temp < newDeskWidth) registersWidth = temp;
			else registersWidth = newDeskWidth;
			temp = (int)(Math.rint(((double)(registersHeight * newDeskHeight)) / desktopPaneHeight));
			if (temp < newDeskHeight) registersHeight = temp;
			else registersHeight = newDeskHeight;
			registerWindow.setBounds(registersX, registersY, registersWidth, registersHeight);
			memoryX = (int)(Math.rint((((double)memoryX * newDeskWidth) / desktopPaneWidth)));
			memoryY = (int)(Math.rint((((double)memoryY * newDeskHeight) / desktopPaneHeight)));
			temp = (int)(Math.rint((((double)memoryWidth * newDeskWidth) / desktopPaneWidth)));
			if (temp < newDeskWidth) memoryWidth = temp;
			else memoryWidth = newDeskWidth;
			temp = (int)(Math.rint((((double)memoryHeight * newDeskHeight) / desktopPaneHeight)));
			if (temp < newDeskHeight) memoryHeight = temp;
			else memoryHeight = newDeskHeight;
			memoryWindow.setBounds(memoryX, memoryY, memoryWidth, memoryHeight);	
			sourceX = (int)(Math.rint((((double)sourceX * newDeskWidth) / desktopPaneWidth)));
			sourceY = (int)(Math.rint((((double)sourceY * newDeskHeight) / desktopPaneHeight)));
			temp = (int)(Math.rint((((double)sourceWidth * newDeskWidth) / desktopPaneWidth)));
			if (temp < newDeskWidth) sourceWidth = temp;
			else sourceWidth = newDeskWidth;
			temp = (int)(Math.rint((((double)sourceHeight * newDeskHeight) / desktopPaneHeight)));
			if (temp < newDeskHeight) sourceHeight = temp;
			else sourceHeight = newDeskHeight;
			sourceWindow.setBounds(sourceX, sourceY, sourceWidth, sourceHeight);
			consoleX = (int)(Math.rint((((double)consoleX * newDeskWidth) / desktopPaneWidth)));
			consoleY = (int)(Math.rint((((double)consoleY * newDeskHeight) / desktopPaneHeight)));
			temp = (int)(Math.rint((((double)consoleWidth * newDeskWidth) / desktopPaneWidth)));
			if (temp < newDeskWidth) consoleWidth = temp;
			else consoleWidth = newDeskWidth;
			temp = (int)(Math.rint((((double)consoleHeight * newDeskHeight) / desktopPaneHeight)));
			if (temp < newDeskHeight) consoleHeight = temp;
			else consoleHeight = newDeskHeight;
			consoleWindow.setBounds(consoleX, consoleY, consoleWidth, consoleHeight);
			sourcePane.setSize(sourceWindow.getContentPane().getWidth(), sourceWindow.getContentPane().getHeight());
			desktopPaneWidth = desktopPane.getWidth();
			desktopPaneHeight = desktopPane.getHeight();
			validate();
		}
	}
	public void componentShown(ComponentEvent ce) {};
	public void componentHidden(ComponentEvent ce) {};

// ===================================================================================
// METHOD:		internalFrameXXXXX
// FUNCTION:	handles WindowEvent events
// ===================================================================================		
	public void internalFrameOpened(InternalFrameEvent ife) {};
	public void internalFrameClosed(InternalFrameEvent ife) {};
	public void internalFrameClosing(InternalFrameEvent ife) {};
	public void internalFrameActivated(InternalFrameEvent ife) {};
	public void internalFrameDeactivated(InternalFrameEvent ife) {};
	public void internalFrameIconified(InternalFrameEvent ife)
	{
		JInternalFrame source = (JInternalFrame)ife.getSource();
		if (source == registerWindow)
		{
			registerWindowIsIcon = true;
			showRegisterWindow.setSelected(false);
		}
		if (source == memoryWindow)
		{
			memoryWindowIsIcon = true;
			showMemoryWindow.setSelected(false);
		}
		if (source == sourceWindow)
		{
			sourceWindowIsIcon = true;
			showSourceWindow.setSelected(false);
		}
		if (source == consoleWindow)
		{
			consoleWindowIsIcon = true;
			showConsoleWindow.setSelected(false);
		}
	}
	public void internalFrameDeiconified(InternalFrameEvent ife)
	{
		JInternalFrame source = (JInternalFrame)ife.getSource();
		if (source == registerWindow)
		{
			registerWindowIsIcon = false;
			showRegisterWindow.setSelected(true);
		}
		if (source == memoryWindow)
		{
			memoryWindowIsIcon = false;
			showMemoryWindow.setSelected(true);
		}
		if (source == sourceWindow)
		{
			sourceWindowIsIcon = false;
			showSourceWindow.setSelected(true);
		}
		if (source == consoleWindow)
		{
			consoleWindowIsIcon = false;
			showConsoleWindow.setSelected(true);
		}
	}

// ===================================================================================
// METHOD:		updateSettings
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Updates the application settings.
  * 
  * @param GPR Defines whether the General Purpose Registers should be displayed in hexadecimal.
  * @param SFP Defines whether the Single Precision Floating Point registers should be displayed in hexadecimal.
  * @param DFP Defines whether the Double Precision Floating Point registers should be displayed in hexadecimal.
  * @param ignoreInvalidATUse Defines whether invalid use of the $at register should be ignored.
*/
	public void updateSettings(boolean GPR, boolean SFP, boolean DFP, boolean ignoreInvalidATUse)
	{
		GPRHex = GPR;
		SFPHex = SFP;
		DFPHex = DFP;
		ignoreInvalidAT = ignoreInvalidATUse;
		displayUpdate();
	}

// ===================================================================================
// METHOD:		loadConfig
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Loads configuration settings from the file jimswing.cfg.
  * 
*/
	private void loadConfig()
	{
		File f = new File("jimswing.cfg");
		RandomAccessFile raf = null;
		if (f.exists())
		{
			try
			{
				raf = new RandomAccessFile(f, "r");
			}
			catch (FileNotFoundException fnfe)
			{
				System.err.println("File not found in TopLevelWindow.loadConfig.");
			}
			try
			{
				raf.seek(0);
				openDir = raf.readUTF();
				applicationX = raf.readInt();
				if (applicationX < 0) applicationX = 0;
				applicationY = raf.readInt();
				if (applicationY < 0) applicationY = 0;
				applicationWidth = raf.readInt();
				applicationHeight = raf.readInt();
				registersX = raf.readInt();
				registersY = raf.readInt();
				registersWidth = raf.readInt();
				registersHeight = raf.readInt();
				registerWindowIsIcon = raf.readBoolean();
				memoryX = raf.readInt();
				memoryY = raf.readInt();
				memoryWidth = raf.readInt();
				memoryHeight = raf.readInt();
				memoryWindowIsIcon = raf.readBoolean();
				sourceX = raf.readInt();
				sourceY = raf.readInt();
				sourceWidth = raf.readInt();
				sourceHeight = raf.readInt();
				sourceWindowIsIcon = raf.readBoolean();
				consoleX = raf.readInt();
				consoleY = raf.readInt();
				consoleWidth = raf.readInt();
				consoleHeight = raf.readInt();
				consoleWindowIsIcon = raf.readBoolean();
				GPRHex = raf.readBoolean();
				SFPHex = raf.readBoolean();
				DFPHex = raf.readBoolean();
				ignoreInvalidAT = raf.readBoolean();
				raf.close();
			}
			catch (IOException ioe)
			{
				System.err.println("Error reading file in TopLevelWindow.loadConfig.");
				JOptionPane.showMessageDialog(this, "Error reading configuration file.", "Error", JOptionPane.INFORMATION_MESSAGE);
			}
		}
		else
		{
			restoreConfigFile();
		}
	}

// ===================================================================================
// METHOD:		performReset
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Resets the simulator and the display windows.
  * 
*/
	private boolean performReset()
	{
		boolean returnValue = false;
		commandProcessor = new CommandProcessor();
		commandResultBuffer = commandProcessor.processCommand("RESET");
		if (commandResultBuffer.sGetNextResponse().startsWith("* RESET OK"))
		{
			if (shouldRun)
			{
				shouldRun = false;
			}
			sourceCode = new Vector();
			sourceLinks = null;
			sourceCode.addElement("No source file loaded.");
			sourceList.setListData(sourceCode);
			Vector memoryContents = new Vector();
			memoryContents.addElement("No modified memory.");
			memoryList.setListData(memoryContents);
			commandResultBuffer = commandProcessor.processCommand("SET REGISTER GPR $PC 0x00400000");
			String setReturn = commandResultBuffer.sGetNextResponse();
			if (!(setReturn.startsWith("* SET DONE")))
			{
				JOptionPane.showMessageDialog(this, "Error resetting program counter.", "Error", JOptionPane.INFORMATION_MESSAGE);
			}
			displayUpdate();
			fileLoaded = false;
			returnValue = true;
		}
		else
		{
			JOptionPane.showMessageDialog(this, "Error resetting simulator.", "Error", JOptionPane.INFORMATION_MESSAGE);
			performExit();
		}
		return returnValue;
	}

// ===================================================================================
// METHOD:		performExit
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Cleans up and exits the application.
  * 
*/
	private void performExit()
	{
		if (commandProcessor != null)
		{
			commandResultBuffer = commandProcessor.processCommand("EXIT");
		}
		applicationX = getX();
		if (applicationX < 0) applicationX = 0;
		applicationY = getY();
		if (applicationY < 0) applicationY = 0;
		applicationWidth = getWidth();
		applicationHeight = getHeight();

		registersX = registerWindow.getX();
		registersY = registerWindow.getY();
		registersWidth = registerWindow.getWidth();
		registersHeight = registerWindow.getHeight();
		registerWindowIsIcon = registerWindow.isIcon();

		memoryX = memoryWindow.getX();
		memoryY = memoryWindow.getY();
		memoryWidth = memoryWindow.getWidth();
		memoryHeight = memoryWindow.getHeight();
		memoryWindowIsIcon = memoryWindow.isIcon();

		sourceX = sourceWindow.getX();
		sourceY = sourceWindow.getY();
		sourceWidth = sourceWindow.getWidth();
		sourceHeight = sourceWindow.getHeight();
		sourceWindowIsIcon = sourceWindow.isIcon();

		consoleX = consoleWindow.getX();
		consoleY = consoleWindow.getY();
		consoleWidth = consoleWindow.getWidth();
		consoleHeight = consoleWindow.getHeight();
		consoleWindowIsIcon = consoleWindow.isIcon();
	
		RandomAccessFile raf = null;
		try
		{
			raf = new RandomAccessFile("jimswing.cfg", "rw");
		}
		catch (FileNotFoundException fnfe)
		{
			System.err.println("File not found in TopLevelWindow.performExit.");
		}
		writeConfig(raf);
		System.exit(0);
	}

// ===================================================================================
// METHOD:		restoreWindows
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Restore the main application window and all subwindows to their default orientation.
  * 
*/
	private void restoreWindows()
	{
		removeComponentListener(this);
		applicationX = (int)(Toolkit.getDefaultToolkit().getScreenSize().getWidth() / 4);
		applicationY = (int)(Toolkit.getDefaultToolkit().getScreenSize().getHeight() / 8);
		applicationWidth = (int)(Toolkit.getDefaultToolkit().getScreenSize().getWidth() / 2);
		applicationHeight = (int)(3*(Toolkit.getDefaultToolkit().getScreenSize().getHeight() / 4));

		desktopPaneWidth = applicationWidth - xCorrection;
		desktopPaneHeight = applicationHeight - yCorrection;

		registersX = 0;
		registersY = 0;
		registersWidth = (int)((Toolkit.getDefaultToolkit().getScreenSize().getWidth() - xCorrection) / 2);
		registersHeight = desktopPaneHeight / 4;
		registerWindowIsIcon = false;
		if (registerWindow != null)
		{
			registerWindow.setBounds(registersX, registersY, registersWidth, registersHeight);
			try
			{
				registerWindow.setSelected(true);
				registerWindow.setIcon(false);
				registerWindowIsIcon = false;
				showRegisterWindow.setSelected(true);
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.out.println("Error iconifying/deiconifying window in TopLevelWindow.restoreWindows.");
			}
		}
		memoryX = 0;
		memoryY = 1 * (desktopPaneHeight / 4);
		memoryWidth = (int)((Toolkit.getDefaultToolkit().getScreenSize().getWidth() - xCorrection) / 2);
		memoryHeight = desktopPaneHeight / 4;
		memoryWindowIsIcon = false;
		if (memoryWindow != null)
		{
			memoryWindow.setBounds(memoryX, memoryY, memoryWidth, memoryHeight);
			try
			{
				memoryWindow.setIcon(false);
				memoryWindowIsIcon = false;
				showMemoryWindow.setSelected(true);
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.out.println("Error iconifying/deiconifying window in TopLevelWindow.restoreWindows.");
			}
		}
		sourceX = 0;
		sourceY = 2 * (desktopPaneHeight / 4);
		sourceWidth = (int)((Toolkit.getDefaultToolkit().getScreenSize().getWidth() - xCorrection) / 2);
		sourceHeight = desktopPaneHeight / 4;
		sourceWindowIsIcon = false;
		if (sourceWindow != null)
		{
			sourceWindow.setBounds(sourceX, sourceY, sourceWidth, sourceHeight);
			try
			{
				sourceWindow.setIcon(false);
				sourceWindowIsIcon = false;
				showSourceWindow.setSelected(true);
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.out.println("Error iconifying/deiconifying window in TopLevelWindow.restoreWindows.");
			}
		}

		consoleX = 0;
		consoleY = 3 * (desktopPaneHeight / 4);
		consoleWidth = (int)((Toolkit.getDefaultToolkit().getScreenSize().getWidth() - xCorrection) / 2);
		consoleHeight = desktopPaneHeight / 4;
		consoleWindowIsIcon = false;
		if (consoleWindow != null)
		{
			consoleWindow.setBounds(consoleX, consoleY, consoleWidth, consoleHeight);
			try
			{
				consoleWindow.setIcon(false);
				consoleWindowIsIcon = false;
				showConsoleWindow.setSelected(true);
			}
			catch (java.beans.PropertyVetoException pve)
			{
				System.out.println("Error iconifying/deiconifying window in TopLevelWindow.restoreWindows.");
			}
		}
		setLocation(applicationX, applicationY);
		setSize(applicationWidth, applicationHeight);
		addComponentListener(this);
	}

// ===================================================================================
// METHOD:		restoreConfigFile
// FUNCTION:
// RETURNS:
// ===================================================================================
	private void restoreConfigFile()
	{
		openDir = "nonespecified";

		restoreWindows();

		GPRHex = true;
		SFPHex = true;
		DFPHex = true;
		ignoreInvalidAT = false;

		File f = new File("jimswing.cfg");
		try
		{
			f.createNewFile();
		}
		catch (IOException ioe)
		{
			System.err.println("Error creating file in TopLevelWindow.restoreConfigFile.");
			JOptionPane.showMessageDialog(this, "Error creating configuration file.", "Error", JOptionPane.INFORMATION_MESSAGE);
		}
		RandomAccessFile raf = null;
		try
		{
			raf = new RandomAccessFile(f, "rw");
		}
		catch (FileNotFoundException fnfe)
		{
			System.err.println("File not found in TopLevelWindow.restoreConfigFile.");
		}
		writeConfig(raf);
	}

// ===================================================================================
// METHOD:		writeConfig
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Writes the configuration settings to the file jimswing.cfg.
  * 
*/
	private void writeConfig(RandomAccessFile raf)
	{
		try
		{
			raf.seek(0);
			raf.writeUTF(openDir);
			raf.writeInt(applicationX);
			raf.writeInt(applicationY);
			raf.writeInt(applicationWidth);
			raf.writeInt(applicationHeight);
			raf.writeInt(registersX);
			raf.writeInt(registersY);
			raf.writeInt(registersWidth);
			raf.writeInt(registersHeight);
			raf.writeBoolean(registerWindowIsIcon);
			raf.writeInt(memoryX);
			raf.writeInt(memoryY);
			raf.writeInt(memoryWidth);
			raf.writeInt(memoryHeight);
			raf.writeBoolean(memoryWindowIsIcon);
			raf.writeInt(sourceX);
			raf.writeInt(sourceY);
			raf.writeInt(sourceWidth);
			raf.writeInt(sourceHeight);
			raf.writeBoolean(sourceWindowIsIcon);
			raf.writeInt(consoleX);
			raf.writeInt(consoleY);
			raf.writeInt(consoleWidth);
			raf.writeInt(consoleHeight);
			raf.writeBoolean(consoleWindowIsIcon);
			raf.writeBoolean(GPRHex);
			raf.writeBoolean(SFPHex);
			raf.writeBoolean(DFPHex);
			raf.writeBoolean(ignoreInvalidAT);
			raf.close();
		}
		catch (IOException ioe)
		{
			System.out.println("Error writing to file in TopLevelWindow.writeConfig.");
			JOptionPane.showMessageDialog(this, "Error writing configuration file.", "Error", JOptionPane.INFORMATION_MESSAGE);
		}		
	}

// ===================================================================================
// METHOD:		updateConsole
// FUNCTION:
// RETURNS:
// ===================================================================================
	private void updateConsole()
	{
		int textCount = consoleText.getText().length() - 1;
		if (textCount >= 0)
		{
			consoleText.setCaretPosition(textCount);
		}
	}

// ===================================================================================
// METHOD:		updateMemory
// FUNCTION:
// RETURNS:
// ===================================================================================
	private void updateMemory()
	{
		if (commandProcessor != null)
		{
			Vector memoryListData = new Vector();
			String s;
			StringBuffer memoryDump = new StringBuffer();
			commandResultBuffer = commandProcessor.processCommand("GET MEMORY MODIFIED");
			while (!(s = commandResultBuffer.sGetNextResponse()).equals("* GET DONE"))
			{
				StringTokenizer st = new StringTokenizer(s);
				StringBuffer response = new StringBuffer();
				response.append(st.nextToken());
				response.append(st.nextToken());
				response.append(st.nextToken());
				String address = st.nextToken();
				memoryDump.append("[" + address + "]: ");
				int iAddress = Utility.iHexStrToInt(address);
				int counter = 0;
				while (st.hasMoreTokens())
				{
					String next = st.nextToken();
					memoryDump.append(next + " ");
					counter++;
					if (counter == 16) 
					{
						counter = 0;
						iAddress += 16;
						if (st.hasMoreTokens())
						{
							memoryListData.addElement(memoryDump.toString());
							memoryDump = new StringBuffer();
							memoryDump.append("[" + Utility.sAsHexPadded(iAddress, 8) + "]: ");
						}
					}
				}
				memoryListData.addElement(memoryDump.toString());
				memoryListData.addElement(" ");
				memoryDump = new StringBuffer();
			}
			if (memoryListData.isEmpty())
			{
				memoryListData.addElement("No modified memory.");
			}
			memoryList.setListData(memoryListData);
		}
	}

// ===================================================================================
// METHOD:		updateRegisters
// FUNCTION:
// RETURNS:
// ===================================================================================
	private void updateRegisters()
	{
		registersContent = new StringBuffer();
		int firstVisibleIndex = 0;
		if (registerList.getFirstVisibleIndex() != -1)		
		{		
			firstVisibleIndex = registerList.getFirstVisibleIndex();
		}
		Vector registerText = new Vector();
		registerText.addElement("PC  = " + getRegisterValue("GPR", "$PC") + "  " + "EPC  = " +
			getRegisterValue("CP0", "$EPC") + "  " + "Cause    = " +
			getRegisterValue("CP0", "$CAUSE") + "  " + "SR   = " +
			getRegisterValue("CP0", "$SR"));
		registerText.addElement("HI  = " +  getRegisterValue("GPR", "$HI") + "  " +  "LO   = " +
			getRegisterValue("GPR", "$LO") + "  " +  "BadVAddr = " +
			getRegisterValue("CP0", "$BADVADDR") + "  " + "PrID = " +
			getRegisterValue("CP0", "$PRID"));
		registerText.addElement(" ");
		registerText.addElement("                      General Registers");
		registerText.addElement("R00 (r0) = " + getRegisterValue("GPR", "$ZERO") + "  " +
			"R08 (t0) = " + getRegisterValue("GPR", "$T0") + "  " + "R16 (s0) = " +
			getRegisterValue("GPR", "$S0") + "  " + "R24 (t8) = " + 
			getRegisterValue("GPR", "$T8"));
		registerText.addElement("R01 (at) = " + getRegisterValue("GPR", "$AT") + "  " +
			"R09 (t1) = " + getRegisterValue("GPR", "$T1") + "  " + "R17 (s1) = " +
			getRegisterValue("GPR", "$S1") + "  " +  "R25 (t9) = " +
			getRegisterValue("GPR", "$T9"));
		registerText.addElement("R02 (v0) = " + getRegisterValue("GPR", "$V0") + "  " +
			"R10 (t2) = " + getRegisterValue("GPR", "$T2") + "  " +  "R18 (s2) = " +
			getRegisterValue("GPR", "$S2") + "  " + "R26 (k0) = " + 
			getRegisterValue("GPR", "$K0"));
		registerText.addElement("R03 (v1) = " + getRegisterValue("GPR", "$V1") + "  " +
			"R11 (t3) = " + getRegisterValue("GPR", "$T3") + "  " + "R19 (s3) = " +
			getRegisterValue("GPR", "$S3") + "  " + "R27 (k1) = " +
			getRegisterValue("GPR", "$K1"));
		registerText.addElement("R04 (a0) = " + getRegisterValue("GPR", "$A0") + "  " +
			"R12 (t4) = " + getRegisterValue("GPR", "$T4") + "  " + "R20 (s4) = " +
			getRegisterValue("GPR", "$S4") + "  " + "R28 (gp) = " +
			getRegisterValue("GPR", "$GP"));
		registerText.addElement("R05 (a1) = " + getRegisterValue("GPR", "$A1") + "  " +
			"R13 (t5) = " + getRegisterValue("GPR", "$T5") + "  " + "R21 (s5) = " +
			getRegisterValue("GPR", "$S5") + "  " + "R29 (sp) = " +
			getRegisterValue("GPR", "$SP"));
		registerText.addElement("R06 (a2) = " + getRegisterValue("GPR", "$A2") + "  " +
			"R14 (t6) = " + getRegisterValue("GPR", "$T6") + "  " + "R22 (s6) = " +
			getRegisterValue("GPR", "$S6") + "  " + "R30 (fp) = " +
			getRegisterValue("GPR", "$FP"));
		registerText.addElement("R07 (a3) = " + getRegisterValue("GPR", "$A3") + "  " +
			"R15 (t7) = " + getRegisterValue("GPR", "$T7") + "  " + "R23 (s7) = " +
			getRegisterValue("GPR", "$S7") + "  " + "R31 (ra) = " +
			getRegisterValue("GPR", "$RA"));
		registerText.addElement(" ");
		registerText.addElement("                      Single Floating Point Registers");
		registerText.addElement("FP0  = " + getFPRegisterValue("CP1", "$FP0") + "  " +
			"FP8  = " + getFPRegisterValue("CP1", "$FP8") + "  " +
			"FP16 = " + getFPRegisterValue("CP1", "$FP16") + "  " +
			"FP24 = " + getFPRegisterValue("CP1", "$FP24"));
		registerText.addElement("FP1  = " + getFPRegisterValue("CP1", "$FP1") + "  " +
			"FP9  = " + getFPRegisterValue("CP1", "$FP9") + "  " +
			"FP17 = " + getFPRegisterValue("CP1", "$FP17") + "  " +
			"FP25 = " + getFPRegisterValue("CP1", "$FP25"));
		registerText.addElement("FP2  = " + getFPRegisterValue("CP1", "$FP2") + "  " +
			"FP10 = " + getFPRegisterValue("CP1", "$FP10") + "  " +
			"FP18 = " + getFPRegisterValue("CP1", "$FP18") + "  " +
			"FP26 = " + getFPRegisterValue("CP1", "$FP26"));
		registerText.addElement("FP3  = " + getFPRegisterValue("CP1", "$FP3") + "  " +
			"FP11 = " + getFPRegisterValue("CP1", "$FP11") + "  " +
			"FP19 = " + getFPRegisterValue("CP1", "$FP19") + "  " +
			"FP27 = " + getFPRegisterValue("CP1", "$FP27"));
		registerText.addElement("FP4  = " + getFPRegisterValue("CP1", "$FP4") + "  " +
			"FP12 = " + getFPRegisterValue("CP1", "$FP12") + "  " +
			"FP20 = " + getFPRegisterValue("CP1", "$FP20") + "  " +
			"FP28 = " + getFPRegisterValue("CP1", "$FP28"));
		registerText.addElement("FP5  = " + getFPRegisterValue("CP1", "$FP5") + "  " +
			"FP13 = " + getFPRegisterValue("CP1", "$FP13") + "  " +
			"FP21 = " + getFPRegisterValue("CP1", "$FP21") + "  " +
			"FP29 = " + getFPRegisterValue("CP1", "$FP29"));
		registerText.addElement("FP6  = " + getFPRegisterValue("CP1", "$FP6") + "  " +
			"FP14 = " + getFPRegisterValue("CP1", "$FP14") + "  " +
			"FP22 = " + getFPRegisterValue("CP1", "$FP22") + "  " +
			"FP30 = " + getFPRegisterValue("CP1", "$FP30"));
		registerText.addElement("FP7  = " + getFPRegisterValue("CP1", "$FP7") + "  " +
			"FP15 = " + getFPRegisterValue("CP1", "$FP15") + "  " +
			"FP23 = " + getFPRegisterValue("CP1", "$FP23") + "  " +
			"FP31 = " + getFPRegisterValue("CP1", "$FP31"));
		registerText.addElement(" ");
		registerText.addElement("                      Double Floating Point Registers");
		registerText.addElement("FP0  = " + getDoubleFPRegisterValue("CP1", "$FP0", "$FP1") + "  " +
			"FP8  = " + getDoubleFPRegisterValue("CP1", "$FP8", "$FP9") + "  " +
			"FP16 = " + getDoubleFPRegisterValue("CP1", "$FP16", "$FP17") + "  " +
			"FP24 = " + getDoubleFPRegisterValue("CP1", "$FP24", "$FP25"));
		registerText.addElement("FP2  = " + getDoubleFPRegisterValue("CP1", "$FP2", "$FP3") + "  " +
			"FP10 = " + getDoubleFPRegisterValue("CP1", "$FP10", "$FP11") + "  " +
			"FP18 = " + getDoubleFPRegisterValue("CP1", "$FP18", "$FP19") + "  " +
			"FP26 = " + getDoubleFPRegisterValue("CP1", "$FP26", "$FP27"));
		registerText.addElement("FP4  = " + getDoubleFPRegisterValue("CP1", "$FP4", "$FP5") + "  " +
			"FP12 = " + getDoubleFPRegisterValue("CP1", "$FP12", "$FP13") + "  " +
			"FP20 = " + getDoubleFPRegisterValue("CP1", "$FP20", "$FP21") + "  " +
			"FP28 = " + getDoubleFPRegisterValue("CP1", "$FP28", "$FP29"));
		registerText.addElement("FP6  = " + getDoubleFPRegisterValue("CP1", "$FP6", "$FP7") + "  " +
			"FP14 = " + getDoubleFPRegisterValue("CP1", "$FP14", "$FP15") + "  " +
			"FP22 = " + getDoubleFPRegisterValue("CP1", "$FP22", "$FP23") + "  " +
			"FP30 = " + getDoubleFPRegisterValue("CP1", "$FP30", "$FP31"));
		registerList.setListData(registerText);
		registerList.ensureIndexIsVisible(firstVisibleIndex);
	}

// ===================================================================================
// METHOD:		assemblyPhase1
// FUNCTION:
// RETURNS:
// ===================================================================================
	private int assemblyPhase1(String fullFileName, String fileDir, String fileName, boolean ignoreUseAT)
	{
		Vector vFile = Utility.vLoadFileAsVector(fullFileName);
		if (vFile == null)
		{
			JOptionPane.showMessageDialog(this, "Error loading " + fullFileName + " for phase 1 assembler processing.", "Error", JOptionPane.ERROR_MESSAGE);
			return 1;
		}
		Assembler assembler = new Assembler(fileDir + File.separator + fileName + ".modified");
		assembler.reset();
		Enumeration e = vFile.elements();
		int errorCount = assemblyPhase1Processing(assembler, e);
		if (!ignoreUseAT && (assembler.iGetLastInvalidUseOfAT() != -1))
		{
			int lastInvalidUse = assembler.iGetLastInvalidUseOfAT();
			JOptionPane.showMessageDialog(this, "Line " + lastInvalidUse + ": Invalid use of $at register.", "Error", JOptionPane.ERROR_MESSAGE);
			errorCount++;
		}
		if (errorCount > 0)
		{
			JOptionPane.showMessageDialog(this, errorCount + " error(s) detected during phase 1.", "Error", JOptionPane.ERROR_MESSAGE);
		}
		return errorCount;
	}

// ===================================================================================
// METHOD:		assemblyPhase2
// FUNCTION:
// RETURNS:
// ===================================================================================
	private int assemblyPhase2(String fullFileName, String fileDir, String fileName, boolean ignoreUseAT)
	{
		Vector vFile = Utility.vLoadFileAsVector(fullFileName);
		if (vFile == null)
		{
			JOptionPane.showMessageDialog(this, "Error loading " + fullFileName + " for phase 2 assembler processing.", "Error", JOptionPane.ERROR_MESSAGE);
			return 1;
		}
		Assembler assembler = new Assembler();
		assembler.reset();
		Enumeration e = vFile.elements();
		int errorCount = assemblyPhase1Processing(assembler, e);
		if (!ignoreUseAT && (assembler.iGetLastInvalidUseOfAT() != -1))
		{
			int lastInvalidUse = assembler.iGetLastInvalidUseOfAT();
			JOptionPane.showMessageDialog(this, "Line " + lastInvalidUse + ": Invalid use of $at register.", "Error", JOptionPane.ERROR_MESSAGE);
			errorCount++;
		}
		if (errorCount > 0)
		{
			JOptionPane.showMessageDialog(this, errorCount + " error(s) detected during phase 1.\nAssembler process cancelled.", "Error", JOptionPane.ERROR_MESSAGE);
		}
		StringBuffer sbMachineCode = null;
		try
		{
			sbMachineCode = assembler.sbGetMachineCode(fileName);
		}
		catch (Exception err)
		{
			JOptionPane.showMessageDialog(this, err.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
		}
		int i = assembler.iGetNumPhaseTwoErrors();
		if (i > 0)
		{
			JOptionPane.showMessageDialog(this, "Error(s) encountered during phase 2 of assembly.", "Error", JOptionPane.ERROR_MESSAGE);
		}
		else
		{
			try
			{
				BufferedWriter out = new BufferedWriter(new FileWriter(fileDir + File.separator + fileName + ".code"));
				out.write(sbMachineCode.toString());
				out.close();
			}
			catch (Exception err)
			{
				JOptionPane.showMessageDialog(this, "Error outputting machine code in assembly phase 2.", "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
		return (i + errorCount);
	}

// ===================================================================================
// METHOD:		assemblyPhase1Processing
// FUNCTION:	Uses the provided assembler to process each element of enumeration e
// RETURNS:		Number of errors encountered
// ===================================================================================
/**
  * Performs the assembly process.
  *
  * @param assembler The Assembler object to use for processing.
  * @param e The source code the process
  * @returns The number of errors encountered during processing.
  * 
*/	private int assemblyPhase1Processing(Assembler assembler, Enumeration e)
	{
		int iCurrLine = 0;
		int iErrorCount = 0;
		showMoreErrors = true;
		while (e.hasMoreElements())
		{
			String s = (String) e.nextElement();
			iCurrLine++;
			int i = 0;
			try
			{
				assembler.iProcessLine(s, iCurrLine);
			}
			catch (Exception error)
			{
				iErrorCount++;
				if (showMoreErrors)
				{
					Object[] options = {"Show more errors", "Stop"};
					int n = JOptionPane.showOptionDialog(this, error.getMessage(), "Error", JOptionPane.YES_NO_OPTION, JOptionPane.ERROR_MESSAGE, null, options, options[0]);
					if (n != JOptionPane.YES_OPTION)
					{
						showMoreErrors = false;
					}
				}
			}
		}
		return iErrorCount;
	}

// ===================================================================================
// METHOD:		getRegisterValue
// FUNCTION:
// RETURNS:
// ===================================================================================
/**
  * Gets the value stored in register reg.
  *    
  * @param regSet The register set containing register reg.
  * @param reg The name of the register to read.
  * @returns A string representing the register value.
  * 
*/
	private String getRegisterValue(String regSet, String reg)
	{
		commandResultBuffer = commandProcessor.processCommand("GET REGISTER " + regSet + " " + reg);
		String response = commandResultBuffer.sGetNextResponse();
		String regVal = new String();
		if (response.startsWith("* GET REGISTER " + regSet.toUpperCase() + " "))
		{
			response = response.substring(22);
			regVal = response;
		}
		if (regSet.toUpperCase().equals("GPR"))
		{
			if (!(reg.toUpperCase().equals("$PC")) && !(reg.toUpperCase().equals("$LO")) && !(reg.toUpperCase().equals("$HI")))
			{
				if (GPRHex)
				{
					regVal = regVal;
				}
				else
				{
					regVal = new Long(Long.parseLong(regVal,16)).toString();
				}
			}
		}
		return Utility.sPad(regVal,8);
	}

// ===================================================================================
// METHOD:		getFPRegisterValue
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Gets the single precision floating point value stored in register reg.
  * 
  * @param regSet The register set containing register reg.
  * @param reg The name of the register to read.
  * @returns A string representing a single precision floating point number.
  * 
*/
	private String getFPRegisterValue(String regSet, String reg)
	{
		String response = getRegisterValue(regSet, reg).trim();
		if (SFPHex)
		{
			response = response;
		}
		else
		{
			int highByte = Integer.parseInt(response.substring(0,1),16);
			int floatAsIntBits = 0;
			float regAsFloat = 0;			
			if (highByte > 7)
			{
				highByte = highByte ^ 0x00000008;
				response = Integer.toHexString(highByte).concat(response.substring(1));
				floatAsIntBits = Integer.parseInt(response,16);
				regAsFloat = Float.intBitsToFloat(floatAsIntBits) * -1;
			}
			else
			{
				floatAsIntBits = Integer.parseInt(response,16);
				regAsFloat = Float.intBitsToFloat(floatAsIntBits);
			}
			response = Float.toString(regAsFloat);
			if (response.length() > 12)
			{
				int exponentStart = response.indexOf("E");
				if (exponentStart != -1)
				{
					String exponent = response.substring(exponentStart);
					int exponentLength = (response.length() - 1) - exponentStart;
					String base = new String();
					if ((exponentStart + exponent.length()) >= 12)
					{
						base = response.substring(0, 12 - exponent.length());
					}
					else
					{
						base = response.substring(0, exponentStart);
					}
					response = base + exponent;
				}
				else
				{
					response = response.substring(12);
				}
			}
		}
		return Utility.sPad(response, 12);
	}

// ===================================================================================
// METHOD:		getDoubleFPRegisterValue
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
  * Gets the Double Precision Floating Point value stored in registers reg1 and reg2.
  *
  * @param regSet The register set containing reg1 and reg2: for floating point registers should be "CP1".
  * @param reg1 The register containing the lower 32-bits of the double precision floating point value.
  * @param reg2 The register containing the upper 32-bits of the double precision floating point value.
  * @returns A string representing a double precision floating point number.
*/
	private String getDoubleFPRegisterValue(String regSet, String reg1, String reg2)
	{
		String response1 = getRegisterValue(regSet, reg1).trim();
		String response2 = getRegisterValue(regSet, reg2).trim();
		String response = new String();
		if (DFPHex)
		{
			response = response2 + response1;
		}
		else
		{
			long doubleAsLongBits = 0;
			double regAsDouble = 0;
			String responseAsHex = response2 + response1;
			int highByte = Integer.parseInt(responseAsHex.substring(0,1),16);
			if (highByte > 7)
			{
				highByte = highByte ^ 0x00000008;
                        responseAsHex = Integer.toHexString(highByte).concat(responseAsHex.substring(1));
				doubleAsLongBits = Long.parseLong(responseAsHex,16);
				regAsDouble = Double.longBitsToDouble(doubleAsLongBits) * -1;
			}
			else
			{
				doubleAsLongBits = Long.parseLong(responseAsHex,16);
				regAsDouble = Double.longBitsToDouble(doubleAsLongBits);
			}
			response = Double.toString(regAsDouble);
			if (response.length() > 12)
			{
				int exponentStart = response.indexOf("E");
				if (exponentStart != -1)
				{
					String exponent = response.substring(exponentStart);
					int exponentLength = (response.length() - 1) - exponentStart;
					String base = new String();
					if ((exponentStart + exponent.length()) >= 12)
					{
						base = response.substring(0, 12 - exponent.length());
					}
					else
					{
						base = response.substring(0, exponentStart);
					}
					response = base + exponent;
				}
				else
				{
					response = response.substring(12);
				}
			}
		}
		return Utility.sPad(response, 12);
	}

// ===================================================================================
// METHOD:		highlighSourceLine
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
	* Highlights the source code instruction specified by programCounter.
	*
	* @param programCounter address of the instruction to be highlighted
*/
	private void highlightSourceLine(String programCounter)
	{
		if (sourceLinks != null)
		{
			Enumeration e = sourceLinks.elements();
			while (e.hasMoreElements())
			{
				sourceLink s = (sourceLink)e.nextElement();
				if (s.pc.equals(programCounter))
				{
					sourceList.setSelectionInterval(s.line - 1, -808);
					sourceList.ensureIndexIsVisible(s.line - 1);
				}
			}
		}
	}

// ===================================================================================
// METHOD:		displayUpdate
// FUNCTION:
// RETURNS:
// ===================================================================================
// javadoc comments follow
// ===================================================================================
/**
	* Updates the application display - register contents, memory contents, source view and i/o console.
	*
*/
	public void displayUpdate()
	{
		updateRegisters();
		updateMemory();
		highlightSourceLine(getRegisterValue("GPR", "$PC"));		
		updateConsole();
		validate();
	}

// ===================================================================================
// Private instance variables
// ===================================================================================
	private JMenuBar menuBar;
	private JMenu fileMenu;
	private JMenuItem open;
	private JMenuItem exit;
	private JMenu simulatorMenu;
	private JMenuItem singleStepF;
	private JMenuItem singleStepB;
	private JMenuItem multipleStepF;
	private JMenuItem multipleStepB;
	private JMenuItem stop;
	private JMenuItem runF;
	private JMenuItem runB;
	private JMenuItem reset;
	private JMenuItem breakpoints;
	private JMenuItem registerModify;
	private JMenuItem memoryModify;
	private JMenuItem status;
	private JMenuItem commandConsole;
	private JMenuItem settings;
	private JMenu windowMenu;
	private JCheckBoxMenuItem showRegisterWindow;
	private JCheckBoxMenuItem showMemoryWindow;
	private JCheckBoxMenuItem showSourceWindow;
	private JCheckBoxMenuItem showConsoleWindow;
	private JMenuItem resetAllWindows;
	private JMenu helpMenu;
	private JMenuItem helpContents;
	private JMenuItem about;
		
	private JInternalFrame registerWindow;
	private JList registerList;
	private JScrollPane registerPane;
	
	private JInternalFrame memoryWindow;
	private JScrollPane memoryPane;
	private JList memoryList;
	
	private JInternalFrame sourceWindow;
	private JList sourceList;
	private JScrollPane sourcePane;
	
	private JInternalFrame consoleWindow;
	private JScrollPane consolePane;
	private JTextArea consoleText;

	private Insets topInsets;	
	private Insets desktopInsets;

	private JToolBar toolBar;
	private JButton openButton;
	private JButton goFButton;
	private JButton goBButton;
	private JButton stopButton;
	private JButton stepForwardButton;
	private JButton multiStepForwardButton;
	private JButton stepBackButton;
	private JButton multiStepBackButton;
	private JButton settingsButton;

	private JPanel contentPane;	
	private JDesktopPane desktopPane;

	private StringBuffer registersContent;

	/**
	  * Directory from which a file was last opened.
	  *
	*/
	private String openDir;
	/**
	  * The application's X coordinate.
	  *
	*/
	private int applicationX;
	/**
	  * The application's Y coordinate.
	  *
	*/
	private int applicationY;
	/**
	  * The application window's width.
	  *
	*/
	private int applicationWidth;
	/**
	  * The application window's height.
	  *
	*/
	private int applicationHeight;
	/**
	  * The register window's X coordinate.
	  *
	*/
	private int registersX;
	/**
	  * The register window's Y coordinate.
	  *
	*/
	private int registersY;
	/**
	  * The register window's width.
	  *
	*/
	private int registersWidth;
	/**
	  * The register window's height.
	  *
	*/
	private int registersHeight;
	/**
	  * The memory window's X coordinate.
	  *
	*/
	private int memoryX;
	/**
	  * The memory window's Y coordinate.
	  *
	*/
	private int memoryY;
	/**
	  * The memory window's width.
	  *
	*/
	private int memoryWidth;
	/**
	  * The memory window's height.
	  *
	*/
	private int memoryHeight;
	/**
	  * The source window's X coordinate.
	  *
	*/
	private int sourceX;
	/**
	  * The source window's Y coordinate.
	  *
	*/
	private int sourceY;
	/**
	  * The source window's width.
	  *
	*/
	private int sourceWidth;
	/**
	  * The source window's height.
	  *
	*/
	private int sourceHeight;
	/**
	  * The console window's X coordinate.
	  *
	*/
	private int consoleX;
	/**
	  * The console window's Y coordinate.
	  *
	*/
	private int consoleY;
	/**
	  * The console window's width.
	  *
	*/
	private int consoleWidth;
	/**
	  * The console window's height.
	  *
	*/
	private int consoleHeight;
	private int desktopPaneWidth;
	private int desktopPaneHeight;

	private Vector sourceCode;
	private Vector sourceLinks;
	private CommandProcessor commandProcessor;
	private CommandResultBuffer commandResultBuffer;
	private runThread simExecutionThread;
	private boolean shouldRun;
	private boolean fileLoaded;
	private boolean showMoreErrors;

	/**
	  * Defines if the General Purpose Registers should be displayed in hexadecimal.
	  *
	*/
	private boolean GPRHex;
	/**
	  * Defines if the Single Precision Floating Point Registers should be displayed in hexadecimal.
	  *
	*/
	private boolean SFPHex;
	/**
	  * Defines if the Double Precision Floating Point Registers should be displayed in hexadecimal.
	  *
	*/
	private boolean DFPHex;
	/**
	  * Defines if invalid use of the $at register should be ignored.
	  *
	*/
	private boolean ignoreInvalidAT;

	private boolean registerWindowIsIcon;
	private boolean memoryWindowIsIcon;
	private boolean sourceWindowIsIcon;
	private boolean consoleWindowIsIcon;

	private int xCorrection;
	private int yCorrection;

// =========================================================================================
// Private convenience classes
// =========================================================================================

	// ===================================================================================
	// CLASS:	runThread
	// FUNCTION:	
	// ===================================================================================
	private class runThread extends Thread
	{
		public runThread(String name, boolean forward)
		{
			super(name);
			runForward = forward;
		}
		public void run()
		{
			while (shouldRun)
			{
				if (runForward)
				{
					performForwardStep(false);
				}
				else
				{
					performBackStep(false);
				}
			}
		}
		private boolean runForward;
	}

	// ===================================================================================
	// CLASS:		sourceLink
	// FUNCTION:	Used for maintaining a correlation between
	//			  a program counter value and a source line
	//			  number for purposes of highlighting the
	//			  current source line
	// ===================================================================================
	private class sourceLink
	{
		public sourceLink(String pc, int line)
		{
			this.pc = pc;
			this.line = line;
		}
		public String pc;
		public int line;
	}
}

