package JIMSCore;
/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    RegisterFile.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

  This class also defines the literal string token for each
  register, as well as a numerical index of each register.
  These are static so that they may be used from other
  classes without requiring an instance of the RegisterFile
  class.

PUBLIC METHODS

  public RegisterFile()
  public void reset()
  public int iGetReg(int i)
  public void setReg(int i, int value)
  public String toString()

*/

public class RegisterFile {

  public static final String REG_ID[][] =
	{
	  {"$0",  "$ZERO"},
	  {"$1",  "$AT"},
	  {"$2",  "$V0"},
	  {"$3",  "$V1"},
	  {"$4",  "$A0"},
	  {"$5",  "$A1"},
	  {"$6",  "$A2"},
	  {"$7",  "$A3"},
	  {"$8",  "$T0"},
	  {"$9",  "$T1"},
	  {"$10", "$T2"},
	  {"$11", "$T3"},
	  {"$12", "$T4"},
	  {"$13", "$T5"},
	  {"$14", "$T6"},
	  {"$15", "$T7"},
	  {"$16", "$S0"},
	  {"$17", "$S1"},
	  {"$18", "$S2"},
	  {"$19", "$S3"},
	  {"$20", "$S4"},
	  {"$21", "$S5"},
	  {"$22", "$S6"},
	  {"$23", "$S7"},
	  {"$24", "$T8"},
	  {"$25", "$T9"},
	  {"$26", "$K0"},
	  {"$27", "$K1"},
	  {"$28", "$GP"},
	  {"$29", "$SP"},
	  {"$30", "$FP"},
	  {"$31", "$RA"},
	// SPECIAL GPR REGISTERS
	  {"$32", "$PC"},
	  {"$33", "$HI"},
	  {"$34", "$LO"}
	};

  public static final int NUM_REGS = 35;

  // register constants
  public static final int R_ZERO = 0;
  public static final int R_AT = 1;
  public static final int R_V0 = 2;
  public static final int R_V1 = 3;
  public static final int R_A0 = 4;
  public static final int R_A1 = 5;
  public static final int R_A2 = 6;
  public static final int R_A3 = 7;
  public static final int R_T0 = 8;
  public static final int R_T1 = 9;
  public static final int R_T2 = 10;
  public static final int R_T3 = 11;
  public static final int R_T4 = 12;
  public static final int R_T5 = 13;
  public static final int R_T6 = 14;
  public static final int R_T7 = 15;
  public static final int R_S0 = 16;
  public static final int R_S1 = 17;
  public static final int R_S2 = 18;
  public static final int R_S3 = 19;
  public static final int R_S4 = 20;
  public static final int R_S5 = 21;
  public static final int R_S6 = 22;
  public static final int R_S7 = 23;
  public static final int R_T8 = 24;
  public static final int R_T9 = 25;
  public static final int R_K0 = 26;
  public static final int R_K1 = 27;
  public static final int R_GP = 28;
  public static final int R_SP = 29;
  public static final int R_FP = 30;
  public static final int R_RA = 31;

  // Special Internal Registers
  public static final int R_PC = 32;
  public static final int R_HI = 33;
  public static final int R_LO = 34;

/***************************************************************/

  //private int[] iaReg;
  public int[] iaReg;
  // iaReg is an integer array that holds the values of
  //   the registers.  It is initialized in the reset()
  //   method.  Use the R_XXX constants to reference
  //   the registers within the array.

  public RegisterFile() {

	reset();

  }    

  public void reset() {

	iaReg = new int[NUM_REGS];

	// It is assumed, by the nature of Java, that the
	//   values in the register array are all initialized
	//   to zero.

	iaReg[R_SP] = MainMemory.DEFAULT_STACK_ADDRESS;
	iaReg[R_PC] = MainMemory.DEFAULT_TEXT_ADDRESS;

  }  // end method reset()    

  public int iGetReg(int iReg) {
	if (iReg == R_ZERO)
	  return 0;
	return iaReg[iReg];
  }    

  public void setReg(int iReg, int iValue) {
	iaReg[iReg] = iValue;
  }    

  public String toString() {
	String result = "";
	int i;
	int cnt = 0;
	for (i = iaReg.length-1; i >= 0; i--) {
	  if (iaReg[i] != 0) {
		result += REG_ID[i]+"="+Utility.sAsHexPadded(iaReg[i],8)+"\t";
		cnt++;
		if (cnt % 3 == 0)
		  result += "\n";
	  }
	}
	return result;
  }    

  public static int iRegLiteralToID(String s) {

	// Returns -1 if the register literal is invalid,
	//   otherwise returns the index of the specified register literal.

	s = s.toUpperCase().trim();
	for (int i = 0; i < RegisterFile.REG_ID.length; i++) {
	  for (int x = 0; x < RegisterFile.REG_ID[i].length; x++) {
		if (s.equals(RegisterFile.REG_ID[i][x]))
		  return i;
	  }
	}
	return -1;

  }  // end method bRegLiteralToID(String)    

}