package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    MainMemory.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/
import java.util.Vector;

public class MainMemory {

  // Default addresses for various memory regions
  public static final int DEFAULT_TEXT_ADDRESS = 0x00400000;
  public static final int DEFAULT_DATA_ADDRESS = 0x10000000;
  public static final int DEFAULT_STACK_ADDRESS = 0x7FFFFFFF;
  public static final int DEFAULT_EXCEPTION_ADDRESS = 0x80000180;

  public static final int NUM_BUCKETS = 256;

  private AlphaMemoryBucket[] memoryBucket;

  public MainMemory() {
	reset();
  }    

  public void reset() {
	// initialize all alpha buckets to null

	if (memoryBucket != null) {
	  // The following loop is intended to help the Java
	  //   garbage collector to free up unused memory.
	  for (int i = 0; i < memoryBucket.length; i++) {
		if (memoryBucket[i] != null) {
		  memoryBucket[i].reset();
		  memoryBucket[i] = null;
		}
	  }
	}

	memoryBucket = new AlphaMemoryBucket[NUM_BUCKETS];
  }  // end method reset()    

  public void storeByte(int addr, byte value) {

	int a1 = (addr & 0xFF000000) >>> 24 ;

	if (memoryBucket[a1] == null) {
	  memoryBucket[a1] = new AlphaMemoryBucket();
	}

	int a2 = (addr & 0x00FF0000) >>> 16 ;
	int a3 = (addr & 0x0000FF00) >>> 8  ;
	int a4 = (addr & 0x000000FF)        ;

	memoryBucket[a1].storeByte(a2, a3, a4, value);

  }    

  public void storeHalfWord(int addr, int value) {
	
	byte value1 = (byte)( (value & 0x0000FF00L) >>> 8  );
	byte value2 = (byte)( (value & 0x000000FFL)        );

	storeByte(addr, value1);
	//storeByte(addr+1, value2);
	storeByte(++addr, value2);

  }    

  public void storeWord(int addr, int value) {

	// Break down the specified value into byte chunks.
	byte value1 = (byte)( (value & 0xFF000000L) >>> 24 );
	byte value2 = (byte)( (value & 0x00FF0000L) >>> 16 );
	byte value3 = (byte)( (value & 0x0000FF00L) >>> 8  );
	byte value4 = (byte)( (value & 0x000000FFL)        );

	// Store each byte at the corresponding address offset.
	storeByte(addr, value1);
	storeByte(++addr, value2);
	storeByte(++addr, value3);
	storeByte(++addr, value4);

  }    

  public byte loadByte(int addr) {

	int a1 = (addr & 0xFF000000) >>> 24 ;

	if (memoryBucket == null || memoryBucket[a1] == null) {
	  return 0;
	}

	int a2 = (addr & 0x00FF0000) >>> 16 ;
	int a3 = (addr & 0x0000FF00) >>> 8  ;
	int a4 = (addr & 0x000000FF)        ;

	return memoryBucket[a1].loadByte(a2, a3, a4);

  }    

  public int loadHalfWord(int addr) {
	// Load the word (4 bytes) starting at the specified address.
	int a1 = (loadByte(addr  ) <<  8) & 0x0000FF00;
	int a2 = (loadByte(addr+1)      ) & 0x000000FF;

	return (a1 | a2);
  }    

  public int loadWord(int addr) {

	// Load the word (4 bytes) starting at the specified address.
	int a1 = (loadByte(addr)   << 24) & 0xFF000000;
	int a2 = (loadByte(++addr) << 16) & 0x00FF0000;
	int a3 = (loadByte(++addr) <<  8) & 0x0000FF00;
	int a4 = (loadByte(++addr)      ) & 0x000000FF;

	return (a1 | a2 | a3 | a4);

  }    

  public Vector vGetModifiedAddresses() {
	Vector vResult = new Vector();

	if (memoryBucket != null) {
	  for (int a = 0; a < memoryBucket.length; a++) {
		if (memoryBucket[a] != null) {
		  if (memoryBucket[a].memoryBucket != null) {
			for (int b = 0; b < memoryBucket[a].memoryBucket.length; b++) {
			  if (memoryBucket[a].memoryBucket[b] != null) {
				if (memoryBucket[a].memoryBucket[b].memoryBucket != null) {
				  for (int c = 0; c < memoryBucket[a].memoryBucket[b].memoryBucket.length; c++) {
					if (memoryBucket[a].memoryBucket[b].memoryBucket[c] != null) {
					  for (int d = 0; d < memoryBucket[a].memoryBucket[b].memoryBucket[c].cell.length; d++) {
						if (memoryBucket[a].memoryBucket[b].memoryBucket[c].modified[d] > 0) {
						  int a1 = a << 24;
						  int a2 = b << 16;
						  int a3 = c << 8;
						  int iAddress = a1 | a2 | a3 | d;
						  vResult.add(new Integer(iAddress));
						}
					  }
					}
				  }
				}
			  }
			}
		  }
		}
	  }
	}

	return vResult;
  }    

  public StringBuffer sbGetMemoryDump(String s) {   
	StringBuffer sbResult = new StringBuffer();
	if (memoryBucket != null) {
	  for (int i = 0; i < memoryBucket.length; i++) {
		if (memoryBucket[i] != null) {
		  sbResult.append(
			memoryBucket[i].sbGetMemoryDump( s + Utility.sAsHexPadded(i,2) )
					   );
		}
	  }
	}
	return sbResult;
  }    

  public String toString() {
	return sbGetMemoryDump("$").toString();
  }    

}