package JIMSCore;

// Modified by Fred Williams, 2001
//

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    InstructionEncoder.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/
import java.util.StringTokenizer;

/**
  * This is the InstructionEncoder class.
  *
  * @author Steve Lewis
  * @author Fred Williams
*/
public class InstructionEncoder
{
	private Assembler assembler;

	public InstructionEncoder(Assembler assembler)
	{
		this.assembler = assembler;
	}

	public void processInstruction(String sInput, int iCurrLine) throws Exception
	{
		processInstruction("", sInput, iCurrLine);
	}

	public void processInstruction(String sRawInput, String sInput, int iCurrLine) throws Exception
	{
		/*
		INPUT
		sRawInput = unmodified input from source file
		sInput = modified (trimmed, comments removed)
		iCurrLine = corresponding line number of sRawInput
		*/

		String opCode = new StringTokenizer(sInput).nextToken();
		// Assume the line starts with some opCode instruction.

		String[] saParam = Assembler.saGetParams(sInput.substring(opCode.length()).trim());
		// Let saParam be a string array that holds each parameter of
		//   the instruction.

		opCode = opCode.toUpperCase();

		int x = sRawInput.indexOf(':');
		if (x > 0)
		{
			int y = sRawInput.indexOf('#');
			if ( (y == -1) || (y > x) )
			ModifiedSource.append(sRawInput.substring(0, x) + ":");
		}

		// The opCode is uppercased for easier string matching.
		int i = iEncodeInstruction(sRawInput, opCode, saParam, iCurrLine);

		if (i == 0)
		{
			x = sRawInput.indexOf('#');
			String s = "";
			if (x >= 0)
			{
				s = sRawInput.substring(x);
			}
			ModifiedSource.append("\t" + sInput + "\t" + s);
		}

	}
	// end method processInstruction(String, int)

	private int iEncodeInstruction(String sRawInput, String opCode, String[] saParam, int iCurrLine) throws Exception
	{
		// RETURNS:
		//   0 if this instruction is encoded as-is
		//   1 if this instruction is replaced by a different instruction
		//  >1 if this instruction is replaced by several different instructions

		int iResult = 0;

		// Prepare a new text entry.  If this is a pseudo instruction,
		//   this won't be used.  But for most instructions, it will be
		//   added to the current TextSection.
		TextEntry textEntry = new TextEntry(iCurrLine);
		textEntry.op = opCode;

		if (opCode.equals("MFHI"))
		{
			// MFHI rd
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-rd-00-10";
			textEntry.formatLen = "6-10-5-5-6";
			textEntry.rd = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("MFLO"))
		{
			// MFLO rd
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-rd-00-12";
			textEntry.formatLen = "6-10-5-5-6";
			textEntry.rd = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("MTHI"))
		{
			// MTHI rs
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-00-11";
			textEntry.formatLen = "6-5-15-6";
			textEntry.rs = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("MTLO"))
		{
			// MTLO rs
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-00-13";
			textEntry.formatLen = "6-5-15-6";
			textEntry.rs = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("RFE"))
		{
			// RFE
			if (saParam.length != 0)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "10-01-00-10";
			textEntry.formatLen = "6-1-19-6";
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("SYSCALL"))
		{
			// SYSCALL
			if (saParam.length != 0)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-0C";
			textEntry.formatLen = "6-20-6";
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("BREAK"))
		{
			// BREAK code
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-code-0D";
			textEntry.formatLen = "6-20-6";
			textEntry.code = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("J"))
		{
			// J target
			textEntry.format = "02-target";
			textEntry.formatLen = "6-26";
			textEntry.target = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("JAL"))
		{
			// JAL target
			textEntry.format = "03-target";
			textEntry.formatLen = "6-26";
			textEntry.target = saParam[0];
			assembler.addToTextVector(textEntry);
		}

		else if (opCode.equals("JR"))
		{
			// JR rs
			String rs = saParam[0];		
			if (rs.indexOf('$') < 0)
			{
				// No register was specified, assume the user meant
				//   to use J instead.
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "j " + rs, iCurrLine);
				iResult++;

			}
			else
			{
				textEntry.format = "00-rs-00-08";
				textEntry.formatLen = "6-5-15-6";
				textEntry.rs = saParam[0];
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("DIV"))
		{
			if (saParam.length == 3)
			{
				// DIV rdest, rsrc1, src2
				iResult += iProcessAsPseudoInstruction(sRawInput, opCode, saParam, iCurrLine);
			}
			else if (saParam.length == 2)
			{
				// DIV rs, rt
				textEntry.format = "00-rs-rt-00-1A";
				textEntry.formatLen = "6-5-5-10-6";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				assembler.addToTextVector(textEntry);
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
		}
		else if (opCode.equals("DIVU"))
		{
			if (saParam.length == 3)
			{
				// DIVU rdest, rsrc1, src2
				iResult += iProcessAsPseudoInstruction(sRawInput, opCode, saParam, iCurrLine);
			}
			else if (saParam.length == 2)
			{
				// DIVU rs, rt
				textEntry.format = "00-rs-rt-00-1B";
				textEntry.formatLen = "6-5-5-10-6";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				assembler.addToTextVector(textEntry);
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
		}
		else if (opCode.equals("MULT"))
		{
			// MULT rs, rt
			if (saParam.length == 2)
			{
				textEntry.format = "00-rs-rt-00-18";
				textEntry.formatLen = "6-5-5-10-6";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				assembler.addToTextVector(textEntry);
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
		}

		else
		if (opCode.equals("MULTU"))
		{
			// MULTU rs, rt
			if (saParam.length == 2)
			{
				textEntry.format = "00-rs-rt-00-19";
				textEntry.formatLen = "6-5-5-10-6";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				assembler.addToTextVector(textEntry);
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
		}
		else if (opCode.equals("ADDI"))
		{
			// ADDI rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "08-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("ADDIU"))
		{
			// ADDIU rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "09-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("ANDI"))
		{
			// ANDI rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0C-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("ORI"))
		{
			// ORI rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0D-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("XORI"))
		{
			// XORI rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0E-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("LUI"))
		{
			// LUI rt, imm
			String t;
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0F-00-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLTI"))
		{
			// SLTI rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0A-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLTIU"))
		{
			// SLTIU rt, rs, imm
			String t, s, imm;
			if (saParam.length == 2)
			{
				t = saParam[0];
				s = saParam[0];
				imm = saParam[1];
			}
			else if (saParam.length == 3)
			{
				t = saParam[0];
				s = saParam[1];
				imm = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "0B-rs-rt-imm";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rt = t;
			textEntry.rs = s;
			textEntry.addr_imm = imm;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BC0T"))
		{
			// BC0T label
			textEntry.format = "10-8-1-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.addr_imm = saParam[0];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BC1T"))
		{
			// BC1T label
			textEntry.format = "11-8-1-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.addr_imm = saParam[0];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BC0F"))
		{
			// BC0F label
			textEntry.format = "10-8-0-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.addr_imm = saParam[0];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BC1F"))
		{
			// BC1F label
			textEntry.format = "11-8-0-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.addr_imm = saParam[0];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BEQ"))
		{
			// BEQ rs, rt, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (saParam[1].indexOf('$') < 0)
			{
				// SYNTHETIC INSTRUCTION
				// beq rs, imm, label
				storeAT();
				processInstruction("# " + sRawInput, "ori $at, $zero, "+saParam[1], iCurrLine);
				processInstruction("beq $at, "+saParam[0]+", "+saParam[2], iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				textEntry.format = "04-rs-rt-branch";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				textEntry.addr_imm = saParam[2];
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("BGEZ"))
		{
			// BGEZ rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "01-rs-01-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BGEZAL"))
		{
			// BGEZAL rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "01-rs-11-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BGTZ"))
		{
			// BGTZ rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "07-rs-00-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BLEZ"))
		{
			// BLEZ rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "06-rs-00-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BLTZAL"))
		{
			// BLTZAL rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "01-rs-10-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BLTZ"))
		{
			// BLTZ rs, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "01-rs-00-branch";
			textEntry.formatLen = "6-5-5-16";
			textEntry.rs = saParam[0];
			textEntry.addr_imm = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("BNE"))
		{
			// BNE rs, rt, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (saParam[1].indexOf('$') < 0)
			{
				// SYNTHETIC INSTRUCTION
				// bne rs, imm, label
				storeAT();
				processInstruction("# " + sRawInput, "ori $at, $zero, "+saParam[1], iCurrLine);
				processInstruction("bne $at, "+saParam[0]+", "+saParam[2], iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
				} else {
				textEntry.format = "05-rs-rt-branch";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saParam[0];
				textEntry.rt = saParam[1];
				textEntry.addr_imm = saParam[2];
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LB"))
		{
			// LB rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lb rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lb "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "20-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LBU"))
		{
			// LBU rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lbu rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lbu "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "24-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LH"))
		{
			// LH rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lh rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lh "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "21-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LHU"))
		{
			// LHU rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lhu rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lhu "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "25-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LW"))
		{
			// LW rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lw rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lw "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "23-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LWC0"))
		{
			// LWC0 rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lwc0 rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lwc0 "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "30-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LWC1"))
		{
			// LWC1 rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lwc1 rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lwc1 "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "31-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LWL"))
		{
			// LWL rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lwl rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lwl "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "22-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("LWR"))
		{
			// LWR rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// lwr rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("lwr "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "26-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SB"))
		{
			// SB rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// sb rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("sb "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "28-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SH"))
		{
			// SH rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// sh rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("sh "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "29-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SW"))
		{
			// SW rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// sw rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("sw "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "2B-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SWC0"))
		{
			// SWC0 rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// swc0 rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("swc0 "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "38-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SWC1"))
		{
			// SWC1 rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// swc1 rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("swc1 "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "39-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}

// **INCOMPLETE
		// SWC2: 0x3a-rs-rt-offset  z == 2
		// SWC3: 0x3b-rs-rt-offset  z == 3
// ** END INCOMPLETE

		else if (opCode.equals("SWL"))
		{
			// SWL rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// swl rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("swl "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "2A-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SWR"))
		{
			// SWR rt, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String[] saAddressInfo = saProcessAddress(saParam[1]);
			if (saAddressInfo[0].equals(""))
			{
				// SYNTHETIC INSTRUCTION
				// swr rt, address
				storeAT();
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("swr "+saParam[0]+", lo16("+saAddressInfo[1]+")($at)", iCurrLine);
				restoreAT(saParam[0], iCurrLine);
				iResult += 2;
			}
			else
			{
				// assume ofs is a 16-bit value as it should be
				textEntry.format = "2E-rs-rt-offset";
				textEntry.formatLen = "6-5-5-16";
				textEntry.rs = saAddressInfo[0];  // address reg
				textEntry.rt = saParam[0];
				textEntry.addr_imm = saAddressInfo[1];  // address offset, or numeric literal
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("MFC0"))
		{
			// MFC0 rt, rd
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "10-00-rt-rd-00";
			textEntry.formatLen = "6-5-5-5-11";
			textEntry.rt = saParam[0];
			textEntry.rd = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("MFC1"))
		{
			// MFC1 rt, rd
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-00-rt-rd-00";
			textEntry.formatLen = "6-5-5-5-11";
			textEntry.rt = saParam[0];
			textEntry.rd = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("MTC0"))
		{
			// MTC0 rd, rt
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "10-04-rt-rd-00";
			textEntry.formatLen = "6-5-5-5-11";
			textEntry.rt = saParam[0];
			textEntry.rd = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("MTC1"))
		{
			// MTC1 rd, rt
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-04-rt-rd-00";
			textEntry.formatLen = "6-5-5-5-11";
			textEntry.rt = saParam[0];
			textEntry.rd = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.EQ.D"))
		{
			// C.EQ.D fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-11-ft-fs-00-3-02";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.EQ.S"))
		{
			// C.EQ.S fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-10-ft-fs-00-3-02";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.LE.D"))
		{
			// C.LE.D fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-11-ft-fs-00-3-0E";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.LE.S"))
		{
			// C.LE.S fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-10-ft-fs-00-3-0E";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.LT.D"))
		{
			// C.LT.D fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-11-ft-fs-00-3-0C";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("C.LT.S"))
		{
			// C.LT.S fs, ft
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "11-10-ft-fs-00-3-0C";
			textEntry.formatLen = "6-5-5-5-5-2-4";
			textEntry.fs = saParam[0];
			textEntry.ft = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("ADD"))
		{
			// ADD rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (t.indexOf('$') < 0)
			{
				// The third parameter does not specify a register,
				//   assume that it is a literal numeric.  Use ADDI instead.
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "addi "+d+", "+s+", "+t, iCurrLine);
				iResult++;
			}
			else
			{
				textEntry.format = "00-rs-rt-rd-00-20";
				textEntry.formatLen = "6-5-5-5-5-6";
				textEntry.rd = d;
				textEntry.rs = s;
				textEntry.rt = t;
				assembler.addToTextVector(textEntry);
			}
		}
		else
		if (opCode.equals("ADDU"))
		{
			// ADDU rd, rs, rt

			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
				} else {
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}

			if (t.indexOf('$') < 0)
			{
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "addiu "+d+", "+s+", "+t, iCurrLine);
				iResult++;
				} else {
				textEntry.format = "00-rs-rt-rd-00-21";
				textEntry.formatLen = "6-5-5-5-5-6";
				textEntry.rd = d;
				textEntry.rs = s;
				textEntry.rt = t;

				assembler.addToTextVector(textEntry);
			}

		}
		else if (opCode.equals("AND"))
		{
			// AND rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (t.indexOf('$') < 0)
			{
				// The third parameter is a literal numeric, assume
				//   that the user meant to use andi instead.
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "andi "+d+","+s+","+t, iCurrLine);
				iResult++;
			}
			else
			{
				textEntry.format = "00-rs-rt-rd-00-24";
				textEntry.formatLen = "6-5-5-5-5-6";
				textEntry.rd = d;
				textEntry.rs = s;
				textEntry.rt = t;
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("NOR"))
		{
			// NOR rd, rs, rt

			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-27";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("OR"))
		{
			// OR rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-25";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLL"))
		{
			// SLL rd, rt, shamt
			String d, t, shamt;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				shamt = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				shamt = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-rt-rd-shamt-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rt = t;
			textEntry.shamt = shamt;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLLV"))
		{
			// SLLV rd, rt, rs
			String d, t, s;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				s = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				s = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-04";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SRA"))
		{
			// SRA rd, rt, shamt
			String d, t, shamt;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				shamt = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				shamt = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-rt-rd-shamt-03";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rt = t;
			textEntry.shamt = shamt;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SRAV"))
		{
			// SRAV rd, rt, rs
			String d, t, s;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				s = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				s = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-07";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SRL"))
		{
			// SRL rd, rt, shamt
			String d, t, shamt;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				shamt = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				shamt = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-00-rt-rd-shamt-02";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rt = t;
			textEntry.shamt = shamt;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SRLV"))
		{
			// SRLV rd, rt, rs
			String d, t, s;
			if (saParam.length == 2)
			{
				d = saParam[0];
				t = saParam[0];
				s = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				t = saParam[1];
				s = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-06";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SUB"))
		{
			// SUB rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (t.indexOf('$') < 0)
			{
				// The third parameter does not specify a register,
				//   assume that it is a literal numeric.  Use ADDI instead.
				long l = Utility.IntegerLiteralToLong(t);
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "addi "+d+", "+s+", "+(-l), iCurrLine);
				iResult++;
			}
			else
			{
				textEntry.format = "00-rs-rt-rd-00-22";
				textEntry.formatLen = "6-5-5-5-5-6";
				textEntry.rd = d;
				textEntry.rs = s;
				textEntry.rt = t;
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("SUBU"))
		{
			// SUBU rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			if (t.indexOf('$') < 0)
			{
				long l = Utility.IntegerLiteralToLong(t);
				// INSTRUCTION TRANSLATION
				processInstruction("# " + sRawInput, "addiu "+d+", "+s+", "+(-l), iCurrLine);
				iResult++;
			}
			else
			{
				textEntry.format = "00-rs-rt-rd-00-23";
				textEntry.formatLen = "6-5-5-5-5-6";
				textEntry.rd = d;
				textEntry.rs = s;
				textEntry.rt = t;
				assembler.addToTextVector(textEntry);
			}
		}
		else if (opCode.equals("XOR"))
		{
			// XOR rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-26";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLT"))
		{
			// SLT rd, rs, rt
			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
			}
			else if (saParam.length == 3)
			{
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}
			else
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			textEntry.format = "00-rs-rt-rd-00-2A";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;
			assembler.addToTextVector(textEntry);
		}
		else if (opCode.equals("SLTU"))
		{
			// SLTU rd, rs, rt

			String d, s, t;
			if (saParam.length == 2)
			{
				d = saParam[0];
				s = saParam[0];
				t = saParam[1];
				} else {
				d = saParam[0];
				s = saParam[1];
				t = saParam[2];
			}

			textEntry.format = "00-rs-rt-rd-00-2B";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rd = d;
			textEntry.rs = s;
			textEntry.rt = t;

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("JALR"))
		{
			// JALR rs, rd

			textEntry.format = "00-rs-00-rd-00-09";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.rs = saParam[0];

			if (saParam.length == 1)
			{
				textEntry.rd = "$31";
				} else {
				textEntry.rd = saParam[1];
			}

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("NOP"))
		{
			// NOP

			textEntry.format = "00-00-00-00-00-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			assembler.addToTextVector(textEntry);

		}
		else

// *** FLOATING POINT INSTRUCTIONS ***

		if (opCode.equals("ABS.D"))
		{
			// ABS.D fd, fs

			textEntry.format = "11-11-00-fs-fd-05";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("ABS.S"))
		{
			// ABS.S fd, fs

			textEntry.format = "11-10-00-fs-fd-05";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("ADD.D"))
		{
			// ADD.D fd, fs, ft

			textEntry.format = "11-11-ft-fs-fd-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("ADD.S"))
		{
			// ADD.S fd, fs, ft

			textEntry.format = "11-10-ft-fs-fd-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.D.S"))
		{
			// CVT.D.S fd, fs

			textEntry.format = "11-11-00-fs-fd-21";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.D.W"))
		{
			// CVT.D.W fd, fs

			textEntry.format = "11-10-00-fs-fd-21";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.S.D"))
		{
			// CVT.S.D fd, fs

			textEntry.format = "11-11-00-fs-fd-20";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.S.W"))
		{
			// CVT.S.W fd, fs

			textEntry.format = "11-10-00-fs-fd-20";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.W.D"))
		{
			// CVT.W.D fd, fs

			textEntry.format = "11-11-00-fs-fd-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("CVT.W.S"))
		{
			// CVT.W.S fd, fs

			textEntry.format = "11-10-00-fs-fd-24";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("DIV.D"))
		{
			// DIV.D fd, fs, ft

			textEntry.format = "11-11-ft-fs-fd-03";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("DIV.S"))
		{
			// DIV.S fd, fs, ft

			textEntry.format = "11-10-ft-fs-fd-03";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("MOV.D"))
		{
			// MOV.D fd, fs

			textEntry.format = "11-11-00-fs-fd-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("MOV.S"))
		{
			// MOV.S fd, fs

			textEntry.format = "11-10-00-fs-fd-00";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("MUL.D"))
		{
			// MUL.D fd, fs, ft

			textEntry.format = "11-11-ft-fs-fd-02";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("MUL.S"))
		{
			// MUL.S fd, fs, ft

			textEntry.format = "11-10-ft-fs-fd-02";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("NEG.D"))
		{
			// NEG.D fd, fs

			textEntry.format = "11-11-00-fs-fd-07";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];

			assembler.addToTextVector(textEntry);

		}
		else
		if (opCode.equals("NEG.S"))
		{
			// NEG.S fd, fs
			textEntry.format = "11-10-00-fs-fd-07";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			assembler.addToTextVector(textEntry);
		}
		else
		if (opCode.equals("SUB.D"))
		{
			// SUB.D fd, fs, ft
			textEntry.format = "11-11-ft-fs-fd-01";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];
			assembler.addToTextVector(textEntry);
		}
		else
		if (opCode.equals("SUB.S"))
		{
			// SUB.S fd, fs, ft

			textEntry.format = "11-10-ft-fs-fd-01";
			textEntry.formatLen = "6-5-5-5-5-6";
			textEntry.fd = saParam[0];
			textEntry.fs = saParam[1];
			textEntry.ft = saParam[2];
			assembler.addToTextVector(textEntry);
		}
		else
		{
			iResult += iProcessAsPseudoInstruction(sRawInput, opCode, saParam, iCurrLine);
		}
		validateRegisterUse(textEntry, sRawInput, iCurrLine);
		return iResult;
	}
	// end method encodeInstruction(String, String, String[], int)

// ***********************************************************
// *** PSEUDO INSTRUCTIONS ***********************************
// ***********************************************************
	private int iProcessAsPseudoInstruction(String sRawInput, String opCode, String[] saParam, int iCurrLine) throws Exception
	{
		int iResult = 0;

		// Temporarily disable warning about the use of $at.
		storeAT();
		// This is re-enabled to the old value at the end of this method.

		if (opCode.equals("DIV"))
		{
			// DIV fd, fs, ft
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}			
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			if (t.indexOf('$') < 0)
			{
				// Assume t is an immediate value, not a register.
				processInstruction("# " + sRawInput, "ori $at, $zero, " + t, iCurrLine);
				processInstruction("div "+s+", $at", iCurrLine);
				processInstruction("mflo "+d, iCurrLine);
				iResult += 3;
			}
			else
			{
				processInstruction("# " + sRawInput, "div "+s+","+t, iCurrLine);
				processInstruction("bne "+t+",$zero,8", iCurrLine);
				processInstruction("break 7", iCurrLine);
				processInstruction("li $at, -1", iCurrLine);
				processInstruction("bne "+t+", $at, 16", iCurrLine);
				processInstruction("lui $at, 0x8000", iCurrLine);
				processInstruction("bne "+s+", $at, 8", iCurrLine);
				processInstruction("break 6", iCurrLine);
				processInstruction("mflo "+d, iCurrLine);
				iResult += 9;
			}
		}
		else if (opCode.equals("DIVU"))
		{
			// DIVU fd, fs, ft
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "divu "+s+", "+t, iCurrLine);
			processInstruction("bne "+t+", $zero, 8", iCurrLine);
			processInstruction("break 7", iCurrLine);
			processInstruction("mflo "+d, iCurrLine);
			iResult += 4;
		}
		else if (opCode.equals("ABS"))
		{
			// ABS rdest, rsrc
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			processInstruction("# " + sRawInput, "addu "+d+", $zero, "+s, iCurrLine);
			processInstruction("bgez "+s+", 8", 0);
			processInstruction("sub "+d+", $zero, "+s, 0);
			iResult += 3;
		}
		else if (opCode.equals("MUL"))
		{
			// MUL rdest, rsrc1, src2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "mult " + s + ", " + t, iCurrLine);
			processInstruction("mflo " + d, 0);
			iResult += 2;
		}
		else if (opCode.equals("MULO"))
		{
			// MULO rdest, rsrc1, src2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "mult "+s+", "+t, iCurrLine);
			processInstruction("mflo " + d, 0);
			processInstruction("sra "+d+", "+d+",31", 0);
			processInstruction("mfhi $at", 0);
			processInstruction("beq "+d+", $at, 8", 0);
			processInstruction("break 6", 0);
			processInstruction("mflo "+d, 0);
			iResult += 7;
		}
		else if (opCode.equals("MULOU"))
		{
			// MULOU rdest, rsrc1, src2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "multu "+s+", "+t, iCurrLine);
			processInstruction("mfhi $at", 0);
			processInstruction("mflo " + d, 0);
			processInstruction("beq $at, $zero, 8", 0);
			processInstruction("break 6", 0);
			iResult += 5;
		}
 		else if (opCode.equals("NEG"))
		{
			// NEG rdest, rscr
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			processInstruction("# " + sRawInput, "sub "+d+", $zero, "+s, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("NEGU"))
		{
			// NEGU rdest, rsrc
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			processInstruction("# " + sRawInput, "subu "+d+", $zero, "+s, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("NOT"))
		{
			// NOT rdest, rsrc
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}			String d = saParam[0];
			String s = saParam[1];
			processInstruction("# " + sRawInput, "nor "+d+", "+s+", $zero", iCurrLine);
			iResult++;
		}
		else
//    if (opCode.equals("REM")) {
// **INCOMPLETE
		// REM rdest, rsrc1, rsrc2
		// psuedo
		// Remainder

		// rem d, s, t ->
		//   bnez t, 1f
		//   div $zero, s, t
		//   break 7
		// 1:
		//   li at,-1
		//   bne t, at, 2f
		//   lui at,32768
		//   bne s, at, 2f
		//   nop
		//   break 6
		// 2:
		//   mfhi d

//    } else
//    if (opCode.equals("REMU")) {
// **INCOMPLETE
		// REMU rdest, rsrc1, rsrc2
		// psuedo
		// Unsigned remainder

		// remu d, s, t ->
		//   bnez t, 1f
		//   divu $zero,s,t
		//   break 7
		// 1:
		//   mfhi d

//      String d = saParam[0];
//      String s = saParam[1];
//      String t = saParam[2];

//    } else
		if (opCode.equals("ROL"))
		{
			// ROL rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "negu $at, "+t, iCurrLine);
			processInstruction("srlv $at, "+s+",$at", 0);
			processInstruction("sllv "+d+", "+s+", "+t, 0);
			processInstruction("or "+d+", "+d+", $at", 0);
			iResult += 4;
		}
		else if (opCode.equals("ROR"))
		{
			// ROR rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "negu $at, "+t, iCurrLine);
			processInstruction("sllv $at, "+s+",$at", 0);
			processInstruction("srlv "+d+", "+s+", "+t, 0);
			processInstruction("or "+d+", "+d+", $at", 0);
			iResult += 4;
		}

		else if (opCode.equals("LI"))
		{
			// LI rdest, imm
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String j = saParam[1];
			long l = Utility.IntegerLiteralToLong(j);
			if ( (l >= 0) && (l <= 65535) )
			{
				processInstruction("# " + sRawInput, "ori "+d+", $zero, "+j, iCurrLine);
				iResult++;
			}
			else if ( (l >= -32768) && (l < 0) )
			{
				processInstruction("# " + sRawInput, "addiu "+d+", $zero, "+j, iCurrLine);
				iResult++;
			}
			else
			{
				processInstruction("# " + sRawInput, "lui "+d+", " + ((l & 0xFFFF0000) >>> 16), iCurrLine);
				processInstruction("ori "+d+", " + (l & 0x0000FFFF), 0);
				iResult += 2;
			}
		}
		else if (opCode.equals("SEQ"))
		{
			// SEQ rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			if (t.indexOf('$') < 0)
			{
				// Assume t is actually an immediate value
				processInstruction("# " + sRawInput, "seqi "+d+", "+s+", "+t, iCurrLine);
				iResult++;
			}
			else
			{
				processInstruction("# " + sRawInput, "beq "+t+", "+s+", 12", iCurrLine);
				processInstruction("ori "+d+", $zero, 0", 0);
				processInstruction("beq $zero, $zero, 8", 0);
				processInstruction("ori "+d+", $zero, 1", 0);
				iResult += 4;
			}
		}
		else if (opCode.equals("SEQI"))
		{
			// SEQI rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "ori $at, $zero, " + t, iCurrLine);
			processInstruction("beq $at, "+s+", 12", 0);
			processInstruction("ori "+d+", $zero, 0", 0);
			processInstruction("beq $zero, $zero, 8", 0);
			processInstruction("ori "+d+", $zero, 1", 0);
			iResult += 5;
		}
		else if (opCode.equals("SGE"))
		{
			// SGE rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "slt "+d+", "+s+", "+t, iCurrLine);
			processInstruction("xori "+d+", "+d+", 1", 0);
			iResult += 2;
		}
		else if (opCode.equals("SGEU"))
		{
			// SGEU rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "sltu "+d+", "+s+", "+t, iCurrLine);
			processInstruction("xori "+d+", "+d+", 1", 0);
			iResult += 2;
		}
		else if (opCode.equals("SGT"))
		{
			// SGT rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			if (t.indexOf('$') < 0)
			{
				processInstruction("# " + sRawInput, "ori $at, $zero, "+t, iCurrLine);
				processInstruction("slt "+d+", $at, "+s, iCurrLine);
				iResult += 2;
			}
			else
			{
				processInstruction("# " + sRawInput, "slt "+d+", "+t+", "+s, iCurrLine);
				iResult++;
			}
		}
		else if (opCode.equals("SGTU"))
		{
			// SGTU rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "sltu "+d+", "+t+", "+s, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("SLE"))
		{
			// SLE rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "slt "+d+","+t+","+s, iCurrLine);
			processInstruction("xori "+d+","+d+",1", 0);
			iResult += 2;
		}
		else if (opCode.equals("SLEU"))
		{
			// SLEU rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "sltu "+d+", "+t+", "+s, iCurrLine);
			processInstruction("xori "+d+", "+d+", 1", 0);
			iResult += 2;
		}
		else if (opCode.equals("SNE"))
		{
			// SNE rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			if (t.indexOf('$') < 0)
			{
				// Assume t is actually an immediate value
				processInstruction("# " + sRawInput, "snei "+d+", "+s+", "+t, iCurrLine);
				iResult++;
			}
			else
			{
				processInstruction("# " + sRawInput, "beq "+t+", "+s+", 12", iCurrLine);
				processInstruction("ori "+d+", $zero, 1", 0);
				processInstruction("beq $zero, $zero, 8", 0);
				processInstruction("ori "+d+", $zero, 0", 0);
				iResult += 4;
			}
		}
		else if (opCode.equals("SNEI"))
		{
			// SNEI rdest, rsrc1, rsrc2
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String d = saParam[0];
			String s = saParam[1];
			String t = saParam[2];
			processInstruction("# " + sRawInput, "ori $at, $zero, " + t, iCurrLine);
			processInstruction("beq $at, "+s+", 12", 0);
			processInstruction("ori "+d+", $zero, 1", 0);
			processInstruction("beq $zero, $zero, 8", 0);
			processInstruction("ori "+d+", $zero, 0", 0);
			iResult += 5;
		}
		else if (opCode.equals("B"))
		{
			// B label
			if (saParam.length != 1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String label = saParam[0];
			processInstruction("# " + sRawInput, "beq $zero, $zero, "+label, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("BEQZ"))
		{
			// BEQZ rsrc, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String label = saParam[1];
			processInstruction("# " + sRawInput, "beq "+s+", $zero, "+label, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("BGE"))
		{
			// BGE rsrc1, rsrc2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "slt $at, "+s+", "+t, iCurrLine);
			processInstruction("beq $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BGEU"))
		{
			// BGEU rsrc1, rsrc2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "sltu $at, "+s+", "+t, iCurrLine);
			processInstruction("beq $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BGT"))
		{
			// BGT rsrc1, src2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "slt $at, "+t+", "+s, iCurrLine);
			processInstruction("bne $at, $zero, "+label, iCurrLine);
			iResult += 2;
		}
		else if (opCode.equals("BGTU"))
		{
			// BGTU rsrc1, src2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "slt $at, "+t+", "+s, iCurrLine);
			processInstruction("beq $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BLE"))
		{
			// BLE rsrc1, src2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "slt $at, "+t+", "+s, iCurrLine);
			processInstruction("beq $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BLEU"))
		{
			// BLEU rsrc1, src2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "sltu $at, "+t+", "+s, iCurrLine);
			processInstruction("beq $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BLT"))
		{
			// BLT rsrc1, rsrc2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "slt $at, "+s+", "+t, iCurrLine);
			processInstruction("bne $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BLTU"))
		{
			// BLTU rsrc1, rsrc2, label
			if (saParam.length != 3)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String t = saParam[1];
			String label = saParam[2];
			processInstruction("# " + sRawInput, "sltu $at, "+s+", "+t, iCurrLine);
			processInstruction("bne $at, $zero, "+label, 0);
			iResult += 2;
		}
		else if (opCode.equals("BNEZ"))
		{
			// BNEZ rsrc, label
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String s = saParam[0];
			String label = saParam[1];
			processInstruction("# " + sRawInput, "bne "+s+", $zero, "+label, iCurrLine);
			iResult++;
		}
		else if (opCode.equals("LA"))
		{
			// LA rdest, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			String rd = saParam[0];
			String address = saParam[1].trim();
			String[] saAddressInfo = saProcessAddress(address);
			if (saAddressInfo[0].equals(""))
			{
				processInstruction("# " + sRawInput, "lui $at, hi16("+address+")", iCurrLine);
				processInstruction("ori "+rd+", $at, lo16("+address+")", 0);
				iResult += 2;
			}
			else
			{
				processInstruction("# " + sRawInput, "lui $at, hi16("+saAddressInfo[1]+")", iCurrLine);
				processInstruction("ori $at, $at, lo16("+saAddressInfo[1]+")", 0);
				processInstruction("add "+rd+", "+saAddressInfo[0]+",$at", 0);
				iResult += 3;
			}
		}

// **INCOMPLETE
// else if (opCode.equals("LD")) {
	// LD rdest, address
	// psuedo
	// Load doubleword
//    } else
//    if (opCode.equals("ULH")) {
		// ULH rdest, address
		// psuedo
		// Unaligned load halfword
//    } else
//    if (opCode.equals("ULHU")) {
		// ULHU rdest, address
		// psuedo
		// Unaligned load halfword unsigned
//    } else
//    if (opCode.equals("ULW")) {
		// ULW rdest, address
		// psuedo
		// Unaligned load word
//    } else
//    if (opCode.equals("SD")) {
		// SD rsrc, address
		// psuedo
		// Store doubleword
//    } else
//    if (opCode.equals("USH")) {
		// USH rsrc, address
		// psuedo
		// Unaligned store halfword
//    } else
//    if (opCode.equals("USW")) {
		// USW rsrc, address
		// psuedo
		// Unaligned store word
//    } else

		else if (opCode.equals("MOVE"))
		{
			// MOVE rdest, rsrc
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.\n\n" + sRawInput.trim());
			}
			String d = saParam[0];
			String s = saParam[1];
			processInstruction("# " + sRawInput, "addu "+d+", $zero, "+s, iCurrLine);
			iResult++;
		}
		else
//    if (opCode.equals("MFC1.D")) {
		// MFC1.D rdest, frsrc1
		// psuedo
		// Move double from coprocessor 1
//    } else
		if (opCode.equals("L.D"))
		{
			// L.D fdest, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			int iReg = Coprocessor1.iRegLiteralToID(saParam[0]);
			processInstruction("# " + sRawInput, "lui $at, hi16("+saParam[1]+")", iCurrLine);
			processInstruction("lwc1 "+saParam[0]+", lo16("+saParam[1]+")($at)", 0);
			processInstruction("lui $at, hi16(+4|"+saParam[1]+")", iCurrLine);
			processInstruction("lwc1 $"+(iReg+1)+", lo16(+4|"+saParam[1]+")($at)", 0);
			iResult += 4;
		}

		else if (opCode.equals("L.S"))
		{
			// L.S fdest, address
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			processInstruction("# " + sRawInput, "lui $at, hi16("+saParam[1]+")", iCurrLine);
			processInstruction("lwc1 "+saParam[0]+", lo16("+saParam[1]+")($at)", 0);
			iResult += 2;
		}
		else if (opCode.equals("S.D"))
		{
			// S.D fdest, address	
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}
			else
			{
				int iReg = Coprocessor1.iRegLiteralToID(saParam[0]);
				processInstruction("# " + sRawInput, "lui $at, hi16("+saParam[1]+")", iCurrLine);
				processInstruction("swc1 "+saParam[0]+", lo16("+saParam[1]+")($at)", 0);
				processInstruction("lui $at, hi16(+4|"+saParam[1]+")", iCurrLine);
				processInstruction("swc1 $"+(iReg+1)+", lo16(+4|"+saParam[1]+")($at)", 0);
				iResult += 4;
			}

		}
		else if (opCode.equals("S.S"))
		{
			// S.S fdest, address
			// psuedo
			// Store floating-point single
			if (saParam.length != 2)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid number of operands.");
			}			processInstruction("# " + sRawInput, "lui $at, hi16("+saParam[1]+")", iCurrLine);
			processInstruction("swc1 "+saParam[0]+", lo16("+saParam[1]+")($at)", 0);
			iResult += 2;
		}
		else
		{
			throw new Exception("Line " + iCurrLine + "\n" + "Unsupported opcode: " + opCode);
		}
		restoreAT("", iCurrLine);
		return iResult;
	}
	// end method iProcessAsPseudoInstruction(String, String, String[], int)

	private String[] saProcessAddress(String s)
	{
		/*
		Format of s (address used in opcode):
		label
		ofs(reg)

		e.g
		lw $t0, x1
		lw $t1, 4($t0)
		*/

		String[] result = new String[2];

		// result[0] == register value | blank
		// result[1] == offset value   | label name

		int i = s.lastIndexOf("(");
		if (i >= 0)
		{
			// ofs(reg)
			int x = s.lastIndexOf(")");
			result[0] = s.substring(i+1,x);
			result[1] = s.substring(0,i);
		}
		else
		{
			// assume just a label
			result[0] = "";
			result[1] = s;
		}
		return result;
	}
	// end method saProcessAddress(String)

	private void storeAT()
	{
		// Store the current value of the WarnAboutAT flag and
		//   temporarily disable the WarnAboutAT flag.
		DirectiveProcessor.bOldWarnAboutAT = DirectiveProcessor.bWarnAboutAT;
		DirectiveProcessor.bWarnAboutAT = false;
	}

	private void restoreAT(String s, int iSourceLine)
	{
		// Restore the saved value of WarnAboutAT flag.
		DirectiveProcessor.bWarnAboutAT = DirectiveProcessor.bOldWarnAboutAT;

		// If string s specifies the "$at" register, then produce
		//   a warning of the WarnAboutAT flag is enabled.
		if ( DirectiveProcessor.bWarnAboutAT && Assembler.isAT(s) )
		{
			Assembler.warnAboutUseOfAT(iSourceLine);
		}
	}
/**
  * Validates the use of registers as operands.
  *
  * @param te The TextEntry to valdiate.
*/
	private void validateRegisterUse(TextEntry te, String sRawInput, int iCurrLine) throws Exception
	{
		// ************************************************************************
		// Determines whether general purpose register operands were used properly.
		// ************************************************************************
		if (te.rs != null)
		{
			if (RegisterFile.iRegLiteralToID(te.rs) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid general-purpose register: " + te.rs + "\n\n" + sRawInput);
			}
		}
		if (te.rt != null)
		{
			if (RegisterFile.iRegLiteralToID(te.rt) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid general-purpose register: " + te.rt + "\n\n" + sRawInput);
			}
		}
		if (te.rd != null)
		{
			if (RegisterFile.iRegLiteralToID(te.rd) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid general-purpose register: " + te.rd + "\n\n" + sRawInput);
			}
		}
		// ************************************************************************
		// Determines whether floating point register operands were used properly.
		// ************************************************************************
		if (te.fs != null)
		{
			if (RegisterFile.iRegLiteralToID(te.fs) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid floating-point register: " + te.fs + "\n\n" + sRawInput);
			}
		}
		if (te.ft != null)
		{
			if (RegisterFile.iRegLiteralToID(te.ft) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid floating-point register: " + te.ft + "\n\n" + sRawInput);
			}
		}
		if (te.fd != null)
		{
			if (RegisterFile.iRegLiteralToID(te.fd) == -1)
			{
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid floating-point register: " + te.fd + "\n\n" + sRawInput);
			}
		}
	}
}
