package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    Coprocessor0.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/

/**
  * This is the Coprocessor0 class.
  *
  * @author Steve Lewis
*/
public class Coprocessor0 {

  // CPU Control Registers

  public static final String REG_ID[][] =
	{
	  {"$0", "$INDEX"},
	  {"$1", "$RANDOM"},
	  {"$2", "$ENTRYLO0"},
	  {"$3", "$ENTRYLO1"},
	  {"$4", "$CONTEXT"},
	  {"$5", "$PAGEMASK"},
	  {"$6", "$WIRES"},
	  {"$7"},
	  {"$8", "$BADVADDR"},
	  {"$9", "$COUNT"},
	  {"$10", "$ENTRYHI"},
	  {"$11", "$COMPARE"},
	  {"$12", "$SR"},
	  {"$13", "$CAUSE"},
	  {"$14", "$EPC"},
	  {"$15", "$PRID"},
	  {"$16", "$CONFIG"},
	  {"$17", "$LLADDR"},
	  {"$18", "$WATCHLO"},
	  {"$19", "$WATCHHI"},
	  {"$20"},
	  {"$21"},
	  {"$22"},
	  {"$23"},
	  {"$24"},
	  {"$25"},
	  {"$26", "$ECC"},
	  {"$27", "$CACHEERR"},
	  {"$28", "$TAGLO"},
	  {"$29", "$TAGHI"},
	  {"$30", "$ERROREPC"},
	  {"$31"}
	};

  public static final int NUM_REGS   = 32;

  public static final int R_INDEX    = 0;
  public static final int R_RANDOM   = 1;
  public static final int R_ENTRYLO0 = 2;
  public static final int R_ENTRYLO1 = 3;
  public static final int R_CONTEXT  = 4;
  public static final int R_PAGEMASK = 5;
  public static final int R_WIRES    = 6;
  public static final int R_BADVADDR = 8;
  public static final int R_COUNT    = 9;   // R4000
  public static final int R_ENTRYHI  = 10;
  public static final int R_COMPARE  = 11;  // R4000
  public static final int R_SR       = 12;
  public static final int R_CAUSE    = 13;
  public static final int R_EPC      = 14;
  public static final int R_PRID     = 15;
  public static final int R_CONFIG   = 16;  // R4000
  public static final int R_LLADDR   = 17;  // R4000
  public static final int R_WATCHLO  = 18;  // R4000
  public static final int R_WATCHHI  = 19;  // R4000
  public static final int R_ECC      = 26;  // R4000
  public static final int R_CACHEERR = 27;  // R4000
  public static final int R_TAGLO    = 28;  // R4000
  public static final int R_TAGHI    = 29;  // R4000
  public static final int R_ERROREPC = 30;  // R4000

  public int[] iaReg;

  public static final int EXCCODE_Int = 0;  // Interrupt
  public static final int EXCCODE_Mod = 1;  // TLB modification
  public static final int EXCCODE_TLBL = 2;  // TLB load
  public static final int EXCCODE_TLBS = 3;  // TLB store
  public static final int EXCCODE_AdEL = 4;  // Address error on load
  public static final int EXCCODE_AdES = 5;  // Address error on store
  public static final int EXCCODE_IBE = 6;  // Bus error: instruction fetch
  public static final int EXCCODE_DBE = 7;  // Bus error: data fetch
  public static final int EXCCODE_Syscall = 8;  // Syscall instruction
  public static final int EXCCODE_Bp = 9;  // Breakpoint (break instruction)
  public static final int EXCCODE_RI = 10;  // Reserved Instruction
  public static final int EXCCODE_CpU = 11;  // Coprocessor unusable
  public static final int EXCCODE_Ov = 12;  // Arithmetic overflow
  public static final int EXCCODE_TRAP = 13;  // MIPS II conditional TRAP
  public static final int EXCCODE_VCEI = 14;  // Virtual coherency error in I-cache
  public static final int EXCCODE_FPE = 15;  // Floating-point exception
  public static final int EXCCODE_C2E = 16;  // Exception from CP2 (not used)
  // 17-22 reserved
  public static final int EXCCODE_Watch = 23;  // Physical addr. match WatchLo/WatchHi
  // 24-30 reserved
  public static final int EXCCODE_VCED = 31;  // Virtual coherency error on data

  public Coprocessor0() {

	reset();

  }    

  public void reset() {

	iaReg = new int[NUM_REGS];

	// It is assumed, by the nature of Java, that 
	//   the values in the register array are all 
	//   initialized to zero.

	iaReg[R_PRID] = 0x00000101;  // R2000, Rev = 1
	// page 49
	// 3          2    1    1
	// 10987654321098765432109876543210
	// 00000000000000000000000100000001
	// Reserved        Imp     Rev

	iaReg[R_SR] = 0x2000FF00;  // Enable CP1, Enable all Exception
	// page 50
	// using MIPS I view
	// 3          2    1    1
	// 10987654321098765432109876543210
	// **10**0**000000011111111**000000
	// 29 = CU1  Coprocessor 1 usable
	// 28 = CU0  Coprocessor 0 usable
	// 25 = RE   
	// 22 = BEV
	// 21 = TS
	// 20 = PE
	// 19 = CM
	// 18 = PZ
	// 17 = SwC
	// 16 = IsC
	// 15-8 = IM  Interrupt mask
	// 5 = KUo
	// 4 = IEo
	// 3 = KUp
	// 2 = IEp
	// 1 = KUc
	// 0 = IEc   global interrupt enable
	// * = always 0

  }  // end method reset()    

  public int iGetReg(int i) {
	return iaReg[i];
  }    

  public void setReg(int i, int value) {
	iaReg[i] = value;
  }    

  public static int iRegLiteralToID(String s) {

	// Returns -1 if the register literal is invalid,
	//   otherwise returns the index of the specified register literal.

	s = s.toUpperCase().trim();
	for (int i = 0; i < Coprocessor0.REG_ID.length; i++) {
	  for (int x = 0; x < Coprocessor0.REG_ID[i].length; x++) {
		if (s.equals(Coprocessor0.REG_ID[i][x]))
		  return i;
	  }
	}
	return -1;

  }  // end method bRegLiteralToID(String)    

}