package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandUndo.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: UNDO [n]

InvalidCommandUseException Messages:
  Unable to translate value of n specified.
  Value of n must be larger than 0.

Command Buffer Results:
  * UNDO OK
  * UNDO OUTPUT
  * UNDO EXCEPTION index
  * UNDO ERR ERR_NO_STATE_CHANGE
  * UNDO ERR ERR_CRITICAL_ERROR index
  * UNDO DONE

*/
import java.util.StringTokenizer;

/**
  * This is the CommandUndo class.
  *
  * @author Steve Lewis
*/
public class CommandUndo extends Command {

  public static final String ERR_NO_STATE_CHANGE = "ERR_NO_STATE_CHANGE";
  public static final String ERR_CRITICAL_ERROR = "ERR_CRITICAL_ERROR";

  private Simulator simulator;
  private StringBuffer sbOutput;

  public CommandUndo(StringTokenizer st, Simulator simulator) {
	super(st);
	this.simulator = simulator;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	int n = 1;
	String s = nextToken();
	if (s.length() > 0) {
	  try {
		n = Integer.parseInt(s);
	  } catch (Exception e) {
		throw new InvalidCommandUseException("Unable to translate value of n specified.");
	  }
	  if (n < 1)
		throw new InvalidCommandUseException("Value of n must be larger than 0.");
	}

	int i = simulator.iPerformUndo(n);
	if (i == 0) {

	  // Undo successful
	  crb.append(RESPONSE + " UNDO OK");

	  // If output was generated during re-executing instructions,
	  //   indicate that an undo buffer is available.
	  sbOutput = simulator.sbGetOutputFromUndo();
	  if (sbOutput != null) {
		crb.append(RESPONSE + " UNDO OUTPUT");
	  }

	} else {

	  switch (i) {
		case -1:
		  crb.append(RESPONSE + " UNDO ERR " + ERR_NO_STATE_CHANGE);
		  break;

		case -5:
		  crb.append(RESPONSE + " UNDO EXCEPTION " + simulator.iGetExceptionCode());
		  break;

		default:
		  // -2 = ERROR APPLYING STATE HISTORY CHANGE
		  // -3 = ERROR APPLYING CHECKPOINT STATE
		  // -4 = ERROR RE-EXECUTING FROM LAST CHECKPOINT
		  // -6 = OUTPUT ERROR WHILE RE-EXECUTING INSTRUCTIONS
		  crb.append(RESPONSE + " UNDO ERR " + ERR_CRITICAL_ERROR + " " + i);

	  }

	}

	crb.append(RESPONSE + " UNDO " + DONE);

  }  // end method execute(CommandResultBuffer)    

  public StringBuffer sbGetOutput() {
	// Returns null if no output was read, or if the buffer
	//   was not initialized.
	return sbOutput;
  }    

}