package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandStep.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: STEP [n]

InvalidCommandUseException Messages:
  Unable to translate value of n specified.
  Value of n must be larger than 0.

Command Buffer Results:
  * STEP BREAKPOINT index
  * STEP GUARD index
  * STEP EXCEPTION index
  * STEP ERR ERR_CYCLE_ERROR
  * STEP DONE

*/
import java.util.StringTokenizer;

/**
  * This is the CommandStep class.
  *
  * @author Steve Lewis
*/
public class CommandStep extends Command {

  public static final String ERR_CYCLE_ERROR = "ERR_CYCLE_ERROR";

  private Simulator simulator;

  public CommandStep(StringTokenizer st, Simulator simulator) {
	super(st);
	this.simulator = simulator;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	// Let n == the number of times to step
	long n = 1;
	// See if the user specified a value for n
	String s = nextToken();
	if (s.length() > 0) {
	  try {
		n = Long.parseLong(s);
	  } catch (Exception e) {
		throw new InvalidCommandUseException("Unable to translate value of n specified.");
	  }
	  if (n < 1) {
		throw new InvalidCommandUseException("Value of n must be larger than 0.");
	  }
	}

	try {
	  int i = simulator.iPerformStep(n);
	  switch (i) {
		case 0:
		  // No Error
		  break;

		case -1:  // = BREAKPOINT ENCOUNTERED (which one is set in iBreakpointIndex)
		  crb.append(RESPONSE + " STEP BREAKPOINT " + simulator.iGetBreakpointIndex());
		  n = 1;
		  break;

		case -2:  // = GUARD ENCOUNTED (which one is set in iGuardIndex)
		  crb.append(RESPONSE + " STEP GUARD " + simulator.iGetGuardIndex());
		  n = 1;
		  break;

		default:  // = CRITICAL ERROR (out of memory?)
		  throw new Exception();
	  }
	} catch (SimulatorException e) {

	  int i = simulator.iGetExceptionCode();
	  crb.append(RESPONSE + " STEP EXCEPTION " + i + " " + e);
	  n = 1;

	} catch (Exception e) {

	  crb.append(RESPONSE + " STEP ERR " + ERR_CYCLE_ERROR + " " + e);

	}

	crb.append(RESPONSE + " STEP " + DONE);

  }  // end method execute()    

}