package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandProcessor.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

  CommandProcessor is the main layer between the user interface
  and the Simulator class.  Clients communicate with the
  simulator by sending commands via an instance of the
  CommandProcessor.  The client must manually parse responses
  to that command.

PUBLIC METHODS

public CommandResultBuffer processCommand(String sCommandLine)
  This method is used to issue commands to the
  simulator.  The format of sCommandLine should match
  one of the command entries in COMMAND.TXT

public void initializeSimulator()
  Used to initialize an instance of the simulator when
  command RESET is called.

public void terminateSimulator()
  Used to clear the simulator instance (remove from memory)
  when command EXIT is called.

public boolean bSimulatorAllocated()
  Use this method to see if the simulator has been allocated
  or not.  If the simulator is not allocated, it is safe to
  terminate the simulator program.  To deallocate the simulator,
  issue an EXIT command.  If the simulator is not allocated, use
  the RESET command to initialize it.

public Command getLastCommand()
  Return a reference of an instance of the last executed
  command instruction.  Commands are organized as
  subclasses of Command.  Some Command instances store
  information that is useful to the client (such as
  CommandOutput).

public long lGetCurrentCycle()
  Returns the current cycle index of the simulator.

*/
import java.util.StringTokenizer;

/**
  * This is the CommandProcessor class.
  *
  * @author Steve Lewis
*/
public class CommandProcessor {

  private static final String[][] COMMAND_TABLE =
	{
	  // *** DO NOT CHANGE THE ORDER OF THESE COMMANDS ***
	  // ********** ADD NEW COMMAND AT THE END ***********
	  { "",                            },  // 0
	  { "ADD",        "+"              },  // 1
	  { "EXIT",       "X", "QUIT"      },  // 2
	  { "GET",        "G", "DUMP"      },  // 3
	  { "HELP",       "H", "?"         },  // 4
	  { "LIST",       "I"              },  // 5
	  { "LOAD",       "L"              },  // 6
	  { "REMOVE",     "-"              },  // 7
	  { "RESET"                        },  // 8
	  { "RUN"                          },  // 9
	  { "SET"                          },  // 10
	  { "STEP",       "S"              },  // 11
	  { "UNDO",       "U"              },  // 12
	  { "HISTORY",    "HI"             },  // 13
	  { "STATUS"                       },  // 14
	  { "INPUT",      "IN"             },  // 15
	  { "OUTPUT",     "OUT"            },  // 16
	  { "CHECKPOINT", "C", "CHECK"     }   // 17
	  // *** FOR NEW COMMANDS, ADD THE APPROPRIATE CODE
	  // *** TO THE SWITCH BLOCK BELOW
	};

  private Simulator simulator = null;

  private Command commandLastCommand = null;

  public void initializeSimulator() {
	// Assume that the calling routine will catch any
	//   errors that occur in this method.  This is
	//   used by CommandReset.
	simulator = new Simulator();
	simulator.reset();
  }    

  public void terminateSimulator() {
	// Clear the simulator.  This is used by CommandExit.
	simulator = null;
  }    

  public boolean bSimulatorAllocated() {
	// Returns TRUE if the simulator has been allocated.
	// Otherwise returns FALSE.
	/*
	   The simulator will not be allocated until a RESET command
	   has been issued.  The simulator can be deallocated by
	   issuing an EXIT command.
	*/
	return (simulator != null);

  }    

  public Command getLastCommand() {
	// Returns a reference to the last Command instance
	//   that was issued.  Certain commands can hold
	//   information (such as CommandOutput) that
	//   the client might want to use.
	return commandLastCommand;
  }    

  public long lGetCurrentCycle() {
	return simulator.lGetCycleIndex();
  }    

  public CommandResultBuffer processCommand(String sCommandLine) {

	/*

	This method attempts to execute the specified command.

	INPUT:
	  sCommandLine - a string containing a command followed by
		any desired parameters.  e.g. "set register $t0 5"
		The first word in the command line is expected to match
		one of the entries in the command table above.
		If it does not, an exception is issued and handle at
		the end of the method.

		NOTE: sCommandLine may consists of mixed case characters.
		  This method will uppercase the command, so that it may
		  be matched to the command table.  The remaining
		  parameters are also uppercased, to make it easier to
		  handle each parameter by the command class.  However,
		  the sCommandLine input parameter itself is never
		  modified.  It may be passed to a command class, should
		  that class need the original mixed-case version of the
		  command line (such as when specifying a literal string).

	OUTPUT:
	  Always returns an instance of CommandResultBuffer.  This
		buffer contains the result(s) of the executed instruction.
		In case an invalid command was specified, or no command at all,
		the buffer is empty.

	NOTES:
	  This method creates instances of the following classes:
		Command (or a sub-class thereof, such as CommandRun)
		StringTokenizer
		String
	  Be aware that this can potentially cause an OutOfMemoryException.
   
	*/

	commandLastCommand = null;

	Command command = null;

	StringTokenizer stParams = new StringTokenizer(sCommandLine.toUpperCase().trim());

	String sCommand = "";
	String sParameters = "";
	try {

	  sCommand = stParams.nextToken();
	  // sCommand is a literal string containing the name of the
	  //   command that was issued.

	  // Left-trim sCommandLine, the original command input, to remove
	  //   any spaces before the command.
	  int i = -1;
	  while (sCommandLine.charAt(0) == ' ')
		i++;
	  if (i >= 0)
		sCommandLine = sCommandLine.substring(i+1);
		
	  // Let sParameters be everything typed after the command.  
	  //   "length()+1" is used to remove the space after the command.
	  sParameters = sCommandLine.substring(sCommand.length()+1);

	} catch (Exception e) {
	  // Probably no next token, which just means no command
	  //   was specified.
	}

	// See if the command matches anything in the command table...
	int iCommandIndex = -1, iAlias = -1;
	for (iCommandIndex = 0; iCommandIndex < COMMAND_TABLE.length; iCommandIndex++) {
	  int iNumAlias = COMMAND_TABLE[iCommandIndex].length;
	  for (iAlias = 0; iAlias < iNumAlias; iAlias++) {
		if (sCommand.equals(COMMAND_TABLE[iCommandIndex][iAlias]))
		  break;
	  }
	  if (iAlias < COMMAND_TABLE[iCommandIndex].length)
		// We found a matching command alias in the inner loop,
		//   no need to continue this outer loop...
		break;
	}

	// If the command matched any entry in the command table,
	//   then iCommandIndex == the index of the corresponding
	//   command table entry.  Otherwise, iCommandIndex == -1
	switch (iCommandIndex) {

	  case 0:  // null command
		command = new Command();
		break;

	  case 1:  // ADD
		command = new CommandAdd(stParams, simulator);
		break;

	  case 2:  // EXIT
		command = new CommandExit(this);
		break;

	  case 3:  // GET
		command = new CommandGet(stParams, simulator);
		break;

	  case 4:  // HELP
		command = new CommandHelp();
		break;

	  case 5:  // LIST
		command = new CommandList(stParams, simulator);
		break;

	  case 6:  // LOAD
		command = new CommandLoad(sParameters, simulator);
		break;
	   
	  case 7:  // REMOVE
		command = new CommandRemove(stParams, simulator);
		break;

	  case 8:  // RESET
		command = new CommandReset(this);
		break;

	  case 9:  // RUN
		command = new CommandRun(this);
		break;

	  case 10:  // SET
		command = new CommandSet(stParams, simulator);
		break;

	  case 11: // STEP
		command = new CommandStep(stParams, simulator);
		break;

	  case 12: // UNDO
		command = new CommandUndo(stParams, simulator);
		break;

	  case 13: // HISTORY
		command = new CommandHistory(stParams, simulator);
		break;

	  case 14: // STATUS
		command = new CommandStatus(stParams, simulator);
		break;

	  case 15: // INPUT
		command = new CommandInput(stParams, sParameters, simulator);
		break;

	  case 16: // OUTPUT
		command = new CommandOutput(stParams, simulator);
		break;

	  case 17: // CHECKPOINT
		command = new CommandCheckpoint(simulator);
		break;

	  // *** ADD CODE FOR NEW COMMAND HERE

	}

	CommandResultBuffer crb = null;
	// An instance of a CommandResultBuffer is used to hold
	//   any information returned from the issued command. 
	//   This always includes at least a status string that
	//   indicates the command was completed.  Often,
	//   this also includes additional information that
	//   is useful to the client.  See COMMANDS.TXT

	try {

	  crb = new CommandResultBuffer();
	  command.execute(crb);
	  commandLastCommand = command;

	} catch (InvalidCommandUseException e) {

	  // This does not mean an error occured in the command.
	  //   It only means the user specified an invalid
	  //   parameter, and the command was not executed.
	  //   To check for errors in executing the command,
	  //   look for the "ERR" token in the result strings.

	  crb = new CommandResultBuffer();
	  crb.append("EXCEPTION " + e.toString());

	} catch (Exception e) {

	  // The user specified an unreconginzed command, or an
	  //   unexpected error occurred.

	  crb = new CommandResultBuffer();
	  crb.append("UNKNOWN_COMMAND " + sCommand + " " + e);

	}

	return crb;

  }  // end method processCommand(String)    

}