package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    CommandGet.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

Command: GET ( M* [ range | single | M* ] ) | ( R* (GPR|CP0|CP1) [ range | single ] )

InvalidCommandUseException Messages:
  Invalid memory range or value specified.
  Invalid register range or value specified.
  Invalid use of GET command.

Command Buffer Results:
  * GET MEMORY (startAddress,8) (values,2)
  * GET REGISTER GPR (index,2) (value,8)
  * GET REGISTER CP0 (index,2) (value,8)
  * GET REGISTER CP1 (index,2) (value,8)
  * GET ERR ERR_CRITICAL errorMsg
  * GET DONE

*/
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.util.Vector;

/**
  * This is the CommandGet class.
  *
  * @author Steve Lewis
*/
public class CommandGet extends Command {

  public static final String ERR_CRITICAL = "ERR_CRITICAL";

  private Simulator simulator;

  public CommandGet(StringTokenizer st, Simulator simulator) {
	super(st);
	this.simulator = simulator;
  }    

  public void execute(CommandResultBuffer crb) throws InvalidCommandUseException {

	State stateSim = simulator.getState();

	try {
	
	  String s = nextToken();  // s == what to get

	  if (s.startsWith("M")) {  // MEMORY keyword

		s = nextToken();

		if (s.equals("") || s.startsWith("M")) {  // MODIFIED keyword

		  Vector v = stateSim.vGetModifiedAddresses();
		  // v is a Vector of Integer objects, where each
		  // Integer represent a modified memory address.

		  if ((v != null) && (v.size() > 0)) {
			// There is at least one modified address.

			Enumeration e = v.elements();

			// Get the first element, so we know which address we
			//   are starting at.
			int i = ((Integer)e.nextElement()).intValue();
			int iAddress = i;
			StringBuffer sbBuffer = new StringBuffer(RESPONSE + " GET MEMORY " + Utility.sAsHexPadded(iAddress, 8) + " ");

			do {

			  // Get the memory value at the current address, and add
			  //   it to the string buffer.
			  byte value = stateSim.loadByte(iAddress);
			  sbBuffer.append(Utility.sAsHexPadded(value, 2) + " ");

			  if (!e.hasMoreElements()) {
				// No more addresses to process.
				break;
			  }

			  // Increment the address counter.  We expect the
			  //   next address to match this incremented value.
			  iAddress++;

			  // Get the next address.
			  i = ((Integer)e.nextElement()).intValue();

			  if (i != iAddress) {
				// The new address does not match the address that
				//   we expected (i.e. we have a disjoint set).
				//   Add the current string buffer value to the
				//   command response, and prepare a new string buffer.
				crb.append(sbBuffer.toString());

				iAddress = i;
				sbBuffer = new StringBuffer(RESPONSE + " GET MEMORY " + Utility.sAsHexPadded(iAddress, 8) + " ");

			  }

			} while (true);

			crb.append(sbBuffer.toString());

		  }

		} else {
		  // Assume the user has specified a memory range or a
		  //   specific memory address.

		  String sMem = sGetMemory(s, stateSim);
		  if (sMem.trim().equals(""))
			throw new InvalidCommandUseException("Invalid memory range or value specified.");
		  crb.append(sMem);

		}
   
	  } else if (s.startsWith("R")) {  // REGISTER keyword

		String sReg = sGetRegister(stSource, stateSim);
		if (sReg.trim().equals(""))
		  throw new InvalidCommandUseException("Invalid register range or value specified.");
		crb.append(sReg);

	  }

	  else {
  
		throw new InvalidCommandUseException("Invalid use of GET command.");

	  }

	} catch (Exception e) {
	  crb.append(RESPONSE + " GET ERR " + ERR_CRITICAL + " " + e);
	}

	crb.append(RESPONSE + " GET " + DONE);
  }    

  private String sGetMemory(String sToken, State stateSim) {

	// This method assumes that the calling routine performs
	//   exception checking, etc.

	String sResult = "";

	String s = sToken;  // s == should be the desired memory range or value

	int[] ia = null;
	if (s.trim().equals("")) {
	  // Use default memory range value: the next 10 words
	  ia = new int[2];
	  ia[0] = stateSim.iGetRegister(RegisterFile.R_PC);
	  ia[1] = ia[0] + 40;  // 10 instructions
	} else {
	  ia = iaGetMemoryRangeOrValue(s);
	  if (ia == null)
		return "";
	}

	StringBuffer sb = new StringBuffer();

	// Limit start and stop to 32-bit address range
	long start = ia[0] & 0x00000000FFFFFFFFL;
	long stop = ia[1] & 0x00000000FFFFFFFFL;

	sb.append(RESPONSE + " GET MEMORY " + Utility.sAsHexPadded(start, 8) + " ");

	for (long l = start; l <= stop; l++) {
	  byte v = stateSim.loadByte((int)l);
	  sb.append(Utility.sAsHexPadded(v, 2) + " ");
	}

	sResult = sb.toString();

	return sResult;

  }  // end method doGetMemory(StringTokenizer)    

  private String sGetRegister(StringTokenizer st, State stateSim) {

	// This method assumes that the calling routine performs
	//   exception checking, etc.

	String sResult = "";

	String s = nextToken();  // s == GPR | CP0 | CP1

	String sWhichReg = nextToken();  // s == desired token

	int[] iaRegs = null;

	if (s.equals("GPR")) {

	  if (sWhichReg.trim().equals("")) {
		// Default to showing all of the register values
		iaRegs = new int[2];
		iaRegs[0] = 0;
		iaRegs[1] = RegisterFile.NUM_REGS-1;
	  } else {
		iaRegs = iaGetRegisterRangeOrValue(sWhichReg, 0);
		if (iaRegs == null)
		  return "";
	  }

	  for (int i = iaRegs[0]; i <= iaRegs[1]; i++) {
		int iRegValue = stateSim.iGetRegister(i);
		sResult += 
		  RESPONSE + " GET REGISTER GPR " +
		  Utility.sAsHexPadded(i, 2) + " " +
		  Utility.sAsHexPadded(iRegValue, 8);
		if (i < iaRegs[1])
		  sResult += "\n";
	  }

	} else if (s.equals("CP0")) {

	  if (sWhichReg.trim().equals("")) {
		// Default to showing all of the register values
		iaRegs = new int[2];
		iaRegs[0] = 0;
		iaRegs[1] = Coprocessor0.NUM_REGS-1;
	  } else {
		iaRegs = iaGetRegisterRangeOrValue(sWhichReg, 1);
		if (iaRegs == null)
		  return "";
	  }

	  for (int i = iaRegs[0]; i <= iaRegs[1]; i++) {
		int iRegValue = stateSim.iGetCP0Register(i);
		sResult += 
		  RESPONSE + " GET REGISTER CP0 " +
		  Utility.sAsHexPadded(i, 2) + " " +
		  Utility.sAsHexPadded(iRegValue, 8);
		if (i < iaRegs[1])
		  sResult += "\n";
	  }

	}
	else if (s.equals("CP1")) {

	  if (sWhichReg.trim().equals("")) {
		// Default to showing all of the register values
		iaRegs = new int[2];
		iaRegs[0] = 0;
		iaRegs[1] = Coprocessor1.NUM_REGS-1;
	  } else {
		iaRegs = iaGetRegisterRangeOrValue(sWhichReg, 2);
		if (iaRegs == null)
		  return "";
	  }

	  for (int i = iaRegs[0]; i <= iaRegs[1]; i++) {
		int iRegValue = stateSim.iGetCP1Register(i);
		sResult += 
		  RESPONSE + " GET REGISTER CP1 " +
		  Utility.sAsHexPadded(i, 2) + " " +
		  Utility.sAsHexPadded(iRegValue, 8);
		if (i < iaRegs[1])
		  sResult += "\n";
	  }

	}

	return sResult;

  }  // end method doGetRegister(StringTokenizer)    

}