package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    Command.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/
import java.util.StringTokenizer;

/**
  * This is the Command class.
  *
  * @author Steve Lewis
*/
public class Command {

  protected StringTokenizer stSource;

  protected static final String RESPONSE = "*";
  protected static final String DONE = "DONE";

  public Command() {
  }    

  public Command(StringTokenizer st) {
	stSource = st;
  }    

  public void execute(CommandResultBuffer erb) throws InvalidCommandUseException {
	// *** OVERRIDE THIS METHOD IN DESCENDENTS ***
	throw new InvalidCommandUseException("Command not yet implemented.");
  }    

  public String nextToken() {
	String result = "";
	try {
	  result = stSource.nextToken();
	} catch (Exception e) {
	  result = "";
	}
	return result;
  }    

  /**********************/
  /*** STATIC METHODS ***/
  /**********************/

  protected static int iGetNumber(String s) throws NumberFormatException {
	// If string s is prepended with 0x, assume the address
	//   is given as a hex number.  Otherwise, assume the
	//   address is given as a dec number.
	int result = 0;
	s = s.toUpperCase();
	try {
	  long lValue = 0;
	  if (s.startsWith("0X")) {
		// assume the address value is specified in hex...
		lValue = Long.parseLong(s.substring(2), 16);
	  } else {
		// assume the address value is specified in dec...
		lValue = Long.parseLong(s);
	  }
	  if ((lValue > 0x00000000FFFFFFFFL) || (lValue < Integer.MIN_VALUE))
		throw new NumberFormatException("Must specify at most a 32-bit number.");
	  result = (int)lValue;
	} catch (Exception e) {
	  throw new NumberFormatException("Invalid number specified.");
	}
	return result;
  }  // end method iGetNumber(String)    

  protected static int iGetRegisterIndex(String s, int iRegType) throws NumberFormatException {
	/*

	Returns the corresponding register index number of the
	  specified register string.  The register string may
	  already by the register index number (e.g. "0, 1, 32")
	  or a literal register string (e.g. "$pc", "$t0").
	We need iRegType to know which set of registers to search.
	  iRegType == 0 (GPR)
	  iRegType == 1 (CP0)
	  iRegType == 2 (CP1)

	RETURNS
	  -1 = Error
	  Otherwise, the number returned is the register index number
		into the corresponding iRegType register file.

	*/

	int iResult = -1;

	try {

	  int iMaxRegIndex = -1;

	  switch (iRegType) {
		case 0:  // GPR
		  iResult = RegisterFile.iRegLiteralToID(s);
		  iMaxRegIndex = RegisterFile.NUM_REGS;
		  break;
		case 1:  // CP0
		  iResult = Coprocessor0.iRegLiteralToID(s);
		  iMaxRegIndex = Coprocessor0.NUM_REGS;
		  break;
		case 2:  // CP1
		  iResult = Coprocessor1.iRegLiteralToID(s);
		  iMaxRegIndex = Coprocessor1.NUM_REGS;
		  break;
		default:
		  break;
	  }

	  if (iResult == -1) {
		// Assume this is not a literal register name,
		//   and instead it is a register number (e.g. 0, 1, ...)
		if (s.charAt(0) == '$')
		  s = s.substring(1);
		iResult = iGetNumber(s);
	  }

	  if ( (iResult < 0) || (iResult >= iMaxRegIndex) )
		throw new NumberFormatException("Invalid register number.");

	} catch (Exception e) {
	  throw new NumberFormatException("Invalid register specified.");
	}

	return iResult;
  }    

  protected static int[] iaGetRegisterRangeOrValue(String s, int iRegType) {

	/*

	Registers a register range, defined by the input string s.
	Examples: s == "0-30", "$t1-$t8", "$pc-$ir", "$0-$3"

	iRegType specifies which set of registers to search.
	  iRegType == 0 (GPR)
	  iRegType == 1 (CP0)
	  iRegType == 2 (CP1)

	RETURNS
	  null = Any error occured (such as invalid register range, etc)
	  Otherwise, returns a integer array where
		i[0] == the beginning register index
		i[1] == the ending register index
	*/

	int[] ia = new int[2];

	try {

	  int i = s.indexOf("-");

	  if (i == -1) {

		// Assume this is a specific register number
		ia[0] = iGetRegisterIndex(s, iRegType);
		ia[1] = ia[0];

	  } else {

		// Assume this is a range of registers
		String s1 = s.substring(0, i);
		String s2 = s.substring(i+1);

		ia[0] = iGetRegisterIndex(s1, iRegType);
		ia[1] = iGetRegisterIndex(s2, iRegType);

		if (ia[1] < ia[0]) {
		  // Swap ia[1] and ia[0]
		  int t = ia[1];
		  ia[1] = ia[0];
		  ia[0] = t;
		}

	  }

	} catch (Exception e) {
	  ia = null;
	}

	return ia;
  }    

  protected static int[] iaGetMemoryRangeOrValue(String s) {
	// START-STOP
	// return an integer array where
	//   the first element is the START value
	//   and the second element is the STOP value
	// if no STOP value is specified, then we assume
	//   STOP == START
	// if either START or STOP is an invalid number,
	//   return null

	int[] ia = new int[2];

	try {
	  int i = s.indexOf("-");
	  if (i == -1) {
		// Assume this is a specific value
		ia[0] = iGetNumber(s);
		ia[1] = ia[0];
	  } else {
		// Assume this is a range of addresses
		String s1 = s.substring(0, i);
		String s2 = s.substring(i+1);
		ia[0] = iGetNumber(s1);
		ia[1] = iGetNumber(s2);
//        if (ia[1] < ia[0])
//          ia = null;
	  }
	} catch (Exception e) {
	  ia = null;
	}
	return ia;
  }    

}