package JIMSCore;

/*

PROJECT: MIPS Simulator With Reversible Debugging Support
FILE:    Assembler.java
AUTHOR:  Steve Lewis

Copyright (C) 2001 University of Florida
All rights reserved.

======================================================================

*/
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.util.Vector;

/**
* This is the Assembler class.
*
* @author Steve Lewis
* @author Fred Williams
*/
public class Assembler
{

	public static final char NEWLINE = '\n';

	public static final int MODE_TEXT = 1;
	public static final int MODE_DATA = 2;
	public static final int MODE_KTEXT = 3;
	public static final int MODE_KDATA = 4;
	public static final int MODE_NONE = 0;

	public static final int TEXT_SECTION = 0;
	public static final int DATA_SECTION = 1;

	public int iCurrentMode;
	public int iCurrentTextIndex;
	public int iCurrentDataIndex;

	public Vector vLabelVector;  // Vector of LabelEntry objects

	public Vector vTextSection;  // Vector of TextSection
	public TextSection currentTextSection;
	public int iCurrentTextSectionIndex;

	public Vector vDataSection;  // Vector of DataSection
	public DataSection currentDataSection;
	public int iCurrentDataSectionIndex;

	private InstructionEncoder instructionEncoder;
	private DirectiveProcessor directiveProcessor;

	private static int iInvalidUseOfAT;
	// Indicates the last source line that used register $at,
	//   when it probably should not have.  A value of -1
	//   indicates that no such error has occurred.

	public Assembler(String sModifiedSourceFilename)
	{
		ModifiedSource.setOutputFile(sModifiedSourceFilename);
		reset();
	}

	public Assembler()
	{
		this(null);
	}

	public void reset()
	{

		iCurrentMode = MODE_NONE;

		vLabelVector = new Vector();
		vTextSection = null;
		iCurrentTextSectionIndex = -1;

		vDataSection = null;
		iCurrentDataSectionIndex = -1;

		instructionEncoder = new InstructionEncoder(this);
		directiveProcessor = new DirectiveProcessor(this);

		Assembler.iInvalidUseOfAT = -1;

		ModifiedSource.start();

	}
	// end constructor Assembler()

	public int iGetLastInvalidUseOfAT()
	{
		return iInvalidUseOfAT;
	}

/***********************************************************
	PHASE 1 -- INITIAL TRANSLATION
	************************************************************/

	public int iProcessLine(String sRawInput, int iCurrLine) throws Exception
	{
		/*

		INPUT
		sRawInput = One line from the assembly file (unmodified)
		iCurrLine = The corresponding line number of sInput

		RETURNS
		0 = SUCCESS
		-1 = Colon error
		-2 = Label already defined
		-3 = Invalid mode while adding a label.
		-4 = Directive error
		-5 = Text error
		*/

		Assembler.iInvalidUseOfAT = -1;

		String sInput = sRawInput.substring(0);  // Copy sRawInput

		if (sInput.trim().equals(""))
		{
			// Nothing to do.
			return 0;
		}

		// Remove anything past the single-line comment character.
		//   However, make sure the comment character is not being
		//   used inside a double-quoted string sequence.
		//   e.g. we could have:
		//     .asciiz "some string using #"  # actual comments
		//   We only want to ignore the portion of the string starting
		//     at the second occurrence of "#".
		boolean bCanComment = true;
		for (int x = 0; x < sInput.length(); x++)
		{
			char c = sInput.charAt(x);
			if (c == '"')
			{
				// Toggle the CanComment flag.
				bCanComment = !bCanComment;
				} else if ((c == '#') && bCanComment) {
				// We encounted a valid comment token.  Let sInput
				//   equal everything to the left of the comment token.
				sInput = sInput.substring(0, x);
				break;
			}
		}

		int iColonIndex = -1;
		// iColonIndex == -1 means no label colon has been found.

		String sLabel = "";
		// Check for the start of a COLON, which would represent the
		//   end of a label.  However, we can not just look for a colon.
		//   If the colon happened to be inside a string, then that
		//   colon belongs to part of that string, not the label.
		//   e.g. we could have:
		//     .asciiz "some string using :"
		//   While the string has a COLON, it does not have a label.
		for (int x = 0; x < sInput.length(); x++)
		{
			char c = sInput.charAt(x);
			if (c == ':')
			{
				// A colon was encounted, so let the colon index be the
				//   current x index position.  Then we terminate the loop.
				iColonIndex = x;
				break;
			}
			if (c == '"')
			{
				// A double quote has been encounted, and we can assume
				//   a colon has not yet been detected.  So a label can not
				//   start on this line.  Just terminate the loop, since
				//   the colon index is already assigned to -1.
				break;
			}
		}

		if (iColonIndex == 0)
		{
			// A line can not start with a colon.
			// return -1;
			throw new Exception("Line " + iCurrLine + "\n" + "Line cannot start with colon." + "\n\n" + sInput);
		}
		if (iColonIndex > 0)
		{
			// A label was found, add it to the label list.

			sLabel = sInput.substring(0,iColonIndex).trim();

			if (iGetLabelIndex(sLabel) != -1)
			{
				// The specified label was already defined earlier.
				// return -2;
				throw new Exception("Line " + iCurrLine + "\n" + "Label already defined: " + sLabel + "\n\n" + sInput);
			}

			sInput = sInput.substring(iColonIndex+1).trim();
			// sInput == everything past the colon

			LabelEntry labelEntry = null;
			switch (iCurrentMode)
			{
				case MODE_KTEXT:
				case MODE_TEXT:
				labelEntry = new LabelEntry(sLabel, TEXT_SECTION, iCurrentTextSectionIndex, iCurrentTextIndex);
				break;
				case MODE_KDATA:
				case MODE_DATA:
				labelEntry = new LabelEntry(sLabel, DATA_SECTION, iCurrentDataSectionIndex, iCurrentDataIndex);
				break;
				default:
				// return -3;
				throw new Exception("Line " + iCurrLine + "\n" + "Invalid mode while adding label: " + sLabel + "\n\n" + sInput);
			}

			vLabelVector.add(labelEntry);
		}

		sInput = sInput.trim();

		if (sInput.equals(""))
		{

			// This line was a label by itself, no code.  So there
			//   is nothing left to do.
			ModifiedSource.append(sRawInput);  // output raw line, with comments, etc

		}
		else if (sInput.charAt(0) == '.')
		{

			// This is a directive.
			int i = iProcessDirective(sRawInput, sInput, iCurrLine);
			if (i != 0)
			{
				// return -4;
				throw new Exception("Line " + iCurrLine + "\n" + "Generic directive error." + "\n\n" + sInput);
			}
		}
		else
		{

			// Assume sInput contains some instruction.
			int i = 0;
			try
			{
				i = iProcessInstruction(sRawInput, sInput, iCurrLine);
			}
			catch (Exception err)
			{
				throw err;
			}
			if (i != 0)
			{
				//System.err.println("Error " + i + " on line " + iCurrLine);
				// return -5;
				throw new Exception("Line " + iCurrLine + "\n" + "Generic text error." + "\n\n" + sInput);
			}
		}
		return 0;
	}
	// end method iProcessLine(String, int)

	private int iProcessInstruction(String sRawInput, String sInput, int iCurrLine) throws Exception
	{
		/*
		INPUT
		sRawInput = The unmodified input from the source assembly file
		sInput = modified version (trimmed and comments removed)
		iCurrLine = Corresponding line number of sRawInput
		RETURNS
		0 = SUCCESS
		-1 = Mode is not text.
		-2 = Error in instruction (invalid format, out of memory)
		*/

		switch (iCurrentMode)
		{
			case Assembler.MODE_KTEXT:
			case Assembler.MODE_TEXT:
			break;
			default:
			// return -1;
			throw new Exception("Line " + iCurrLine + "\n" + "Mode is not text." + "\n\n" + sInput);
		}

		instructionEncoder.processInstruction(sRawInput, sInput, iCurrLine);
// Next line modified by Fred Williams, 2001: line commented out
//	  System.err.println(e + " " + sInput);

		return 0;

	}
	// end method iProcessInstruction(String, int);

	private int iProcessDirective(String sRawInput, String sInput, int iCurrLine)
	{
		/*
		RETURNS
		0 = SUCCESS
		-1 = General directive error
		-2 = Critical exception error
		*/

		try
		{

			int i = directiveProcessor.iProcessDirective(sInput, iCurrLine);
			if (i != 0)
			return -1;
			ModifiedSource.append(sRawInput);

			} catch (Exception e) {
			return -2;
		}

		return 0;

	}
	// end method iProcessDirective(String, int)

/***********************************************************
	PHASE 2 -- FINAL TRANSLATION
	************************************************************/

	private int iPhaseTwoErrors;

	public StringBuffer sbGetMachineCode(String sSourceName) throws Exception
	{
		iPhaseTwoErrors = 0;
		StringBuffer sbResult = new StringBuffer();

		if (!sSourceName.equals(""))
		sbResult.append("S " + sSourceName + "\n");

		if (vTextSection != null)
		{
			Enumeration e = vTextSection.elements();
			while (e.hasMoreElements())
			{
				TextSection textSection = (TextSection)e.nextElement();
				sbResult.append(sbProcessTextSection(textSection));
			}
		}
		if (vDataSection != null)
		{
			Enumeration e = vDataSection.elements();
			while (e.hasMoreElements())
			{
				DataSection dataSection = (DataSection)e.nextElement();
				sbResult.append(sbProcessDataSection(dataSection));
			}
		}
		return sbResult;
	}
	// end method phase2()

	public int iGetNumPhaseTwoErrors()
	{
		// RETURNS
		//   0 = if no errors
		//  >0 = indicates number of errors detected
		return iPhaseTwoErrors;
	}

	private StringBuffer sbProcessTextSection(TextSection textSection) throws Exception
	{

		StringBuffer result = new StringBuffer();

		Enumeration e = textSection.vTextVector.elements();
		int iCurrentAddress = textSection.iAddress;
		while (e.hasMoreElements())
		{
			TextEntry textEntry = (TextEntry)e.nextElement();
			String s = sFinalizeInstruction(textEntry.sGetInstruction(), textEntry.iSourceLine, iCurrentAddress);

			result.append(
			"T " +
			Utility.sAsHexPadded(iCurrentAddress,8) + " " +
			sCompressBitSequence(s) +
			" L" + textEntry.iSourceLine
			);

			result.append(NEWLINE);
			iCurrentAddress += DirectiveProcessor.SIZE_WORD;
		}
		return result;
	}
	// end method processTextSection(TextSection)

	private StringBuffer sbProcessDataSection(DataSection dataSection)
	{

		StringBuffer result = new StringBuffer();

		Enumeration e = dataSection.vDataVector.elements();
		while (e.hasMoreElements())
		{
			DataEntry dataEntry = (DataEntry)e.nextElement();
			String s = sFinalizeData(dataEntry.sData);

			result.append(
			"D " +
			Utility.sAsHexPadded(dataEntry.iAddress,8) + " " +
			s
			);

			result.append(NEWLINE);
		}

		return result;

	}
	// end method processDataSection(DataSection)

// ===================================================================================
// METHOD:		sFinalizeData
// FUNCTION:	
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Translates addr(X) into a 32-bit address.
  * 
  * @param s The data directive to be processed.
  * @returns The processed directive.
*/
	private String sFinalizeData(String s)
	{
		/*
		Look for and translate
		addr(X), 32-bit
		*/

		int i;
		do
		{
			i = s.indexOf("addr(");
			if (i >= 0)
			{
				int x = i+5;
				while (s.charAt(x) != ')')
				x++;
				String s1 = s.substring(0, i);

				String s2 = s.substring(i+5, x);
				if (!bLabelDefined(s2))
				{
				}
				else
				{
					int iValue = iGetLabelAddress(s2);
					s2 = Utility.sAsBinaryPadded(iValue, 32);
				}

				String s3 = s.substring(x+1);

				s = s1 + s2 + s3;

			}
			else
			{
				break;
			}
		}
		while (true);
		return sProcessedDataString(s);
	}

// ===================================================================================
// METHOD:		sFinalizeInstruction
// FUNCTION:	
// ===================================================================================		
// javadoc comments follow
// ===================================================================================
/**
  * Translates shamt, hi16, lo16, target, offset, branch, imm.
  * 
  * @param s The instruction to be processed.
  * @param iCurrentAddress The address of the instruction.
  * @returns The processed instruction.
*/
	private String sFinalizeInstruction(String s, int iSourceLine, int iCurrentAddress) throws Exception
	{
		/*
		Look for and translate
		shamt(X), 5 bit
		hi16(X)
		lo16(X)
		target(X), 26 bit
		offset(X), 16 bit
		branch(X), 16 bit
		imm(X), 16 bit
		*/

		int i;
		i = s.indexOf("shamt(");
		if (i >= 0)
		{
			int x = i+6;
			while (s.charAt(x) != ')')
			x++;
			String s1 = s.substring(0, i);

			String s2 = s.substring(i+6, x);
			int iValue = (int)Utility.IntegerLiteralToLong(s2);
			s2 = Utility.sAsBinaryPadded(iValue, 5);

			String s3 = s.substring(x+1);
			s = s1 + s2 + s3;
		}
		i = s.indexOf("hi16(");
		if (i >= 0)
		{
			int x = i+5;
			while (s.charAt(x) != ')')
			x++;
			String s1 = s.substring(0, i);

			String s2 = s.substring(i+5, x);
			int iPlus = 0;
			if (s2.startsWith("+4|"))
			{
				s2 = s2.substring(3);
				iPlus = 4;
			}
			int addr = 0;
			if (bLabelDefined(s2))
			{
				addr = iGetLabelAddress(s2);
			}
			else
			{
				// assume it is a numeric integer value
				try
				{
					addr = (int)Utility.IntegerLiteralToLong(s2);
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			addr += iPlus;
			addr = (addr & 0xFFFF0000) >>> 16;

			String s3 = s.substring(x+1);

			s = s1 + addr + s3;
		}
		i = s.indexOf("lo16(");
		if (i >= 0)
		{
			int x = i+5;
			while (s.charAt(x) != ')')
			x++;
			String s1 = s.substring(0, i);

			String s2 = s.substring(i+5, x);
			int iPlus = 0;
			if (s2.startsWith("+4|"))
			{
				s2 = s2.substring(3);
				iPlus = 4;
			}
			int addr = 0;
			if (bLabelDefined(s2))
			{
				addr = iGetLabelAddress(s2);
			}
			else
			{
				// assume it is a numeric integer value
				try
				{
					addr = (int)Utility.IntegerLiteralToLong(s2);
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			addr += iPlus;
			addr = (addr & 0x0000FFFF);

			String s3 = s.substring(x+1);
			s = s1 + addr + s3;
		}
		i = s.indexOf("target(");
		if (i >= 0)
		{
			// process 26-bit target
			int x = i+6;
			while (s.charAt(x) != ')')
			x++;

			String s1 = s.substring(0, i);

			String s2 = s.substring(i+7, x);

			int addr = 0;
			if (bLabelDefined(s2))
			{
				addr = iGetLabelAddress(s2);
			}
			else
			{
				try
				{
					addr = (int)Utility.IntegerLiteralToLong(s2);
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			addr = (addr & 0x03FFFFFF) >> 2;
			s2 = Utility.sAsBinaryPadded(addr, 26);

			String s3 = s.substring(x+1);

			s = s1 + s2 + s3;

			return s;
		}

		i = s.indexOf("imm(");
		if (i >= 0)
		{
			// process 16-bit offset
			int x = i+4;
			while (s.charAt(x) != ')')
			x++;

			String s1 = s.substring(0, i);

			String s2 = s.substring(i+4, x);
			int iPlus = 0;
			if (s2.startsWith("+4|"))
			{
				s2 = s2.substring(3);
				iPlus = 4;
			}
			int iValue = 0;
			if (bLabelDefined(s2))
			{
				iValue = iGetLabelAddress(s2);
			}
			else
			{
				try
				{
					iValue = (int)Utility.IntegerLiteralToLong(s2);
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			s2 = Utility.sAsBinaryPadded(iValue + iPlus, 16);

			String s3 = s.substring(x+1);
			s = s1 + s2 + s3;
			return s;
		}

		i = s.indexOf("branch(");
		if (i >= 0)
		{
			// process 16-bit offset
			int x = i+7;
			while (s.charAt(x) != ')')
			x++;

			String s1 = s.substring(0, i);

			String s2 = s.substring(i+7, x);
			int branch = 0;
			if (bLabelDefined(s2))
			{
				branch = iGetLabelAddress(s2);
				branch = (branch - iCurrentAddress) >> 2;
			}
			else
			{
				try
				{
					branch = (int)Utility.IntegerLiteralToLong(s2) >> 2;
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			s2 = Utility.sAsBinaryPadded(branch, 16);
			String s3 = s.substring(x+1);
			s = s1 + s2 + s3;
			return s;
		}

		i = s.indexOf("offset(");
		if (i >= 0)
		{
			// process 16-bit offset
			int x = i+6;
			while (s.charAt(x) != ')')
			x++;

			String s1 = s.substring(0, i);

			String s2 = s.substring(i+7, x);
			int ofs = 0;
			if (bLabelDefined(s2))
			{
				ofs = iGetLabelAddress(s2);
			}
			else
			{
				try
				{
					ofs = (int)Utility.IntegerLiteralToLong(s2);
				}
				catch (Exception e)
				{
					iPhaseTwoErrors++;
					throw new Exception("Line " + iSourceLine + "\n" + "Undefined label or invalid value: " + s2);
				}
			}
			s2 = Utility.sAsBinaryPadded(ofs, 16);
			String s3 = s.substring(x+1);
			s = s1 + s2 + s3;
			return s;
		}
		return s;
	}
	// end method finalizeInstruction(String)

	public String toString()
	{

		StringBuffer sb = new StringBuffer();

		if (vLabelVector != null)
		{
			sb.append("\nLABEL:\n");
			Enumeration e = vLabelVector.elements();
			while (e.hasMoreElements())
			{
				LabelEntry labelEntry = (LabelEntry)e.nextElement();
				sb.append(labelEntry + "\n");
			}
		}

		if (vTextSection != null)
		{
			sb.append("\nTEXT:\n");
			Enumeration e = vTextSection.elements();
			while (e.hasMoreElements())
			{
				TextSection textSection = (TextSection)e.nextElement();
				sb.append(textSection + "\n");
			}
		}

		if (vDataSection != null)
		{
			sb.append("\nDATA:\n");
			Enumeration e = vDataSection.elements();
			while (e.hasMoreElements())
			{
				DataSection dataSection = (DataSection)e.nextElement();
				sb.append(dataSection + "\n");
			}
		}

		return sb.toString();

	}
	// end method toString()

// ******************************************************
//   UTILITY METHODS
// ******************************************************

	public boolean bLabelDefined(String s)
	{
		boolean result = false;
		Enumeration e = vLabelVector.elements();
		while (e.hasMoreElements())
		{
			LabelEntry labelEntry = (LabelEntry)e.nextElement();
			if (labelEntry.sName.equals(s))
			{
				result = true;
				break;
			}
		}
		return result;
	}
	// end method bLabelDefined(String)

	public int iGetLabelAddress(String s)
	{
		// Assume that the label does exists
		int result = 0;
		Enumeration e = vLabelVector.elements();
		while (e.hasMoreElements())
		{
			LabelEntry labelEntry = (LabelEntry)e.nextElement();
			if (labelEntry.sName.equals(s))
			{
				if (labelEntry.iSection == TEXT_SECTION)
				{

					TextSection textSection = (TextSection)vTextSection.elementAt(labelEntry.iSectionIndex);
					result = textSection.iAddress + (labelEntry.iIndex * DirectiveProcessor.SIZE_WORD);

					} else {
					// Assume labelEntry.iSection == DATA_SECTION

					DataSection dataSection = (DataSection)vDataSection.elementAt(labelEntry.iSectionIndex);
					DataEntry dataEntry = (DataEntry)dataSection.vDataVector.elementAt(labelEntry.iIndex);
					result = dataEntry.iAddress;
				}
				break;
			}
		}
		return result;
	}
	// end method iGetLabelAddress(String)

	private int iGetLabelIndex(String sLabel)
	{
		/*
		RETURNS
		-1 = The label sLabel has not been defined.
		>0 = The index into vLabelVector of the label sLabel

		*/
		Enumeration e = vLabelVector.elements();
		int i = 0;
		while (e.hasMoreElements())
		{
			LabelEntry labelEntry = (LabelEntry)e.nextElement();
			if (labelEntry.sName.equals(sLabel))
			return i;
		}
		return -1;
	}

	public void addToDataVector(DataEntry dataEntry)
	{
		currentDataSection.vDataVector.add(dataEntry);
		iCurrentDataIndex++;
	}
	// end method addToDataVector(DataEntry)

	public void addToTextVector(TextEntry textEntry)
	{
		if (DirectiveProcessor.bWarnAboutAT)
		{
			if (isAT(textEntry.rs) ||
			isAT(textEntry.rt) ||
			isAT(textEntry.rd)
			)
			{
				warnAboutUseOfAT(textEntry.iSourceLine);
			}
		}
		currentTextSection.vTextVector.add(textEntry);
		iCurrentTextIndex++;
	}
	// end method addToTextVector(TextEntry)

	private String sProcessedDataString(String sInput)
	{
		// String sInput should have form "AABBCC...DD"
		// Want to expand to "AA BB CC ... DD"

		String s = sInput.trim();
		s = Utility.sRemoveChar(s, ' ');
/*
		int i = 0;
		do
		{
			i = s.indexOf(' ');
			if (i >= 0)
			s = s.substring(0,i) + s.substring(i+1);
			else
			break;
		}
		while (true);
		*/

		String sResult = "";
		do
		{
			String z = s.substring(0,2);
			sResult += z + " ";
			s = s.substring(2);
		}
		while (!s.equals(""));

		return sResult.trim();

	}
	// end method sProcessedDataString(String)

	private String sCompressBitSequence(String sInput)
	{
		/*
		sInput should be a string that contains a binary sequence
		(0 and 1) but may contain spaces.  Remove all spaces,
		and return the hex value of the binary sequence (length
		of the binary sequence should be a multiple of 8).

		Example:
		sInput        == "01101011 01010010 11010101 11011010 10111011"
		sBinaryString == "0110101101010010110101011101101010111011"
		sbResult      == "6B 52 D5 DA BB"
		*/

		String sBinaryString = Utility.sRemoveChar(sInput, ' ');

		StringBuffer sbResult = new StringBuffer();
		int x = 0;
		int iLen = sBinaryString.length();
		while (x < iLen)
		{
			int iValue = Integer.parseInt(sBinaryString.substring(x,x+8), 2);
			sbResult.append(Utility.sAsHexPadded(iValue,2) + " ");
			x += 8;
		}

		return sbResult.toString();
	}

// *** STATIC METHODS **********************************************

	public static void warnAboutUseOfAT(int iSourceLine)
	{
		Assembler.iInvalidUseOfAT = iSourceLine;
	}

	public static boolean isAT(String s)
	{
		if (s == null)
		return false;

		s = s.toUpperCase();
		if (s.equals("$AT") ||
		s.equals("AT") ||
		s.equals("$1") ||
		s.equals("1")
		)
		{
			return true;
		}
		return false;
	}

	public static String[] saGetParams(String s)
	{

		Vector vParam = vGetParams(s);
		if (vParam == null)
		return null;

		String[] saResult = new String[vParam.size()];
		Enumeration e = vParam.elements();
		int i = 0;
		while (e.hasMoreElements())
		{
			saResult[i] = (String)e.nextElement();
			i++;
		}

/*
		System.out.print(saResult.length +": " );
		for (i = 0; i < saResult.length; i++)
		{
			System.out.print("["+saResult[i]+"]\t");
		}
		System.out.println();
		*/

		return saResult;

	}
	// end method saGetParams(String)

	private static Vector vGetParams(String sInput)
	{

		// Examples of sInput format:
		//   .data x1, x2,x3,x4
		//   add $t1 $t2 $t3
		//   .asciiz "text, ok!"

		Vector vResult = new Vector();  // Vector of String objects

		char[] ca = sInput.toCharArray();

		StringBuffer sbBuffer = new StringBuffer();

		boolean bTokenExpected = true;

		for (int i = 0; i < ca.length; i++)
		{

			char c = ca[i];

			if ( (c == ',') || (c == ' ') || (c == '\t') )
			{
				if (bTokenExpected)
				{
					// Do nothing, we are waiting for a token.
					if (c == ',')
					{
						System.err.println("ERROR: Invalid use of comma (,).");
						return null;
					}
					} else {
					// Reached end of this parameter
					if (sbBuffer.length() > 0)
					{
						vResult.add(sbBuffer.toString());
						sbBuffer = new StringBuffer();
					}
					bTokenExpected = true;
				}

				} else if (c == '"') {
				// Start of a string literal

				if (sbBuffer.length() > 0)
				{
					System.err.println("ERROR: Invalid content between string declarations.");
					return null;
//          sbBuffer = new StringBuffer();
				}

				sbBuffer.append(c);
				do
				{
					i++;
					if (i >= ca.length)
					{
						System.err.println("ERROR: String literal parameter not properly terminated.");
						return null;
					}
					c = ca[i];
					sbBuffer.append(c);
				}
				while (c != '"');
				vResult.add(sbBuffer.toString());
				sbBuffer = new StringBuffer();

				} else {
				bTokenExpected = false;
				sbBuffer.append(c);
			}
		}
		if (sbBuffer.length() > 0)
		{
			vResult.add(sbBuffer.toString());
		}

		return vResult;

	}
	// end method vGetParams(String)


}
