function [ lambda_new, a_new, more_new ] = ytb_next ( n, lambda, a, more )

%% YTB_NEXT computes the next Young tableau for a given shape.
%
%  Discussion:
%
%    When the routine is called with MORE = FALSE (the first time), and
%    if LAMBDA on this call has M parts, with M < N, then the user
%    must also make sure that LAMBDA(M+1) = 0.
%
%  Modified:
%
%    22 June 2004
%
%  Reference:
%
%    A Nijenhuis and H Wilf,
%    Combinatorial Algorithms,
%    Academic Press, 1978, second edition,
%    ISBN 0-12-519260-6.
%
%  Parameters:
%
%    Input, integer N, the integer which is to be partitioned.
%
%    Input, integer LAMBDA(N), contains a partition of N. 
%    The elements of LAMBDA are nonnegative integers that sum to N. 
%    On the first call, with MORE = FALSE, the user sets LAMBDA.
%    After the first call, the input value of LAMBDA is not important.
%
%    Input, integer A(N).  On the first call, with MORE = FALSE,
%    no value of A needs to be set.  After the first call, the input
%    value of A should be the output value of A_NEW from the previous call.
%
%    Input, logical MORE.  Set MORE to FALSE before the first call.
%    Thereafter, set it to the output value of MORE_NEW on the previous call.
%
%    Output, integer LAMBDA_NEW(N), contains the partition of N,
%    corresponding to the Young tableau.
%
%    Output, integer A_NEW(N), the next Young tableau.  A_NEW(I) is the 
%    row containing I in the output tableau.
%
%    Output, logical MORE_NEW, is TRUE until the last tableau is returned,
%    when the value of MORE_NEW is FALSE.
%
  lambda_new(1:n) = lambda(1:n);
  a_new(1:n) = a(1:n);
  more_new = more;

  it = n;

  if ( more_new )

    lambda_new(1) = 1;
    lambda_new(2:n) = 0;

    isave = 0;

    for i = 2 : n

      lambda_new(a_new(i)) = lambda_new(a_new(i)) + 1;

      if ( a_new(i) < a_new(i-1) )
        isave = i;
        break;
      end

    end

    if ( isave == 0 )
      more_new = 0;
      return
    end

    it = lambda_new(1+a_new(isave));

    for i = n : -1 : 1

      if ( lambda_new(i) == it )
        a_new(isave) = i;
        lambda_new(i) = lambda_new(i) - 1;
        it = isave - 1;
        break;
      end

    end

  end

  k = 1;
  ir = 1;

  while ( 1 )

    if ( n < ir )
      break;
    end

    if ( lambda_new(ir) ~= 0 )
      a_new(k) = ir;
      lambda_new(ir) = lambda_new(ir) - 1;
      k = k + 1;
      ir = ir + 1;
      continue;
    end

    if ( it < k )
      break;
    end

    ir = 1;

  end

  if ( n == 1 )
    more_new = 0;
    return
  end

  for j = 2 : n
    if ( a_new(j) < a_new(j-1) )
      more_new = 1;
      return
    end
  end

  more_new = 0;
