function [ r, s ] = rfrac_to_jfrac ( m, p, q )

%% RFRAC_TO_JFRAC converts a rational polynomial fraction to a J fraction.
%
%  Discussion:
%
%    The routine accepts
%
%    P(1) + P(2) * X + ... + P(M) * X**(M-1)
%    -------------------------------------------------------
%    Q(1) + Q(2) * X + ... + Q(M) * X**(M-1) + Q(M+1) * X**M
%
%    and returns the equivalent J-fraction:
%
%    R(1)/ ( X + S(1) + R(2) / ( X + S(2) + R(3) / ... + R(M) / ( X + S(M) )... ))
%
%  Modified:
%
%    31 July 2004
%
%  Reference:
%
%    Hart, Cheney, Lawson, Maehly, Mesztenyi, Rice, Thacher, Witzgall,
%    Computer Approximations,
%    Wiley, 1968.
%
%  Parameters:
%
%    Input, integer M, defines the number of P, R, and S coefficients,
%    and is one less than the number of Q coefficients.
%
%    Input, real P(M), Q(M+1), the coefficients defining the rational
%    polynomial fraction.
%
%    Output, real R(M), S(M), the coefficients defining the
%    J-fraction.
%
  a(1:m+1,1) = q(1:m+1)';
  a(1:m,  2) = p(1:m)';

  if ( 1 < m )

    r(1) = a(m,2) / a(m+1,1);
    s(1) = ( r(1) * a(m,1) - a(m-1,2) ) / a(m,2);

    for k = 1 : m-2

      a(1,k+2) = r(k) * a(1,k) - s(k) * a(1,k+1);

      for i = 2 : m-k

        a(i,k+2) = r(k) * a(i,k) - a(i-1,k+1) - s(k) * a(i,k+1);

        if ( a(m-k,k+2) == 0.0E+00 )
          fprintf ( 1, '\n' );
          fprintf ( 1, 'RFRAC_TO_JFRAC - Fatal error!\n' );
          fprintf ( 1, '  A(M-K,K+2) = 0 for K = %d\n', k );
          error ( 'RFRAC_TO_JFRAC - Fatal error!' );
        end

        r(k+1) = a(m-k,k+2) / a(m-k+1,k+1);
        s(k+1) = ( r(k+1) * a(m-k,k+1) - a(m-k-1,k+2) ) / a(m-k,k+2);

      end

    end

    a(1,m+1) = r(m-1) * a(1,m-1) - s(m-1) * a(1,m);

  end

  r(m) = a(1,m+1) / a(2,m);
  s(m) = a(1,m) / a(2,m);

