function [ p_new, more_new ] = perm_next3 ( n, p, more )

%% PERM_NEXT3 computes all of the permutations of N objects, one at a time.
%
%  Discussion:
%
%    The routine is initialized by calling with MORE = TRUE, in which case
%    it returns the identity permutation.
%
%    If the routine is called with MORE = FALSE, then the successor of the
%    input permutation is computed.
%
%    Trotter's algorithm is used.
%
%  Modified:
%
%    01 August 2004
%
%  Reference:
%
%    H Trotter,
%    PERM, Algorithm 115,
%    Communications of the Association for Computing Machinery,
%    Volume 5, 1962, pages 434-435.
%
%  Parameters:
%
%    Input, integer N, the number of objects being permuted.
%
%    Input, integer P(N), is the previous permutation, if MORE is TRUE.
%    But on a startup call, with MORE FALSE, the input value of P is not needed.
%
%    Input, logical MORE, should be set to FALSE on an initialization call.
%    Thereafter, set MORE to the output value of MORE_NEW.
%
%    Output, integer P_NEW(N), the next permutation; if MORE is FALSE, then 
%    P_NEW is the first permutation in the sequence.
%
%    Output, logical MORE_NEW, is TRUE if there was a next permutation to produce,
%    or FALSE if there are no more permutations to produce.
%
  persistent nfact;
  persistent rank;

  more_new = more;

  if ( ~more_new )

    p_new = i4vec_indicator ( n );
    more_new = 1;
    rank = 1;

    nfact = i4_factorial ( n );

  else

    p_new(1:n) = p(1:n);

    n2 = n;
    m2 = rank;
    s = n;

    while ( 1 )

      q = mod ( m2, n2 );
      t = mod ( m2, 2 * n2 );

      if ( q ~= 0 )
        break
      end

      if ( t == 0 )
        s = s - 1;
      end

      m2 = floor ( m2 / n2 );
      n2 = n2 - 1;

    end

    if ( q == t )
      s = s - q;
    else
      s = s + q - n2;
    end

    [ p_new(s), p_new(s+1) ] = i4_swap ( p_new(s), p_new(s+1) );

    rank = rank + 1;

    if ( rank == nfact );
      more_new = 0;
    end

  end
