function [ change_num_new, change_new, done_new] = change_next ( total, ...
  coin_num, coin_value, change_num, change, done  )

%% CHANGE_NEXT computes the next set of change for a given sum.
%
%  Examples:
%
%    Total = 17
%    COIN_NUM = 3
%    COIN_VALUE = (/ 1, 5, 10 /)
%
%
%        #  CHANGE              COIN_VALUE(CHANGE)
%
%    1   4  3 2 1 1             10 5 1 1
%    2   8  3 1 1 1 1 1 1 1     10 1 1 1 1 1 1 1
%    3   5  2 2 2 1 1            5 5 5 1 1
%    4   9  2 2 1 1 1 1 1 1 1    5 5 1 1 1 1 1 1 1
%    5  13  2 1 1 1 1 1 1 1 1 1  5 1 1 1 1 1 1 1 1 1
%           1 1 1                1 1 1
%    6  17  1 1 1 1 1 1 1 1 1 1  1 1 1 1 1 1 1 1 1 1 1
%           1 1 1 1 1 1 1        1 1 1 1 1 1
%
%  Modified:
%
%    17 July 2004
%
%  Author:
%
%    John Burkardt
%
%  Parameters:
%
%    Input, integer TOTAL, the total for which change is to be made.
%
%    Input, integer COIN_NUM, the number of types of coins.
%
%    Input, integer COIN_VALUE(COIN_NUM), the value of each coin.
%    The values should be in ascending order.
%
%    Input, integer CHANGE_NUM, the output value of CHANGE_NUM_NEW
%    from the previous call.  This value is not needed on a startup call.
%
%    Input, integer CHANGE(CHANGE_NUM), the output value of CHANGE_NEW
%    from the previous call.  This value is not needed on a startup call.
%
%    Input, logical DONE.  The user sets DONE = TRUE on the
%    first call to tell the routine this is the beginning of a computation.
%    Thereafter, DONE should be set to the output value of DONE_NEW from]
%    the previous call.
%
%    Output, integer CHANGE_NUM_NEW, the number of coins given in change
%    for the next set of change.
%
%    Output, integer CHANGE_NEW(CHANGE_NUM_NEW), the indices of the coins
%    used in this set of change.
%
%    Output, logical DONE_NEW, is FALSE until the last possible set of change
%    has been made.
%
  done_new = done;

  if ( done_new )
%
%  Make sure the coin values are sorted.
%
    if ( ~i4vec_ascends ( coin_num, coin_value ) )
      fprintf ( 1, '\n' );
      fprintf ( 1, 'CHANGE_NEXT - Fatal error!\n' );
      fprintf ( 1, '  The array COIN_VALUE is not in ascending order.\n' );
      error ( 'CHANGE_NEXT - Fatal error!' );
    end
%
%  Start with the greedy change.
%
    [ change_num_new, change_new ] = change_greedy ( total, coin_num, coin_value );
%
%  In a few cases, like change for 4 cents, we're done after the first call.
%
    if ( change_num_new == total )
      done_new = 1;
    else
      done_new = 0;
    end

    return

  else

    change_num_new = change_num;
    change_new(1:change_num_new) = change(1:change_num_new);
%
%  Find the last location in the input change which is NOT a penny.
%
    last = 0;

    for i = change_num_new : -1 : 1
      if ( change_new(i) ~= 1 )
        last = i;
        break
      end
    end
%
%  If that location is still 0, an error was made.
%
    if ( last == 0 )
      done_new = 1;
      return
    end
%
%  Sum the entries from that point to the end.
%
    total2 = sum ( coin_value ( change_new(last:change_num_new) ) );
%
%  Make greedy change for the partial sum using coins smaller than that one.
%
    coin_num2 = change_new(last) - 1;

    [ change_num2, change2 ] = change_greedy ( total2, coin_num2, coin_value );
    change_new(last:last+change_num2-1) = change2(1:change_num2);
    
    change_num_new = ( last - 1 ) + change_num2;

  end
