function avw_view(avw),

% AVW_VIEW: Create and navigate ortho views of Analyze file
%
% avw_view(avw)
%
% avw    -  a struct, created by avw_img_read
%
% The navigation is by sliders and mouse clicks on the
% images in any of the ortho views.
% 
% Fiducial points can be selected, which are returned
% into mriFID or p.mriFID in the base workspace.  These
% points are given in meters, with an origin translated 
% from the center of the MRI volume to (0,0,0).
% 
% +X is left (L), +Y is anterior (A), +Z is superior (S),
% the default LAS orientation of Analyze MRI files.  This
% is the radiological convention, as opposed to the
% neurological convention (RAS).
% 
% See also, AVW_IMG_READ
% 

% Licence:  GNU GPL, no express or implied warranties
% History:  06/2002, Darren.Weber@flinders.edu.au
%           10/2002, Darren.Weber@flinders.edu.au
%                    added fiducial point determination
%                    changed plots from surf to imagesc commands
%                    added handling of datatype for avw.img
%                    altered daspect to use avw.hdr.dime.pixdim
%                    altered color scheme
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


if ~exist('avw','var'),
    msg = sprintf('...no input avw - see help avw_view\n');
    error(msg);
end

% determine the datatype of avw.img
switch double(avw.hdr.dime.bitpix),
case 1,
    fprintf('...converting avw.img to uint8 for viewing only.\n');
    avw.img = uint8(avw.img);
case 8,
    fprintf('...converting avw.img to uint8 for viewing only.\n');
    avw.img = uint8(avw.img);
case 16,
    fprintf('...converting avw.img to uint16 for viewing only.\n');
    avw.img = uint16(avw.img);
case {32,64},
    % make sure it is double, not single
    avw.img = double(avw.img);
otherwise,
    % do nothing, leave it as is
end

clim = [0 max(max(max(avw.img))) ];

%clim = double(clim) .* 0.5;


% GUI General Parameters
GUIwidth  = 150;
GUIheight = 50;
if isfield(avw,'fileprefix'),
    if isempty(avw.fileprefix),
        name = 'AVW View';
    else
        format = strcat('%+',sprintf('%d',length(avw.fileprefix)+1),'s');
        name = strcat('AVW View - ',sprintf(format,avw.fileprefix));
    end
else
    name = 'AVW View';
end

GUI = figure('Name',name,'Tag','AVWVIEW','units','characters',...
             'NumberTitle','off','color',[0 0 0],...
             'MenuBar','figure','Position',[1 1 GUIwidth GUIheight]);
movegui(GUI,'center');

AVWVIEW.gui = GUI;


Font.FontName   = 'Helvetica';
Font.FontUnits  = 'Pixels';
Font.FontSize   = 12;
Font.FontWeight = 'normal';
Font.FontAngle  = 'normal';


shading flat


xdim = size(avw.img,1);
ydim = size(avw.img,2);
zdim = size(avw.img,3);

SagSlice = 1;
CorSlice = 1;
AxiSlice = 1;
if xdim > 1, SagSlice = floor(xdim/2); end
if ydim > 1, CorSlice = floor(ydim/2); end
if zdim > 1, AxiSlice = floor(zdim/2); end

AVWVIEW.origin  = [SagSlice,CorSlice,AxiSlice];             % vol origin

pixdim = double(avw.hdr.dime.pixdim(2:4));

AVWVIEW.scale   = pixdim ./ 1000; % vol scale in meters
AVWVIEW.daspect = pixdim ./ max(pixdim);

% Initialise some window handles
G.Ha = 0;
G.Hc = 0;
G.Hs = 0;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Axial Slice
if xdim > 1 & ydim > 1,
	
	G.Aa = subplot('position',[0.05 0.56 0.4 0.4]);
	colormap('gray');
    
	Saxial = squeeze(avw.img(:,:,AxiSlice));
    G.Ha = imagesc([0,xdim],[0,ydim],Saxial',clim);
    G.Haax = gca;
    set(gca,'Color','none','YDir','normal','ClimMode','manual',...
        'YColor',[1 1 1],'XColor',[1 1 1])
    
    daspect(AVWVIEW.daspect([2 1 3]));
    
    %xlabel('(Left <<) X (>> Right)')
    %xlabel('(Right <<) X (>> Left)')
    xlabel('X')
	ylabel('Y')
	title('Axial')
	
	% This callback navigates with left click
	set(G.Ha,'ButtonDownFcn',...
        strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
        'currentpoint = get(get(AVWVIEW.handles.Ha,''Parent''),''CurrentPoint''); ',...
        'SagSlice = round(currentpoint(2,1)); ',...
        'CorSlice = round(currentpoint(2,2)); ',...
        'AxiSlice = round(str2num(get(AVWVIEW.handles.Taxi,''String''))); ',...
        'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
        'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
        'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'if ishandle(AVWVIEW.handles.Hs) & AVWVIEW.handles.Hs, ',...
        '   Ssag = squeeze(AVWVIEW.avw.img(SagSlice,:,:));',...
        '   set(AVWVIEW.handles.Hs,''CData'',Ssag''); ',...
        '   set(AVWVIEW.handles.Tsag,''String'',num2str(SagSlice));',...
        '   set(AVWVIEW.handles.Ssag,''Value'',SagSlice);',...
        '   clear Ssag; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'if ishandle(AVWVIEW.handles.Hc) & AVWVIEW.handles.Hc, ',...
        '   Scor = squeeze(AVWVIEW.avw.img(:,CorSlice,:));',...
        '   set(AVWVIEW.handles.Hc,''CData'',Scor''); ',...
        '   set(AVWVIEW.handles.Tcor,''String'',num2str(CorSlice));',...
        '   set(AVWVIEW.handles.Scor,''Value'',CorSlice);',...
        '   clear Scor; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'clear currentpoint imgvalue AxiSlice CorSlice SagSlice AVWVIEW;'));
    
    if zdim > 1,
		slider_step(1) = 1/(zdim);
		slider_step(2) = 1/(zdim);
		G.Saxi = uicontrol('Parent',GUI,'Style','slider','Units','Normalized', Font, ...
            'Position',[.46 .56 .03 .40], 'HorizontalAlignment', 'center',...
            'BusyAction','queue',...
            'Min',1,'Max',zdim,'SliderStep',slider_step,'Value',AxiSlice,...
            'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'AxiSlice = round(get(AVWVIEW.handles.Saxi,''Value''));',...
            'set(AVWVIEW.handles.Saxi,''Value'',AxiSlice);',...
            'Saxi = squeeze(AVWVIEW.avw.img(:,:,AxiSlice));',...
            'set(AVWVIEW.handles.Ha,''CData'',Saxi''); ',...
            'set(AVWVIEW.handles.Taxi,''String'',num2str(AxiSlice));',...
            'CorSlice = round(get(AVWVIEW.handles.Scor,''Value''));',...
            'SagSlice = round(get(AVWVIEW.handles.Ssag,''Value''));',...
            'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
            'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgvalue Saxi AxiSlice CorSlice SagSlice AVWVIEW;'));
    end
	G.Taxi = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
        'Position',[.46 .51 .03 .05], 'HorizontalAlignment', 'center',...
        'BusyAction','queue',...
        'String',num2str(AxiSlice));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Coronal Slice
if xdim > 1 & zdim > 1,
	
	subplot('position',[0.55 0.56 0.4 0.4])
	colormap('gray');
    
	Scor = squeeze(avw.img(:,CorSlice,:));
    G.Hc = imagesc([0,xdim],[0,zdim],Scor',clim);
    G.Hcax = gca;
    set(gca,'YDir','normal','ClimMode','manual',...
        'YColor',[1 1 1],'XColor',[1 1 1])
    
    daspect(AVWVIEW.daspect([3 1 2]));
	
    %xlabel('(Left <<) X (>> Right)')
    %xlabel('(Right <<) X (>> Left)')
    xlabel('X')
	ylabel('Z')
	title('Coronal')
	
	% This callback navigates with left click
	set(G.Hc,'ButtonDownFcn',...
        strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
        'currentpoint = get(get(AVWVIEW.handles.Hc,''Parent''),''CurrentPoint''); ',...
        'SagSlice = round(currentpoint(2,1)); ',...
        'AxiSlice = round(currentpoint(2,2)); ',...
        'CorSlice = round(str2num(get(AVWVIEW.handles.Tcor,''String''))); ',...
        'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
        'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
        'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'if ishandle(AVWVIEW.handles.Hs) & AVWVIEW.handles.Hs, ',...
        '   Ssag = squeeze(AVWVIEW.avw.img(SagSlice,:,:));',...
        '   set(AVWVIEW.handles.Hs,''CData'',Ssag''); ',...
        '   set(AVWVIEW.handles.Tsag,''String'',num2str(SagSlice));',...
        '   set(AVWVIEW.handles.Ssag,''Value'',SagSlice);',...
        '   clear Ssag; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'if ishandle(AVWVIEW.handles.Ha) & AVWVIEW.handles.Ha, ',...
        '   Saxi = squeeze(AVWVIEW.avw.img(:,:,AxiSlice));',...
        '   set(AVWVIEW.handles.Ha,''CData'',Saxi''); ',...
        '   set(AVWVIEW.handles.Taxi,''String'',num2str(AxiSlice));',...
        '   set(AVWVIEW.handles.Saxi,''Value'',AxiSlice);',...
        '   clear Saxi; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'clear currentpoint imgvalue AxiSlice CorSlice SagSlice AVWVIEW;'));
    
    if ydim > 1,
        slider_step(1) = 1/(ydim);
		slider_step(2) = 1/(ydim);
		G.Scor = uicontrol('Parent',GUI,'Style','slider','Units','Normalized', Font, ...
            'Position',[.96 .56 .03 .40], 'HorizontalAlignment', 'center',...
            'BusyAction','queue',...
            'Min',1,'Max',ydim,'SliderStep',slider_step,'Value',CorSlice,...
            'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'CorSlice = round(get(AVWVIEW.handles.Scor,''Value''));',...
            'set(AVWVIEW.handles.Scor,''Value'',CorSlice);',...
            'Scor = squeeze(AVWVIEW.avw.img(:,CorSlice,:));',...
            'set(AVWVIEW.handles.Hc,''CData'',Scor''); drawnow;',...
            'set(AVWVIEW.handles.Tcor,''String'',num2str(CorSlice));',...
            'AxiSlice = round(get(AVWVIEW.handles.Saxi,''Value''));',...
            'SagSlice = round(get(AVWVIEW.handles.Ssag,''Value''));',...
            'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
            'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgvalue Scor AxiSlice CorSlice SagSlice AVWVIEW;'));
    end
	G.Tcor = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
        'Position',[.96 .51 .03 .05], 'HorizontalAlignment', 'center',...
        'BusyAction','queue',...
        'String',num2str(CorSlice));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Sagittal Slice
if ydim > 1 & ydim > 1,
		
% 	[y,z] = meshgrid(1:ydim,1:zdim);
% 	Ysag = y'; clear y;
% 	Zsag = z'; clear z;
% 	Xsag = zeros(ydim,zdim);
	
	subplot('position',[0.05 0.06 0.4 0.4])
	colormap('gray');
    
	Ssag = squeeze(avw.img(SagSlice,:,:));
    G.Hs = imagesc([0,ydim],[0,zdim],Ssag',clim);
    G.Hsax = gca;
    set(gca,'YDir','normal','ClimMode','manual',...
        'YColor',[1 1 1],'XColor',[1 1 1])
    
    daspect(AVWVIEW.daspect([3 2 1]));
    
    xlabel('Y')
	ylabel('Z')
	title('Sagittal')
	
	% This callback navigates with mouse click
	set(G.Hs,'ButtonDownFcn',...
        strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
        'currentpoint = get(get(AVWVIEW.handles.Hs,''Parent''),''CurrentPoint''); ',...
        'CorSlice = round(currentpoint(1,1)); ',...
        'AxiSlice = round(currentpoint(1,2)); ',...
        'SagSlice = round(str2num(get(AVWVIEW.handles.Tsag,''String'')));',...
        'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
        'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
        'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'if ishandle(AVWVIEW.handles.Hc) & AVWVIEW.handles.Hc, ',...
        '   Scor = squeeze(AVWVIEW.avw.img(:,CorSlice,:));',...
        '   set(AVWVIEW.handles.Hc,''CData'',Scor''); ',...
        '   set(AVWVIEW.handles.Tcor,''String'',num2str(CorSlice));',...
        '   set(AVWVIEW.handles.Scor,''Value'',CorSlice);',...
        '   clear Scor; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'if ishandle(AVWVIEW.handles.Ha) & AVWVIEW.handles.Ha, ',...
        '   Saxi = squeeze(AVWVIEW.avw.img(:,:,AxiSlice));',...
        '   set(AVWVIEW.handles.Ha,''CData'',Saxi''); ',...
        '   set(AVWVIEW.handles.Taxi,''String'',num2str(AxiSlice));',...
        '   set(AVWVIEW.handles.Saxi,''Value'',AxiSlice);',...
        '   clear Saxi; ',...
        '   set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
        'end; ',...
        'clear currentpoint imgvalue AxiSlice CorSlice SagSlice AVWVIEW;'));

    
    if xdim > 1,
		slider_step(1) = 1/(xdim);
		slider_step(2) = 1/(xdim);
		G.Ssag = uicontrol('Parent',GUI,'Style','slider','Units','Normalized', Font, ...
            'Position',[.46 .06 .03 .4], 'HorizontalAlignment', 'center',...
            'BusyAction','queue',...
            'Min',1,'Max',xdim,'SliderStep',slider_step,'Value',SagSlice,...
            'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'SagSlice = round(get(AVWVIEW.handles.Ssag,''Value''));',...
            'set(AVWVIEW.handles.Ssag,''Value'',SagSlice);',...
            'Ssag = squeeze(AVWVIEW.avw.img(SagSlice,:,:));',...
            'set(AVWVIEW.handles.Hs,''CData'',Ssag''); drawnow;',...
            'set(AVWVIEW.handles.Tsag,''String'',num2str(SagSlice));',...
            'AxiSlice = round(get(AVWVIEW.handles.Saxi,''Value''));',...
            'CorSlice = round(get(AVWVIEW.handles.Scor,''Value''));',...
            'imgvalue = double(AVWVIEW.avw.img(SagSlice,CorSlice,AxiSlice)); ',...
            'set(AVWVIEW.handles.imval,''String'',sprintf(''%8.2f'',imgvalue));',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgvalue Ssag AxiSlice CorSlice SagSlice AVWVIEW;'));
    end
	G.Tsag = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
        'Position',[.46 .01 .03 .05], 'HorizontalAlignment', 'center',...
        'BusyAction','queue',...
        'String',num2str(SagSlice));
end



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Intensity Value at Mouse Click

G.Timval = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
    'Position',[.575 .40 .20 .05], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [0 0 0],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'String','Image Intensity');
G.imval = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
    'Position',[.775 .40 .20 .05], 'HorizontalAlignment', 'right',...
    'BackgroundColor', [0 0 0],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'String','x');

% Nasion Location
G.Tnasion = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.575 .35 .20 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Update Nasion - should be toward +Y',...
    'String','Fiducial: Nasion',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'SagSlice = get(AVWVIEW.handles.Ssag,''Value'');',...
            'CorSlice = get(AVWVIEW.handles.Scor,''Value'');',...
            'AxiSlice = get(AVWVIEW.handles.Saxi,''Value'');',...
            'imgXYZ   = [SagSlice,CorSlice,AxiSlice]; ',...
            'imgXYZ = (imgXYZ - AVWVIEW.origin) .* AVWVIEW.scale; ',...
            'set(AVWVIEW.handles.nasion,''String'',sprintf(''%6.3f %6.3f %6.3f'',imgXYZ));',...
            'AVWVIEW.p.mriFID(1,:) = imgXYZ; ',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgXYZ AxiSlice CorSlice SagSlice AVWVIEW;'));
G.nasion = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
    'Position',[.775 .35 .20 .04], 'HorizontalAlignment', 'right',...
    'BackgroundColor', [0 0 0],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','In meters, origin at (0,0,0), should be toward +Y',...
    'String','x,y,z');

% Right Preauricular Location
G.Trpa = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.575 .30 .20 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Update Right Preauricular - should be toward +X',...
    'String','Fiducial: RPA',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'SagSlice = get(AVWVIEW.handles.Ssag,''Value'');',...
            'CorSlice = get(AVWVIEW.handles.Scor,''Value'');',...
            'AxiSlice = get(AVWVIEW.handles.Saxi,''Value'');',...
            'imgXYZ   = [SagSlice,CorSlice,AxiSlice]; ',...
            'imgXYZ = (imgXYZ - AVWVIEW.origin) .* AVWVIEW.scale; ',...
            'set(AVWVIEW.handles.rpa,''String'',sprintf(''%6.3f %6.3f %6.3f'',imgXYZ));',...
            'AVWVIEW.p.mriFID(2,:) = imgXYZ; ',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgXYZ AxiSlice CorSlice SagSlice AVWVIEW;'));
G.rpa = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
    'Position',[.775 .30 .20 .04], 'HorizontalAlignment', 'right',...
    'BackgroundColor', [0 0 0],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','In meters, origin at (0,0,0), should be toward +X',...
    'String','x,y,z');

% Left Preauricular Location
G.Tlpa = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.575 .25 .20 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Update Left Preauricular - should be toward -X',...
    'String','Fiducial: LPA',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
            'SagSlice = get(AVWVIEW.handles.Ssag,''Value'');',...
            'CorSlice = get(AVWVIEW.handles.Scor,''Value'');',...
            'AxiSlice = get(AVWVIEW.handles.Saxi,''Value'');',...
            'imgXYZ   = [SagSlice,CorSlice,AxiSlice]; ',...
            'imgXYZ = (imgXYZ - AVWVIEW.origin) .* AVWVIEW.scale; ',...
            'set(AVWVIEW.handles.lpa,''String'',sprintf(''%6.3f %6.3f %6.3f'',imgXYZ));',...
            'AVWVIEW.p.mriFID(3,:) = imgXYZ; ',...
            'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
            'clear imgXYZ AxiSlice CorSlice SagSlice AVWVIEW;'));
G.lpa = uicontrol('Parent',GUI,'Style','text','Units','Normalized', Font, ...
    'Position',[.775 .25 .20 .04], 'HorizontalAlignment', 'right',...
    'BackgroundColor', [0 0 0],...
    'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','In meters, origin at (0,0,0), should be toward -X',...
    'String','x,y,z');


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

G.dimmer = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.575 .20 .10 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Dim by 1%',...
    'String','Dimmer',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
    'if ishandle(AVWVIEW.handles.Ha) & AVWVIEW.handles.Ha, ',...
    '   clim = get(AVWVIEW.handles.Haax,''clim'');',...
    '   set(AVWVIEW.handles.Haax,''Clim'',double(clim) .* 1.1); ',...
    '   clear clim; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hc) & AVWVIEW.handles.Hc, ',...
    '   clim = get(AVWVIEW.handles.Hcax,''clim'');',...
    '   set(AVWVIEW.handles.Hcax,''Clim'',double(clim) .* 1.1); ',...
    '   clear clim; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hs) & AVWVIEW.handles.Hs, ',...
    '   clim = get(AVWVIEW.handles.Hsax,''clim'');',...
    '   set(AVWVIEW.handles.Hsax,''Clim'',double(clim) .* 1.1); ',...
    '   clear clim; ',...
    'end; ',...
    'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
    'clear xdim AVWVIEW;'));

G.bright = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.675 .20 .10 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Brighten by 1%',...
    'String','Brighter',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
    'if ishandle(AVWVIEW.handles.Ha) & AVWVIEW.handles.Ha, ',...
    '   clim = get(AVWVIEW.handles.Haax,''clim'');',...
    '   set(AVWVIEW.handles.Haax,''Clim'',double(clim) .* 0.9); ',...
    '   clear clim; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hc) & AVWVIEW.handles.Hc, ',...
    '   clim = get(AVWVIEW.handles.Hcax,''clim'');',...
    '   set(AVWVIEW.handles.Hcax,''Clim'',double(clim) .* 0.9); ',...
    '   clear clim; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hs) & AVWVIEW.handles.Hs, ',...
    '   clim = get(AVWVIEW.handles.Hsax,''clim'');',...
    '   set(AVWVIEW.handles.Hsax,''Clim'',double(clim) .* 0.9); ',...
    '   clear clim; ',...
    'end; ',...
    'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
    'clear xdim AVWVIEW;'));




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

G.flip = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.575 .15 .20 .04], 'HorizontalAlignment', 'left',...
    'BackgroundColor', [.3 .3 .3],'ForegroundColor', [1 1 1],...
    'BusyAction','queue',...
    'TooltipString','Flip Left and Right (viewer only).',...
    'String','Flip L/R',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
    'xdim = size(AVWVIEW.avw.img,1); ',...
    'AVWVIEW.avw.img = AVWVIEW.avw.img(xdim:-1:1,:,:);',...
    'if ishandle(AVWVIEW.handles.Ha) & AVWVIEW.handles.Ha, ',...
    '   AxiSlice = get(AVWVIEW.handles.Saxi,''Value'');',...
    '   Saxi = squeeze(AVWVIEW.avw.img(:,:,AxiSlice));',...
    '   set(AVWVIEW.handles.Ha,''CData'',Saxi''); ',...
    '   clear Saxi AxiSlice; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hc) & AVWVIEW.handles.Hc, ',...
    '   CorSlice = get(AVWVIEW.handles.Scor,''Value'');',...
    '   Scor = squeeze(AVWVIEW.avw.img(:,CorSlice,:));',...
    '   set(AVWVIEW.handles.Hc,''CData'',Scor''); ',...
    '   clear Scor CorSlice; ',...
    'end; ',...
    'if ishandle(AVWVIEW.handles.Hs) & AVWVIEW.handles.Hs, ',...
    '   SagSlice = get(AVWVIEW.handles.Ssag,''Value'');',...
    '   Ssag = squeeze(AVWVIEW.avw.img(SagSlice,:,:));',...
    '   set(AVWVIEW.handles.Hs,''CData'',Ssag''); ',...
    '   clear Ssag SagSlice; ',...
    'end; ',...
    'set(AVWVIEW.gui,''UserData'',AVWVIEW);',...
    'clear xdim AVWVIEW;'));


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Font.FontWeight = 'bold';

% OK: Return the avw!
G.Bhdr = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.8 .01 .08 .04],...
    'String','HDR','BusyAction','queue',...
    'TooltipString','Save the hdr parameters.',...
    'BackgroundColor',[0.0 0.0 0.5],...
    'ForegroundColor',[1 1 1], 'HorizontalAlignment', 'center',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
        'avw_view_hdr(AVWVIEW.avw);',...
        'clear AVWVIEW;'));

% Cancel
G.Bquit = uicontrol('Parent',GUI,'Style','pushbutton','Units','Normalized', Font, ...
    'Position',[.9 .01 .08 .04],...
    'String','RETURN','BusyAction','queue',...
    'BackgroundColor',[0.75 0.0 0.0],...
    'ForegroundColor', [1 1 1], 'HorizontalAlignment', 'center',...
    'Callback',strcat('AVWVIEW = get(gcbf,''Userdata''); ',...
        'if isfield(AVWVIEW,''p''), ',...
        '  if isfield(AVWVIEW.p,''mriFID''), ',...
        '    if exist(''p'',''var''), ',...
        '      p.mriFID = AVWVIEW.p.mriFID; ',...
        '    else, ',...
        '      mriFID = AVWVIEW.p.mriFID;',...
        '    end; ',...
        '  end; ',...
        'end; ',...
        'clear AVWVIEW; close gcbf;'));

% Update the gui_struct handles for this gui
AVWVIEW.avw = avw;
AVWVIEW.handles = G;
set(AVWVIEW.gui,'Userdata',AVWVIEW);
set(AVWVIEW.gui,'HandleVisibility','callback');

return


function slice_img(avw),

    figure
    xslice = 128;
    slice = squeeze( avw.img(xslice,:,:) );
    imagesc(slice); axis image; colormap('gray')
    figure
    yslice = 128;
    slice = squeeze( avw.img(:,yslice,:) );
    imagesc(slice); axis image; colormap('gray')
    figure
    zslice = 128;
    slice = squeeze( avw.img(:,:,zslice) );
    imagesc(slice); axis image; colormap('gray')

return
